"""
This program is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License,
or (at your option) any later version.


This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
See the GNU General Public License for more details.


You should have received a copy of the GNU General Public License
 along with this program.  If not, see <https://www.gnu.org/licenses/>.

Copyright © 2019 Cloud Linux Software Inc.

This software is also available under ImunifyAV commercial license,
see <https://www.imunify360.com/legal/eula>
"""
import hashlib
import logging
import os
from pathlib import Path
import tempfile
from contextlib import ExitStack

from defence360agent import utils
from defence360agent.contracts.config import Core
from defence360agent.contracts.messages import MessageType
from defence360agent.contracts.plugins import MessageSink, expect
from defence360agent.utils import Scope
from imav.malwarelib.utils import malware_response

logger = logging.getLogger(__name__)


class ServerPullFileSender(MessageSink):
    SCOPE = Scope.IM360
    loop = None

    async def create_sink(self, loop):
        self.loop = loop

    @expect(MessageType.MalwareSendFiles)
    async def upload_files(self, message):
        reason = message["reason"]
        for file in message["files"]:
            try:
                with ExitStack() as stack:
                    # Save directory contents into tempfile and upload it to MRS
                    if os.path.isdir(file):
                        tf = stack.enter_context(
                            tempfile.NamedTemporaryFile(dir=Core.TMPDIR)
                        )
                        tf.write(await utils.check_run(["ls", "-al", file]))
                        tf.seek(0)
                        file = malware_response.MalwareHitPath(tf.name, file)
                        hash_ = hashlib.sha256(
                            Path(tf.name).read_bytes()
                        ).hexdigest()
                        logger.info(
                            "Uploading temporary file %s with hash: %s",
                            tf.name,
                            hash_,
                        )
                    await malware_response.upload_file(
                        file, upload_reason=reason
                    )
            except Exception as e:
                logger.warning(str(e))
