"""
This program is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License,
or (at your option) any later version.


This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
See the GNU General Public License for more details.


You should have received a copy of the GNU General Public License
 along with this program.  If not, see <https://www.gnu.org/licenses/>.

Copyright © 2019 Cloud Linux Software Inc.

This software is also available under ImunifyAV commercial license,
see <https://www.imunify360.com/legal/eula>

MalwareIgnorePath plugin will update malware paths filters.

It listens for MalwareIgnorePathUpdated message, and calls
realtime_av.update() on receiving it. This call also is performed on
plugin initialization.
"""

import asyncio
import logging
from contextlib import suppress

from defence360agent.contracts.config import Malware
from defence360agent.contracts.messages import MessageType
from defence360agent.contracts.plugins import MessageSink, expect
from defence360agent.files import REALTIME_AV_CONF
from defence360agent.subsys.persistent_state import load_state, save_state
from defence360agent.utils import recurring_check
from imav.subsys import realtime_av

logger = logging.getLogger(__name__)


class BaseMalwareFilters(MessageSink):
    def __init__(self):
        self._reload_task = None
        _state = load_state("MalwareFilters").get("crontabs_scan_enabled")
        self._crontabs_scan_enabled = _state or Malware.CRONTABS_SCAN_ENABLED
        self._reload_event = asyncio.Event()

    async def create_sink(self, loop) -> None:
        self._loop = loop
        self._reload_task = self._loop.create_task(self.reload_services())
        await self.call_update()

    @recurring_check(0)
    async def reload_services(self):
        await self._reload_event.wait()
        self._reload_event.clear()
        if not realtime_av.is_installed():
            logger.warning("Realtime is not installed. Nothing to reload.")
            return
        try:
            await realtime_av.reload_services()
        except Exception as e:
            logger.warning("failed to reload realtime services: %s", e)

    async def shutdown(self):  # pragma: no cover
        save_state(
            "MalwareFilters",
            {"crontabs_scan_enabled": self._crontabs_scan_enabled},
        )
        if self._reload_task:
            self._reload_task.cancel()
            with suppress(asyncio.CancelledError):
                await self._reload_task

    @expect(MessageType.ConfigUpdate)
    async def on_config_update(self, _) -> None:
        enabled = Malware.CRONTABS_SCAN_ENABLED
        if self._crontabs_scan_enabled != enabled:
            self._crontabs_scan_enabled = enabled
            await self.call_update()

    @expect(MessageType.MalwareIgnorePathUpdated)
    @expect(MessageType.FilesUpdated, files_type=REALTIME_AV_CONF)
    async def on_message(self, _) -> None:
        await self.call_update()

    @expect(
        MessageType.EnsureServiceState,
        service=realtime_av.REALTIME_SERVICE_NAME,
    )
    async def ensure_realtime_service_state(self, _) -> None:
        realtime_av.generate_configs()
        self._reload_event.set()

    async def call_update(self) -> None:
        try:
            if realtime_av.generate_configs():
                # reload services only if configs were changed
                self._reload_event.set()
        except Exception as e:
            logger.warning("failed to update malware path filters: %s", e)
