"""
This program is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License,
or (at your option) any later version.


This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
See the GNU General Public License for more details.


You should have received a copy of the GNU General Public License
 along with this program.  If not, see <https://www.gnu.org/licenses/>.

Copyright © 2019 Cloud Linux Software Inc.

This software is also available under ImunifyAV commercial license,
see <https://www.imunify360.com/legal/eula>
"""
from logging import getLogger

from defence360agent.contracts.config import Malware as Config
from defence360agent.utils import recurring_check
from defence360agent.contracts.plugins import (
    MessageSink,
    MessageSource,
)
from imav.malwarelib.subsys import pure_ftpd
from defence360agent.subsys.panels import hosting_panel
from defence360agent.utils import Scope

logger = getLogger(__name__)


class Conflicts(MessageSource, MessageSink):
    SCOPE = Scope.IM360

    def __init__(self):
        self._loop = None
        self._sink = None
        self._pure_check_task = None

    async def create_sink(self, loop):
        pass

    async def create_source(self, loop, sink):
        self._loop = loop
        self._sink = sink

        self._pure_check_task = self._loop.create_task(
            recurring_check(Config.CONFLICTS_CHECK_PERIOD)(self._check_pure)()
        )

    async def shutdown(self):
        for t in [self._pure_check_task]:
            t.cancel()
            await t

    @staticmethod
    async def _check_pure():
        if not pure_ftpd.detect():
            return  # pure-ftpd is not installed

        # ftp scan integration is enabled in config
        active_in_config = Config.PURE_SCAN
        # should ftp scan integration be enabled
        enabled = active_in_config and not Config.INOTIFY_ENABLED
        running = await pure_ftpd.uploadscript_status()
        thirdparty_uploadscript = pure_ftpd.thirdparty_uploadscript()

        if thirdparty_uploadscript is not None:
            if running:
                logger.info(
                    "Third-party pure-uploadscript detected: %s,"
                    " disabling pure-ftpd scans",
                    thirdparty_uploadscript,
                )
                await pure_ftpd.uploadscript_disable()
            return

        panel = hosting_panel.HostingPanel()
        if enabled:
            if not running:
                logger.info("Enabling pure-ftpd service")
                await pure_ftpd.uploadscript_enable()
            if not pure_ftpd.scan_in_config_enabled(panel):
                logger.info("Enable upload script in pure-ftpd configs")
                await pure_ftpd.enable_scan_in_config(panel)
            # active & enabled & configured & running
            logger.info("FTP scan is enabled")
        elif running and not enabled:
            logger.info("Disabling pure-ftpd scans")
            await pure_ftpd.disable_purescan(panel)
