"""
This program is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License,
or (at your option) any later version.


This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
See the GNU General Public License for more details.


You should have received a copy of the GNU General Public License
 along with this program.  If not, see <https://www.gnu.org/licenses/>.

Copyright © 2019 Cloud Linux Software Inc.

This software is also available under ImunifyAV commercial license,
see <https://www.imunify360.com/legal/eula>
"""
from contextlib import ExitStack, contextmanager
from tempfile import NamedTemporaryFile
from typing import List

from peewee import (
    BooleanField,
    CharField,
    ForeignKeyField,
    IntegerField,
    Model,
    SqliteDatabase,
)


class User(Model):
    class Meta:
        db_table = "users"

    name = CharField()
    parent = CharField(default="")
    email = CharField(default="")
    language = CharField(default="")
    suspended = BooleanField(default=False)
    level = IntegerField(default=0)


class Domain(Model):
    class Meta:
        db_table = "domains"

    name = CharField()
    user = ForeignKeyField(User, backref="domains")


class Path(Model):
    class Meta:
        db_table = "paths"

    name = CharField()
    domain = ForeignKeyField(Domain, backref="paths")


@contextmanager
def patchman_db(models: List[Model], *, target_dir: str, prefix: str = None):
    tmp_file = NamedTemporaryFile(
        suffix=".db",
        prefix="patchman_" + f"{prefix}_" if prefix else "",
        dir=target_dir,
        delete=False,
    )
    db = SqliteDatabase(tmp_file.name)

    stack = ExitStack()
    stack.enter_context(db)
    stack.enter_context(db.bind_ctx(models))

    with stack:
        db.create_tables(models)
        yield db
