"""
This program is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License,
or (at your option) any later version.


This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
See the GNU General Public License for more details.


You should have received a copy of the GNU General Public License
 along with this program.  If not, see <https://www.gnu.org/licenses/>.

Copyright © 2019 Cloud Linux Software Inc.

This software is also available under ImunifyAV commercial license,
see <https://www.imunify360.com/legal/eula>
"""
from imav.patchman.fs_scanner.matcher import HashesMatcher


class DummyDB:
    def __init__(self):
        self.inserted = []

        class DummyTable:
            def __init__(self, outer):
                self.outer = outer

            def buffered_insert(self, row):
                self.outer.inserted.append(row)

        self.hashes_matches = DummyTable(self)


def test_hashes_matcher_filters_and_matches(tmp_path):
    hashes_content = (
        "1:1000:deadbeef:0\n"  # malware, should be ignored
        "2:2000:goodhash1:0\n"  # valid
        "2:2001:goodhash2:2\n"  # superseded, should be ignored
        "7:3000:deadbeef2:0\n"  # malware rule, should be ignored
        "8:4000:deadbeef3:0\n"  # malware rule dryrun, should be ignored
        "2:2002:goodhash3:1\n"  # valid
        "2:2002:goodhash3:1\n"  # valid, duplicate, should be ignored
    )
    hashes_file = tmp_path / "hashes"
    hashes_file.write_text(hashes_content)

    file1 = tmp_path / "file1.txt"
    file1.write_text("data1")
    file2 = tmp_path / "file2.txt"
    file2.write_text("data2")

    class DummyHashCalculator:
        def calc_hash(self, path, apply_normalization=False):
            if path == str(file1):
                return "goodhash1"
            if path == str(file2):
                return "goodhash3"
            return "nope"

    matcher = HashesMatcher(str(hashes_file))
    db = DummyDB()

    matched1 = matcher.match_and_save(
        str(file1), "file1.txt", db, DummyHashCalculator()
    )
    matched2 = matcher.match_and_save(
        str(file2), "file2.txt", db, DummyHashCalculator()
    )
    matched3 = matcher.match_and_save(
        str(tmp_path / "file3.txt"), "file3.txt", db, DummyHashCalculator()
    )
    matched4 = matcher.match_and_save(
        str(file2), "file2.txt", db, DummyHashCalculator()
    )

    assert matched1 is True
    assert matched2 is True
    assert matched3 is False
    assert matched4 is False
    assert len(db.inserted) == 2
    assert db.inserted[0] == (str(file1), "goodhash1", 2, 2000, 0)
    assert db.inserted[1] == (str(file2), "goodhash3", 2, 2002, 1)
