"""
This program is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License,
or (at your option) any later version.


This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
See the GNU General Public License for more details.


You should have received a copy of the GNU General Public License
 along with this program.  If not, see <https://www.gnu.org/licenses/>.

Copyright © 2019 Cloud Linux Software Inc.

This software is also available under ImunifyAV commercial license,
see <https://www.imunify360.com/legal/eula>
"""
import os

from defence360agent.contracts.config import Core, ANTIVIRUS_MODE
from defence360agent.utils import run_cmd_and_log, OsReleaseInfo
from defence360agent.subsys.features.abstract_feature import (
    AbstractFeature,
    FeatureError,
    FeatureStatus,
)
from imav.patchman.constants import PATCHMAN_PACKAGE
from imav.subsys.realtime_av import (
    REALTIME_PACKAGE,
    is_installed as is_realtime_av_installed,
)
from imav.patchman.state import is_installed as is_patchman_installed


class _AbstractPackageFeature(AbstractFeature):
    """
    Provides an interface for installing *PACKAGE_NAME* in the background
    """

    NAME: str = None
    PACKAGE_NAME: str = None
    INSTALL_LOG_FILE_MASK: str = None
    REMOVE_LOG_FILE_MASK: str = None

    @property
    def package_manager(self):
        return (
            "apt-get"
            if OsReleaseInfo.id_like() & OsReleaseInfo.DEBIAN
            else "yum"
        )

    async def _check_installed_impl(self) -> bool:
        return self.is_installed

    @AbstractFeature.raise_if_shouldnt_install_now
    async def install(self):
        return await run_cmd_and_log(
            f"{self.package_manager} -y install {self.PACKAGE_NAME}",
            self.INSTALL_LOG_FILE_MASK,
            env=dict(os.environ, DEBIAN_FRONTEND="noninteractive"),
        )

    @AbstractFeature.raise_if_shouldnt_remove_now
    async def remove(self):
        return await run_cmd_and_log(
            f"{self.package_manager} -y remove {self.PACKAGE_NAME}",
            self.REMOVE_LOG_FILE_MASK,
        )

    async def has_status(self, status: FeatureStatus) -> bool:
        current_status = (await self.status())["items"]["status"]
        return current_status == status


class PatchmanFeature(_AbstractPackageFeature):
    NAME = "patchman"
    PACKAGE_NAME = PATCHMAN_PACKAGE
    INSTALL_LOG_FILE_MASK = f"/var/log/{Core.PRODUCT}/install-patchman.log.*"
    REMOVE_LOG_FILE_MASK = f"/var/log/{Core.PRODUCT}/remove-patchman.log.*"

    @property
    def is_installed(self):
        return is_patchman_installed()


class RealtimeFeature(_AbstractPackageFeature):
    NAME = "realtime-av"
    PACKAGE_NAME = REALTIME_PACKAGE
    INSTALL_LOG_FILE_MASK = (
        f"/var/log/{Core.PRODUCT}/install-realtime-av.log.*"
    )
    REMOVE_LOG_FILE_MASK = f"/var/log/{Core.PRODUCT}/remove-realtime-av.log.*"

    @property
    def is_installed(self) -> bool:
        return is_realtime_av_installed()

    async def remove(self):
        if not ANTIVIRUS_MODE:
            raise FeatureError(
                "Can't remove imunify-realtime-av, "
                "since it is a part of imunify360-firewall"
            )
        await super().remove()
