"""
This program is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License,
or (at your option) any later version.


This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
See the GNU General Public License for more details.


You should have received a copy of the GNU General Public License
 along with this program.  If not, see <https://www.gnu.org/licenses/>.

Copyright © 2019 Cloud Linux Software Inc.

This software is also available under ImunifyAV commercial license,
see <https://www.imunify360.com/legal/eula>
"""
import configparser
from pathlib import Path
from typing import Optional


class PatchmanConfig:
    PATH = Path("/etc/patchman/patchman.ini")

    @classmethod
    def _get_config(cls) -> dict:
        conf = configparser.ConfigParser()
        if cls.PATH.exists():
            conf.read(cls.PATH)
        return conf

    @classmethod
    def get(cls, section: str, option: str, default=None) -> Optional[str]:
        """
        Return *option* value in *section* in config if exist,
        None otherwise.
        """
        conf = cls._get_config()
        try:
            return conf.get(section, option)
        except configparser.Error:
            return default

    @classmethod
    def set(cls, section: str, option: str, value: str) -> None:
        """
        Set *value* for the *option* of the corresponding *section*
        in the patchman config.
        """
        conf = cls._get_config()
        if not conf.has_section(section):
            conf.add_section(section)
        conf.set(section, option, value)
        with cls.PATH.open("w") as config_file:
            conf.write(config_file)
