"""
This program is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License,
or (at your option) any later version.


This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
See the GNU General Public License for more details.


You should have received a copy of the GNU General Public License
 along with this program.  If not, see <https://www.gnu.org/licenses/>.

Copyright © 2019 Cloud Linux Software Inc.

This software is also available under ImunifyAV commercial license,
see <https://www.imunify360.com/legal/eula>
"""
import logging

from imav.malwarelib.utils import malware_response
from imav.malwarelib.config import PENDING

logger = logging.getLogger(__name__)

FALSE_POSITIVE = "fp"
FALSE_NEGATIVE = "fn"

SUBMIT_SUCCESS = "OK"
SUBMIT_PENDING = PENDING
SUBMIT_FAILED = "failed"


async def submit_malware(filename, type, reason=None):
    """
    Submit file to MRS for analysis
    :param filename: str -> path to file to submit
    :param type: str -> report type (fp or fn)
    :param scanner: int -> scanner id
    :return: one of SUBMIT_SUCCESS, SUBMIT_PENDING, SUBMIT_FAILED
    :raises LicenseError:
    """

    # submit to MRS
    # Convert CLI format to short format
    upload_reason = (
        malware_response.FALSE_POSITIVE
        if type == FALSE_POSITIVE
        else malware_response.FALSE_NEGATIVE
    )
    try:
        await malware_response.upload_with_retries(
            filename,
            upload_reason=upload_reason,
            notify_timeout=malware_response.SUBMIT_TIMEOUT,
        )
        return SUBMIT_SUCCESS
    except malware_response.TimeoutError as e:
        logger.warning(
            f"File {filename} uploading timed out. Marking as pending. {e}"
        )
        return SUBMIT_PENDING
    except malware_response.ClientUploadError as e:
        logger.error(f"File {filename} uploading failed. {e}")
        return SUBMIT_FAILED
