"""
This program is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License,
or (at your option) any later version.


This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
See the GNU General Public License for more details.


You should have received a copy of the GNU General Public License
 along with this program.  If not, see <https://www.gnu.org/licenses/>.

Copyright © 2019 Cloud Linux Software Inc.

This software is also available under ImunifyAV commercial license,
see <https://www.imunify360.com/legal/eula>

 chattr.py
This is a module for managing file attributes via ioctl.
Used IOCTL request constants:
    FS_IOC_GETFLAGS - get fs flags
    FS_IOC_SETFLAGS - set fs flags
Supported attributes:
    FS_IMMUTABLE_FL - immutable attribute
Refer to include/linux/fs.h for values of above constants:
https://elixir.bootlin.com/linux/v2.6.32.71/source/include/linux/fs.h
"""

import sys
from array import array
from fcntl import ioctl

# IOCTL requests
FS_IOC_SETFLAGS = 0x40086602
FS_IOC_GETFLAGS = 0x80086601

# File attribute flags
FS_IMMUTABLE_FL = 0x00000010


def get_flags(fd):
    """Get attributes of a file.
    :param fd: descriptor of a file to examine, read-only access is enough
    :return: flags as integer value
    """
    flags_ptr = array("i", [0])
    ioctl(fd, FS_IOC_GETFLAGS, flags_ptr, 1)
    return int.from_bytes(flags_ptr, byteorder=sys.byteorder)


def _set_flags(fd, flags):
    """Replace existing file attributes with the ones provided.
    Calling _set_flags() directly is most likely NOT what you want. Consider
     using append_flags() and subtract_flags().
    :param fd: descriptor of a file to modify, read-only access is enough
    :param flags: attribute flags combined by bitwise OR operator |
    """
    flags_ptr = array("i", [flags])
    ioctl(fd, FS_IOC_SETFLAGS, flags_ptr)


def append_flags(fd, flags):
    """Enable given attributes of a file. See _set_flags() for details."""
    new_flags = get_flags(fd) | flags
    _set_flags(fd, new_flags)


def subtract_flags(fd, flags):
    """Disable given attributes of a file. See _set_flags() for details."""
    new_flags = get_flags(fd) & ~flags
    _set_flags(fd, new_flags)
