"""
This program is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License,
or (at your option) any later version.


This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
See the GNU General Public License for more details.


You should have received a copy of the GNU General Public License
 along with this program.  If not, see <https://www.gnu.org/licenses/>.

Copyright © 2019 Cloud Linux Software Inc.

This software is also available under ImunifyAV commercial license,
see <https://www.imunify360.com/legal/eula>
"""
import hashlib
import os
from datetime import datetime, timedelta

from defence360agent.contracts.config import (
    ConfigFile,
    MalwareScanSchedule,
    MalwareScanScheduleInterval,
)
from defence360agent.contracts.license import LicenseCLN


def get_memory(intensity_ram):
    return str(intensity_ram) + "M"


def hash_path(full_path: str, hash_provider=hashlib.sha256) -> str:
    """Calculate hash from full path"""
    hash_object = hash_provider()
    hash_object.update(os.fsencode(full_path))
    return hash_object.hexdigest()


def reset_malware_schedule(config_file_path=None):
    """
    When ImunifyAV+ falls to free license, we should
    reset scheduled scanner settings to monthly mode
    (daily and weekly modes are not supported in free version)
    """
    if LicenseCLN.is_valid_av_plus():
        return
    if MalwareScanSchedule.INTERVAL not in (
        MalwareScanScheduleInterval.WEEK,
        MalwareScanScheduleInterval.DAY,
    ):
        return
    tomorrow = datetime.today() + timedelta(days=1)
    next_day = tomorrow.day
    new_data = {
        "MALWARE_SCAN_SCHEDULE": {
            "interval": "month",
            "day_of_month": next_day,
            "hour": 3,
        }
    }

    ConfigFile(path=config_file_path).dict_to_config(new_data)
