"""
This program is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License,
or (at your option) any later version.


This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
See the GNU General Public License for more details.


You should have received a copy of the GNU General Public License
 along with this program.  If not, see <https://www.gnu.org/licenses/>.

Copyright © 2019 Cloud Linux Software Inc.

This software is also available under ImunifyAV commercial license,
see <https://www.imunify360.com/legal/eula>
"""
import asyncio
import datetime
from logging import getLogger
from typing import Tuple

from imav.malwarelib.config import MalwareScanType
from imav.malwarelib.scan.scanner import MalwareScanner
from defence360agent.contracts.config import BackupRestore

logger = getLogger(__name__)


def _get_scan_function(sink, tmpdir=None):
    async def _custom_scan_function(files: list) -> list:
        """Used for checking files from backups before their recovering"""
        if not files:
            return []
        still_infected = []
        scanner = MalwareScanner(sink=sink, tmpdir=tmpdir)
        scanner.start(
            files,
            scan_type=MalwareScanType.RESTORE_FROM_BACKUP,
            use_filters=False,
        )
        result = await scanner.async_wait()
        if result is not None:
            still_infected = [r["file_name"] for r in result["results"]]
        return still_infected

    return _custom_scan_function


async def restore_files(
    files: list, until: datetime, sink, tmp_dir=None
) -> Tuple[list, list]:
    from restore_infected.restore import async_restore_infected

    backend = BackupRestore.backup_system()
    assert backend, "Backup system is not selected!"
    restored, not_restored = [], files

    try:
        scan_function = _get_scan_function(sink)
        restored, not_restored = await async_restore_infected(
            backend=backend,
            files=files,
            until=until,
            scan_func=scan_function,
            tmp_dir=tmp_dir,
        )
    except asyncio.CancelledError:
        raise
    except Exception:
        logger.exception("Error during restore from backup")
    return restored, not_restored
