"""
This program is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License,
or (at your option) any later version.


This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
See the GNU General Public License for more details.


You should have received a copy of the GNU General Public License
 along with this program.  If not, see <https://www.gnu.org/licenses/>.

Copyright © 2019 Cloud Linux Software Inc.

This software is also available under ImunifyAV commercial license,
see <https://www.imunify360.com/legal/eula>
"""
import asyncio
import logging

from imav.malwarelib.config import (
    MalwareScanResourceType,
    MalwareScanType,
    QueuedScanState,
)
from imav.malwarelib.scan.detached import DetachedState
from imav.malwarelib.scan.queue import QueuedScanBase, ScanQueue
from defence360agent.utils import create_task_and_log_exceptions
from defence360agent.contracts.messages import MessageType

logger = logging.getLogger(__name__)


class QueueSupervisorSync:
    queue = ScanQueue()

    def __init__(self, loop=None, sink=None):
        self.loop = loop or asyncio.get_event_loop()
        self.sink = sink

        current_scan = self.queue.current_scan
        if (
            current_scan
            and current_scan.detached_scan.get_detached_process_state()
            in (
                DetachedState.RUNNING,
                DetachedState.FINISHED,
            )
        ):
            current_scan.state = QueuedScanState.running

    async def put(
        self,
        paths,
        resource_type: MalwareScanResourceType,
        prioritize=False,
        **scan_args,
    ):
        logger.info(
            "Queuing %s scan for %s", resource_type.value, ", ".join(paths)
        )

        running_scan = self.queue.current_scan

        for path in paths:
            if scan := self.queue.find(path=path, resource_type=resource_type):
                self.queue.remove(scan)

        if (
            running_scan
            and running_scan.path in paths
            and running_scan.resource_type == resource_type
        ):
            await running_scan.detached_scan.handle_aborted_process(
                sink=self.sink, kill=True, scan_started=running_scan.started
            )

        await self.queue.put(
            paths,
            resource_type=resource_type,
            prioritize=prioritize,
            **scan_args,
        )

        self.recheck()

    def clear(self):
        while self.queue:
            self.queue.remove()

    def clear_background(self):
        while True:
            scan = self.queue.find(scan_type=MalwareScanType.BACKGROUND)
            if not scan:
                break
            self.queue.remove(scan)

    def remove(self, scan_ids=None):
        if scan_ids:
            scans = self.queue.find_all(scan_ids)
        else:
            scans = [scan] if (scan := self.queue.current_scan) else []

        paths = [s.path for s in scans]
        logger.info("Removing scans from the queue for %s", ", ".join(paths))

        for scan in scans:
            self.queue.remove(scan)

        self.recheck()

    def recheck(self):
        if not self.queue:
            return

        scan = self.queue.current_scan
        if scan is not None:
            if scan.state == QueuedScanState.running:
                return

            if scan.state == QueuedScanState.stopped:
                self.queue.remove(scan)

        if self.queue and self.queue.current_scan:
            if self.queue.current_scan.state == QueuedScanState.queued:
                self._create_scan_task()

    def is_empty(self):
        return not bool(self.queue)

    def status(self):
        active_scan = self.queue.current_scan
        if active_scan:
            return {
                **active_scan.status(),
                "queued": len(self.queue) - 1,
                "resource_type": active_scan.resource_type.value,
                "status": QueuedScanState.running.value,
            }
        return {
            "queued": 0,
            "status": QueuedScanState.stopped.value,
        }

    def get_scans_from_paths(self, paths):
        return self.queue.get_scans_from_paths(paths)

    def _create_scan_task(self):
        scan = self.queue.current_scan
        scan.state = QueuedScanState.running
        create_task_and_log_exceptions(self.loop, self._scan_coro, scan)

    async def _scan_coro(self, scan: QueuedScanBase):
        scan_data = await scan.start()
        if scan_data is not None:
            await self.sink.process_message(
                MessageType.MalwareScan(**scan_data)
            )
        else:
            self.recheck()

    async def update_scan_status(self, scan_ids, status):
        self.queue.update(scan_ids, status)
