"""
This program is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License,
or (at your option) any later version.


This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
See the GNU General Public License for more details.


You should have received a copy of the GNU General Public License
 along with this program.  If not, see <https://www.gnu.org/licenses/>.

Copyright © 2019 Cloud Linux Software Inc.

This software is also available under ImunifyAV commercial license,
see <https://www.imunify360.com/legal/eula>
"""
import subprocess
from asyncio import CancelledError
from base64 import b64encode
from codecs import encode
from contextlib import suppress
from typing import Dict, Iterator

from defence360agent.contracts.config import FileBasedResourceLimits
from imav.malwarelib.config import APP_VERSION_DETECTOR_INTENSITY_KEY
from imav.malwarelib.scan import ScanFailedError
from defence360agent.utils import resource_limits


class AVDExecutionError(ScanFailedError):
    pass


class AppVersionDetector:
    _cmd = [
        "/opt/app-version-detector/app-version-detector-wrapper.sh",
        # DB is hardcoded in MDS
        (
            "--sqlite-db-report="
            "/var/lib/cloudlinux-app-version-detector/components_versions.sqlite3"  # noqa: E501
        ),
        "--stdin-dirs",
        "--paths-in-base64",
        "--scan-depth=3",
    ]

    def __init__(self):
        self._proc = None

    async def start(
        self, paths: Iterator[bytes], doc_roots: Dict[str, str]
    ) -> None:
        # add domain to known doc_root directories for compatibility
        # with the result of running update_components_version.py script
        domain_path_pairs = []
        for path in paths:
            domain_path_pairs.append(
                b",".join(
                    [
                        encode(doc_roots.get(path.decode(), ""), "idna"),
                        b64encode(path),
                    ]
                )
            )
        stdin = b"\n".join(domain_path_pairs)
        self._proc = await resource_limits.create_subprocess(
            self._cmd,
            intensity_cpu=FileBasedResourceLimits.CPU,
            intensity_io=FileBasedResourceLimits.IO,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            stdin=subprocess.PIPE,
            key=APP_VERSION_DETECTOR_INTENSITY_KEY,
        )

        out = err = None
        try:
            out, err = await self._proc.communicate(input=stdin)
        except CancelledError:
            with suppress(ProcessLookupError):
                self._proc.kill()
                await self._proc.wait()

        if self._proc.returncode != 0:
            raise AVDExecutionError(
                message="AVD exited with non-zero return code",
                command=self._cmd,
                return_code=self._proc.returncode,
                out=out,
                err=err,
                path=None,
                scan_id=None,
            )
