"""
This program is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License,
or (at your option) any later version.


This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
See the GNU General Public License for more details.


You should have received a copy of the GNU General Public License
 along with this program.  If not, see <https://www.gnu.org/licenses/>.

Copyright © 2019 Cloud Linux Software Inc.

This software is also available under ImunifyAV commercial license,
see <https://www.imunify360.com/legal/eula>
"""
import logging
from collections import defaultdict
from typing import Iterable
from urllib.parse import urljoin
from urllib.request import Request

from defence360agent.api.server import API, APIError
from defence360agent.internals.iaid import (
    IAIDTokenError,
    IndependentAgentIDAPI,
)

logger = logging.getLogger(__name__)


class VulnerabilityAPI(API):
    URL = urljoin(API._BASE_URL, "/api/patch/vulnerabilities")
    _STUB_VULNERABILITY_INFO = {
        "cveId": "",
        "app": "",
        "type": "",
        "name": "",
        "severity": "",
    }

    @classmethod
    async def get_details(cls, ids: Iterable) -> dict:
        """
        Get vulnerabilities details for specific *ids*.
        More details in DEF-32152
        """
        info = defaultdict(cls._STUB_VULNERABILITY_INFO.copy)
        if not ids:
            return info

        try:
            token = await IndependentAgentIDAPI.get_token()
        except IAIDTokenError as exc:
            logger.error(
                "Can't get iaid token: %s. "
                "Return default vulnerabilities details.",
                exc,
            )
            return info

        url = cls.URL + f"?ids={','.join(ids)}"
        request = Request(
            url,
            headers={"X-Auth": token, "Content-Type": "application/json"},
        )
        try:
            result = await cls.async_request(request)
        except APIError as exc:
            logger.error(
                "Failed to get vulnerabilities details: %s. "
                "Return default vulnerabilities details.",
                exc,
            )
            return info

        info.update(normalize_vulnerabilities(result["vulnerabilities"]))
        return info


def normalize_vulnerabilities(
    vulnerabilities: dict[str, dict[str, str]],
) -> dict[str, dict[str, str]]:
    """
    Normalize vulnerabilities data.
    """
    return {
        _id: normalize_vulnerability(vulnerability)
        for _id, vulnerability in vulnerabilities.items()
    }


def normalize_vulnerability(vulnerability: dict[str, str]) -> dict[str, str]:
    """
    Normalize vulnerability data.
    """
    return {
        "cveId": vulnerability.get("cveId", ""),
        "app": vulnerability.get("app", ""),
        "type": vulnerability.get("type", ""),
        "name": vulnerability.get("name", ""),
        "severity": vulnerability.get("severity", "").upper(),
    }
