"""
This program is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License,
or (at your option) any later version.


This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
See the GNU General Public License for more details.


You should have received a copy of the GNU General Public License
 along with this program.  If not, see <https://www.gnu.org/licenses/>.

Copyright © 2019 Cloud Linux Software Inc.

This software is also available under ImunifyAV commercial license,
see <https://www.imunify360.com/legal/eula>
"""
import peewee as pw

from defence360agent.model.simplification import FilenameField


def migrate(migrator, database, fake=False, **kwargs):
    MalwareScan = migrator.orm["malware_scans"]

    class VulnerabilityHit(pw.Model):
        """Represents a vulnerable file."""

        class Meta:
            db_table = "vulnerability_hits"

        #: An id of a scan, unique per server.
        id = pw.PrimaryKeyField()
        #: A reference to :class:`MalwareScan`.
        scanid = pw.ForeignKeyField(
            MalwareScan,
            null=False,
            related_name="vulnerabilities",
            on_delete="CASCADE",
        )
        #: The owner of the file.
        owner = pw.CharField(null=False)
        #: The user a file belongs to (is in user's home but owned by another user)
        user = pw.CharField(null=False)
        #: The original path to the file.
        orig_file = FilenameField(null=False)
        #: The type of infection (signature).
        type = pw.CharField(null=False)
        #: The hash of the files as provided by AI-BOLIT.
        hash = pw.CharField(null=True)
        #: The size of the file.
        size = pw.CharField(null=True)
        #: The exact timestamp when AI-BOLIT has detected the file.
        timestamp = pw.FloatField(null=True)
        #: The current status of the file.
        #: Must be one of :class:`.VulnerabilityHitStatus`.
        status = pw.CharField(default="vulnerable")
        #: Timestamp when the file was last patched.
        patched_at = pw.FloatField(null=True)

    migrator.create_model(VulnerabilityHit)


def rollback(migrator, database, fake=False, **kwargs):
    VulnerabilityHit = migrator.orm["vulnerability_hits"]
    migrator.remove_model(VulnerabilityHit)
