"""
This program is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License,
or (at your option) any later version.


This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
See the GNU General Public License for more details.


You should have received a copy of the GNU General Public License
 along with this program.  If not, see <https://www.gnu.org/licenses/>.

Copyright © 2019 Cloud Linux Software Inc.

This software is also available under ImunifyAV commercial license,
see <https://www.imunify360.com/legal/eula>
"""
import logging
import os

from peewee import IntegerField, Model, SqliteDatabase, TextField

from imav.migration_utils.other import skip_for_im360
from imav.migration_utils.revisium import find_revisium_db, get_all_domains

logger = logging.getLogger(__name__)

ra = SqliteDatabase(None)


class HashException(ra.Model):
    domain_id = IntegerField()
    file_sub_path = TextField()

    class Meta:
        db_table = "hash_exception"


# noinspection PyPep8Naming
class db:
    MalwareIgnorePath: Model

    @classmethod
    def init(cls, migrator):
        cls.MalwareIgnorePath = migrator.orm["malware_ignore_path"]


def get_ignore_list(database) -> set[str]:
    ra.init(database)

    try:
        domains = get_all_domains()
    except Exception as e:
        logger.warning("Failed to get Plesk domains: %r", e)
        return set()

    ignore_list = set()
    for hash_exception in HashException:
        try:
            doc_root = domains[str(hash_exception.domain_id)]["document_root"]
        except KeyError:
            continue

        file_sub_path = hash_exception.file_sub_path.lstrip(os.path.sep)
        path = os.path.join(doc_root, file_sub_path)
        ignore_list.add(path)

    return ignore_list


@skip_for_im360
def migrate(migrator, database, fake=False, **kwargs):
    if fake:
        return

    revisium_db_path = find_revisium_db()
    if revisium_db_path is None:
        logger.info("No legacy ImunifyAV database found. Skipping...")
        return

    ignore_list = get_ignore_list(revisium_db_path)

    db.init(migrator)
    for path in ignore_list:
        try:
            db.MalwareIgnorePath.create(path=path, resource_type="file")
        except Exception as e:
            logger.warning("Failed to add %r to the ignore list: %r", path, e)


@skip_for_im360
def rollback(migrator, database, fake=False, **kwargs):
    pass
