"""
This program is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License,
or (at your option) any later version.


This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
See the GNU General Public License for more details.


You should have received a copy of the GNU General Public License
 along with this program.  If not, see <https://www.gnu.org/licenses/>.

Copyright © 2019 Cloud Linux Software Inc.

This software is also available under ImunifyAV commercial license,
see <https://www.imunify360.com/legal/eula>
"""
import logging
import os
from functools import lru_cache
from glob import iglob
from hashlib import sha1 as hash_func
from itertools import cycle
from pathlib import Path
from tempfile import NamedTemporaryFile
from typing import Tuple

from defence360agent.utils import run_coro
from imav.malwarelib.cleanup.storage import CleanupStorage
from imav.migration_utils.other import skip_for_im360
from imav.migration_utils.revisium import get_all_domains, get_vhosts_dir

logger = logging.getLogger(__name__)

ENCRYPT_KEY = b"IMUNIFY"

REVISIUM_DIR_PREFIX = ".revisium"
BACKUP_FILE_SUFFIX = ".imunify"
BACKUP_LOCATION = (
    ".revisium_antivirus_cache",
    f"{REVISIUM_DIR_PREFIX}*",
    "backup",
    f"*{BACKUP_FILE_SUFFIX}",
)


@lru_cache(maxsize=1)
def _get_backup_file_slices() -> Tuple[slice, slice, slice]:
    """Get backup file slices for splitting it apart"""
    key_size = len(ENCRYPT_KEY)
    digest_size = len(hash_func().hexdigest())
    meta_size = key_size + digest_size
    return (
        # content: from the beginning to the meta data
        slice(None, -meta_size),
        # key: the first part of the meta data
        slice(-meta_size, -digest_size),
        # digest: the last part of the meta data
        slice(-digest_size, None),
    )


def decrypt(encrypted: bytes) -> bytes:
    """Decrypt ex-Revisium backup file content"""
    decrypted = bytes(c ^ k for c, k in zip(encrypted, cycle(ENCRYPT_KEY)))

    content_slice, key_slice, digest_slice = _get_backup_file_slices()

    content = decrypted[content_slice]
    key = decrypted[key_slice]
    digest = decrypted[digest_slice]

    assert key == ENCRYPT_KEY
    assert hash_func(content).hexdigest() == digest.decode("latin-1")

    return content


def get_orig_filename(filename: str) -> str:
    """Figure out what the original filename of ex-Revisium backup"""
    path = Path(filename)
    domain_id = path.parent.parent.name.removeprefix(REVISIUM_DIR_PREFIX)
    orig_dir = get_all_domains()[domain_id]["document_root"]
    return os.path.join(orig_dir, path.name.removesuffix(BACKUP_FILE_SUFFIX))


def transit_backup(filename: str) -> None:
    """
    Decrypt ex-Revisium backup file and copy it to Imunify360 cleanup storage
    """
    with open(filename, "rb") as f:
        st = os.stat(f.fileno())
        encrypted = f.read()

    decrypted = decrypt(encrypted)

    with NamedTemporaryFile() as temp:
        temp.write(decrypted)
        temp.flush()

        fd = temp.fileno()
        os.chmod(fd, st.st_mode)
        os.chown(fd, st.st_uid, st.st_gid)

        orig = get_orig_filename(filename)
        dst = CleanupStorage.path / CleanupStorage.storage_name(orig)
        # noinspection PyProtectedMember,PyTypeChecker
        run_coro(
            CleanupStorage._copy(temp.name, dst, safe_src=True, safe_dst=True)
        )


def main() -> None:
    for file in iglob(os.path.join(get_vhosts_dir(), "*", *BACKUP_LOCATION)):
        try:
            transit_backup(file)
        except Exception as e:
            logger.warning("Failed to transit a backup file %r: %r", file, e)


@skip_for_im360
def migrate(migrator, database, fake=False, **kwargs):
    if fake:
        return

    main()


@skip_for_im360
def rollback(migrator, database, fake=False, **kwargs):
    pass


if __name__ == "__main__":
    main()
