"""
This program is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License,
or (at your option) any later version.


This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
See the GNU General Public License for more details.


You should have received a copy of the GNU General Public License
 along with this program.  If not, see <https://www.gnu.org/licenses/>.

Copyright © 2019 Cloud Linux Software Inc.

This software is also available under ImunifyAV commercial license,
see <https://www.imunify360.com/legal/eula>
"""
import logging
import time
from peewee import SqliteDatabase, TextField
from imav.migration_utils.revisium import find_revisium_db
from imav.migration_utils.settings import SettingsExtractor

logger = logging.getLogger(__name__)

db = SqliteDatabase(None)


class KeyValue(db.Model):
    """ex-Revisium settings database model"""

    key = TextField()
    value = TextField()

    class Meta:
        primary_key = False


class Revisium(SettingsExtractor):
    """ex-Revisium database settings extractor"""

    def _get(self):
        return KeyValue.get(KeyValue.key == self.key).value


def migrate_eula(migrator, database=None):
    db.init(database)
    eula_value = Revisium("ra_eula", 0).get()
    last_updated_eula_date = "2021-06-29"
    logger.info("EULA value=%s from Revisium database", str(eula_value))
    if eula_value >= 2:
        logger.info("Accepting eula for date=%s", last_updated_eula_date)
        Eula = migrator.orm["eula"]
        Eula.insert(
            updated=last_updated_eula_date, accepted=time.time()
        ).execute()


def migrate(migrator, database, fake=False, **kwargs):
    if fake:
        return

    revisium_db_path = find_revisium_db()
    if revisium_db_path is None:
        logger.info("No legacy ImunifyAV database found. Skipping...")
        return

    try:
        migrate_eula(migrator, revisium_db_path)
    except Exception as e:
        logger.warning("Failed to migrate eula for ImunifyAV: %r", e)


def rollback(migrator, database, fake=False, **kwargs):
    pass


if __name__ == "__main__":
    print(migrate_eula(find_revisium_db()))
