"""
This program is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License,
or (at your option) any later version.


This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
See the GNU General Public License for more details.


You should have received a copy of the GNU General Public License
 along with this program.  If not, see <https://www.gnu.org/licenses/>.

Copyright © 2019 Cloud Linux Software Inc.

This software is also available under ImunifyAV commercial license,
see <https://www.imunify360.com/legal/eula>
"""
import os
from functools import lru_cache
from typing import Dict

from peewee import SqliteDatabase

from imav.migration_utils.plesk_sdk import PleskSdk

PLESK_CONFIG = "/etc/psa/psa.conf"
PLESK_CONFIG_DEFAULT = "/etc/psa/psa.conf.default"


def _psa_get(conf_file: str, key: str) -> str:
    """Get a specified value from Plesk config file"""
    with open(conf_file) as f:
        for line in f:
            line = line.strip()
            if line and not line.startswith("#"):
                k, value = line.split()
                if k == key:
                    return value


def get_vhosts_dir() -> str:
    """Get virtual hosts directory from Plesk config"""
    vhosts_dir = "/var/www/vhosts"  # the default

    for conf_file in (PLESK_CONFIG_DEFAULT, PLESK_CONFIG):
        # noinspection PyBroadException
        try:
            vhosts_dir = _psa_get(conf_file, "HTTPD_VHOSTS_D")
        except Exception:
            pass

    return vhosts_dir


@lru_cache(maxsize=1)
def find_revisium_db():
    # noinspection PyBroadException
    try:
        revisium_dir = PleskSdk.context__get_var_dir()
        assert revisium_dir
        revisium_db_path = os.path.join(revisium_dir, "ra.db")
        assert os.path.exists(revisium_db_path)
    except Exception:
        return None

    return revisium_db_path


@lru_cache(maxsize=1)
def get_all_domains() -> Dict[str, Dict[str, str]]:
    """Retrieve all domains from Plesk"""
    return PleskSdk.domain__get_all_domains()


def get_domain_docroot(domain_id: str) -> str:
    return get_all_domains()[domain_id]["document_root"]


@lru_cache(maxsize=1)
def get_revisium_db() -> SqliteDatabase:
    db_path = find_revisium_db()
    db = SqliteDatabase(db_path)
    return db
