"""
This program is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License,
or (at your option) any later version.


This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
See the GNU General Public License for more details.


You should have received a copy of the GNU General Public License
 along with this program.  If not, see <https://www.gnu.org/licenses/>.

Copyright © 2019 Cloud Linux Software Inc.

This software is also available under ImunifyAV commercial license,
see <https://www.imunify360.com/legal/eula>
"""
from __future__ import annotations

from typing import NamedTuple
from peewee import CharField, IntegerField, TimestampField
from defence360agent.model import instance, Model


class WPSite(NamedTuple):
    docroot: str
    domain: str
    uid: int
    version: str = "1.0.0"

    @classmethod
    def from_wordpress_site(cls, site: WordpressSite) -> WPSite:
        """Create a WPSite instance from a WordpressSite instance."""
        return cls(
            docroot=site.docroot,
            domain=site.domain,
            uid=site.uid,
            version=site.version,
        )

    def build_with_version(self, version: str) -> WPSite:
        """Create a new WPSite instance with an updated version."""
        return WPSite(
            docroot=self.docroot,
            domain=self.domain,
            uid=self.uid,
            version=version,
        )

    def __eq__(self, other):
        if not isinstance(other, WPSite):
            return NotImplemented
        # Ignore version and manually_deleted_at for equality check.
        return (self.docroot, self.domain, self.uid) == (
            other.docroot,
            other.domain,
            other.uid,
        )

    def __hash__(self):
        return hash((self.docroot, self.domain, self.uid))


class WordpressSite(Model):
    class Meta:
        database = instance.db
        db_table = "wordpress_site"

    docroot = CharField(primary_key=True, null=False)
    domain = CharField(null=False)
    uid = IntegerField(null=False)
    manually_deleted_at = TimestampField(default=None, null=True)
    version = CharField(default="1.0.0", null=False)
