<?php
declare(strict_types = 1);

/**
 * Gestionnaire de cookies.
 *
 * @license https://www.gnu.org/licenses/gpl-3.0.html
 * @link https://www.igalerie.org/
 */
class Cookie
{
	/**
	 * Nom du cookie.
	 *
	 * @var string
	 */
	public $name;

	/**
	 * Contenu du cookie.
	 *
	 * @var array
	 */
	public $value = [];



	/**
	 * Initialisation : récupération du contenu du cookie.
	 *
	 * @param string $name
	 *   Nom du cookie.
	 *
	 * @return void
	 */
	public function __construct(string $name)
	{
		$this->name = $name;

		if (isset($_COOKIE[$name]) && Utility::isJson($_COOKIE[$name]))
		{
			$this->value = Utility::jsonDecode($_COOKIE[$name]);
		}
	}

	/**
	 * Ajoute des données au cookie.
	 *
	 * @param string $k
	 * @param string $v
	 *
	 * @return void
	 */
	public function add(string $k, string $v): void
	{
		$this->value[$k] = $v;
	}

	/**
	 * Supprime tout ou une partie du cookie.
	 *
	 * @param string $k
	 *
	 * @return void
	 */
	public function delete(string $k = ''): void
	{
		if ($k)
		{
			unset($this->value[$k]);
		}
		else
		{
			$this->value = [];
		}
	}

	/**
	 * Lecture d'une valeur contenue dans le cookie.
	 *
	 * @param string $k
	 *
	 * @return string
	 */
	public function read(string $k): string
	{
		return array_key_exists($k, $this->value) ? (string) $this->value[$k] : '';
	}

	/**
	 * Écriture du cookie.
	 *
	 * @param int $expires
	 *   Durée d'expiration du cookie.
	 *
	 * @return bool
	 */
	public function write(int $expires = 0): bool
	{
		// Encodage des données.
		$value = Utility::jsonEncode($this->value);

		// On supprime les données par sécurité.
		$this->value = [];

		// Durée d'expiration.
		// Ne fonctionne pas comme attendu avec certains navigateurs si $expires = 0.
		$expires = $expires ? time() + $expires : 0;

		// Chemin.
		switch (CONF_GALLERY_PATH)
		{
			case '' :
				$path = App::getGalleryPath() . '/';
				break;

			case '/' :
				$path = '/';
				break;

			default :
				$path = CONF_GALLERY_PATH . '/';
		}

		// Écriture du cookie.
		if (version_compare(PHP_VERSION, '7.3', '>='))
		{
			return setcookie($this->name, $value,
			[
				'expires' => $expires,
				'path' => $path,
				'secure' => GALLERY_HTTPS,
				'httponly' => TRUE,
				'samesite' => 'Lax'
			]);
		}
		return setcookie($this->name, $value, $expires, $path, '', GALLERY_HTTPS, TRUE);
	}
}
?>