<?php
declare(strict_types = 1);

require_once(__DIR__ . '/Admin.class.php');

/**
 * Gestion des tags.
 *
 * @license https://www.gnu.org/licenses/gpl-3.0.html
 * @link https://www.igalerie.org/
 */
class AdminTags extends Admin
{
	/**
	 * Actions sur la sélection.
	 *
	 * @return void
	 */
	public static function actions(): void
	{
		if (isset($_POST['cancel']) || !isset($_POST['selection']))
		{
			return;
		}
		switch (self::_getSelectedIds($selected_ids))
		{
			case 'delete' :
				$r = Tags::delete($selected_ids);
				if ($r < 0)
				{
					Report::error();
				}
				else if ($r < 1)
				{
					Report::info(__('Aucun tag n\'a été supprimé.'));
				}
				else
				{
					Report::success($r > 1
						? sprintf(__('%s tags ont été supprimés.'), $r)
						: __('1 tag a été supprimé.'));
				}
				break;
		}
	}

	/**
	 * Ajoute des tags.
	 *
	 * @return void
	 */
	public static function add(): void
	{
		if (!isset($_POST['new_tags']) || !is_array($_POST['new_tags']))
		{
			return;
		}

		$r = Tags::add($_POST['new_tags']);
		if ($r < 0)
		{
			Report::error();
		}
		else if ($r < 1)
		{
			Report::info(__('Aucun tag n\'a été ajouté.'));
		}
		else
		{
			Report::success($r > 1
				? sprintf(__('%s tags ont été ajoutés.'), $r)
				: __('1 tag a été ajouté.'));
		}
	}

	/**
	 * Édition des informations.
	 *
	 * @return void
	 */
	public static function edit(): void
	{
		if (!isset($_POST['tags']) || !isset($_POST['save']))
		{
			return;
		}

		$r = Tags::edit($_POST['tags']);
		if ($r < 0)
		{
			Report::error();
		}
		else if ($r < 1)
		{
			Report::info(__('Aucun tag n\'a été modifié.'));
		}
		else
		{
			Report::success($r > 1
				? sprintf(__('%s tags ont été modifiés.'), $r)
				: __('1 tag a été modifié.'));
		}
	}

	/**
	 * Récupération des tags.
	 *
	 * @return void
	 */
	public static function getTags(): void
	{
		// Fil d'Ariane.
		$breadcrumb =
		[
			[
				'current' => TRUE,
				'name' => __('Tags'),
				'url' => App::getURL($_GET['q_pageless'])
			]
		];
		Template::set('breadcrumb', $breadcrumb);
		Template::set('objects_count', 0);
		Template::set('tags', []);

		// Nombre de tags par page.
		$nb_per_page = Auth::$infos['user_prefs']['tags']['nb_per_page'];

		// Clause WHERE.
		self::_sqlWhere($sql_where, $params);

		// Nombre de tags.
		if (!DB::execute("SELECT COUNT(*) FROM {tags} WHERE $sql_where", $params))
		{
			return;
		}
		Template::set('objects_count', $nb_tags = DB::fetchVal());

		// Nombre de pages.
		Template::set('nb_pages', ceil($nb_tags / $nb_per_page));
		$sql_limit_start = $nb_per_page * ($_GET['page'] - 1);

		// Critère de tri.
		$sql_order_by = 'tag_' . Auth::$infos['user_prefs']['tags']['order_by_column'];
		$sql_order_by = substr($sql_order_by, -4) == 'name'
			? "LOWER($sql_order_by)"
			: $sql_order_by;
		$sql_order_by_order = Auth::$infos['user_prefs']['tags']['order_by_order'];

		// Récupération des tags.
		$sql = "SELECT t.*,
					   COUNT(ti.item_id) AS tag_count
				  FROM {tags} AS t
			 LEFT JOIN {tags_items} AS ti
					ON t.tag_id = ti.tag_id
				 WHERE $sql_where
			  GROUP BY t.tag_id
			  ORDER BY $sql_order_by $sql_order_by_order
			     LIMIT $sql_limit_start,$nb_per_page";
		if (!DB::execute($sql, $params))
		{
			return;
		}
		$tags = DB::fetchAll();
		if (!self::_objectsNoResult(count($tags)))
		{
			return;
		}

		// Formatage des données.
		$tags_formated = [];
		foreach ($tags as &$i)
		{
			$tags_formated[] =
			[
				'count' => $i['tag_count'],
				'id' => $i['tag_id'],
				'link' => $i['tag_count'] > 0
					? App::getURL('category/1/tag/' . $i['tag_id'])
					: NULL,
				'name' => $i['tag_name'],
				'url' => $i['tag_url']
			];
		}
		Template::set('tags', $tags_formated);
	}

	/**
	 * Informations de template pour le moteur de recherche.
	 *
	 * @param array $functions
	 *
	 * @return void
	 */
	public static function tplSearch(array &$functions): void
	{
		Template::set('search', ['columns' => $functions['columns'](
		[
			'tag_name' => TRUE,
			'tag_url' => TRUE
		])]);
	}



	/**
	 * Clause WHERE d'une requête SQL.
	 *
	 * @param mixed $sql
	 * @param mixed $params
	 *
	 * @return void
	 */
	private static function _sqlWhere(&$sql, &$params): void
	{
		$sql = '1=1';
		$params = [];
		switch ($_GET['filter'] ?? '')
		{
			case 'search' :
				if (!$search = self::$search->sql())
				{
					App::redirect($_GET['q_filterless']);
				}
				$sql = $search['sql'];
				$params = $search['params'];
				break;
		}
	}
}
?>