/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { createComponent } from "../Component.js";
import { InputField } from "./InputField";
/**
 * Select field
 *
 * @see Form
 */
export class SelectField extends InputField {
    constructor() {
        super(...arguments);
        this.baseCls = "goui-form-field select";
        /**
         * The field of the select options that is used as value
         */
        this.valueField = 'value';
        this.fireChangeOnBlur = false;
        /**
         * Renderer function. Defaults to returning a 'name' property.
         *
         * @param record
         */
        this.textRenderer = (record) => record.name;
    }
    createInput() {
        return document.createElement("select")
            .on('change', _ => this.fireChange());
    }
    // turned off fireChangeOnBlur but override onFocusIn() to get the oldValue
    onFocusIn(e) {
        this.captureValueForChange();
    }
    /**
     * Redraw the options. Can be useful when this.textRenderer() produces another result
     */
    drawOptions() {
        const v = this.value;
        this.options = this.options;
        this.value = v;
    }
    /**
     * Provide select input with options
     *
     * It should have at least have a field that corresponds with {@link Select.valueField}
     *
     * By default, it should have a "value" and "name" property. This can be changed with the {@link Select.valueField} and
     * {@link Select.textRenderer}.
     *
     * @param opts
     */
    set options(opts) {
        this._options = opts;
        this.input.empty();
        opts.forEach((o) => {
            const opt = new Option();
            if (o[this.valueField]) {
                opt.value = o[this.valueField];
            }
            opt.innerHTML = this.textRenderer(o);
            this.input.appendChild(opt);
        });
    }
    get options() {
        var _a;
        return (_a = this._options) !== null && _a !== void 0 ? _a : [];
    }
    /**
     * A store to provide the {@link Select.options}.
     * @param store
     */
    set store(store) {
        this._store = store;
        this.options = store.items;
        store.on("datachanged", () => this.options = store.items);
    }
    get store() {
        return this._store;
    }
    set value(v) {
        super.value = v;
    }
    get value() {
        const opts = (this.store ? this.store.items : this.options);
        let index = this.input.selectedIndex;
        let v;
        if (opts[index]) {
            v = opts[index][this.valueField];
        }
        else {
            v = undefined;
        }
        return v;
    }
}
/**
 * Shorthand function to create {@see SelectField}
 *
 * @param config
 */
export const select = (config) => createComponent(new SelectField(), config);
//# sourceMappingURL=SelectField.js.map