<?php

namespace GO\Projects2\Report;

use GO;
use GO\Base\Fs\File;
use GO\Base\Util\Date;
use GO\Base\Db\FindCriteria;
use GO\Base\Db\FindParams;
use go\core\util\DateTime;
use GO\Projects2\Model\Project;
use GO\Projects2\Model\Template;
use GO\Projects2\Report\AbstractReport;


final class CapitalReport extends AbstractReport
{
	/** @var DateTime */
	private $_nowUnix;

	private $_xlsObject;

	const SUSTAINABLE_BG_COLOR_TEAL = "00147669";
	const SUSTAINABLE_BG_COLOR_GREEN = "003DBA99";
	const SUSTAINABLE_BG_COLOR_YELLOW = "00FFC000";
	const SUSTAINABLE_BG_COLOR_ORANGE = "00E38225";
	const SUSTAINABLE_BG_COLOR_RED = "00F15A44";
	const SUSTAINABLE_BG_COLOR_DARKBLUE = "002F5696";

	public function name()
	{
		return "CA Case Management Log";
	}

	public function fileExtension()
	{
		return 'xls';
	}

	public function supportedProjectTypes()
	{
		return array(Template::PROJECT_TYPE_PROJECT, Template::PROJECT_TYPE_CONTAINER);
	}


	public function supportsStatusFilter()
	{
		return false;
	}

	public function supportsSelectedProject()
	{
		return true;
	}

	public function supportsBatchReport()
	{
		return true;
	}


	public function supportsDateRange()
	{
		return true;
	}


	public function render($return = false)
	{
		GO::setMaxExecutionTime(300);
		GO::setMemoryLimit(0);
		$tz = new \DateTimeZone('Europe/London');

		$this->_periodStartUnix = DateTime::createFromFormat('d/m/Y', $_REQUEST['startdate'], $tz);
		$this->_periodEndUnix = DateTime::createFromFormat('d/m/Y', $_REQUEST['enddate'], $tz);
		$this->_nowUnix = new DateTime();
		$this->_nowUnix->setTimezone($tz);
		$periodStartString = $this->_periodStartUnix->format("Y-m-d");
		$periodEndString = $this->_periodEndUnix->format("Y-m-d");

		$filePath = \GO::config()->tmpdir.$this->_nowUnix->format('Y-m-d').' '.$this->name().' '.$periodStartString.'_'.$periodEndString.'.'.$this->fileExtension();
		$file = new File($filePath);
		if($file->exists()) {
			$file->delete();
			$file = new File($filePath);
		}

		$this->_xlsObject = new \GO\Base\Fs\XlsFile($filePath);
		$currSheet = $this->_xlsObject->getCurrentSheet();

		$this->addHeaders($currSheet);

		$this->_xlsObject->writeToFile();

		// TODO: Retrieve data, add as cells

		if ($return) {
			return $file->getContents();
		} else {
			$file->touch();
			$file->putContents($file->getContents());
			\GO\Base\Util\Http::outputDownloadHeaders($file);
			echo $file->getContents();
		}
	}


	/**
	 * Create pretty headers
	 *
	 * @return void
	 */
	private function addHeaders(&$currSheet)
	{
		$this->_xlsObject->putRecord( array("CA Case Management Log " . $this->_nowUnix->format('d/m/Y')));

		// First row - Teal BG
		$currSheet->mergeCells('A1:H1');
		$this->fillRange($currSheet, 'A1:H1', self::SUSTAINABLE_BG_COLOR_TEAL);
		$currSheet->getStyle('A1:H1')->getFont()->setSize(22)->getColor()->setARGB(\PHPExcel_Style_Color::COLOR_WHITE);
		$currSheet->getRowDimension('1')->setRowHeight(32);

		// Second row - Light green BG
		$this->_xlsObject->putRecord( array("Project Management - Tech & Project"));
		$currSheet->mergeCells('A2:H2');
		$this->fillRange($currSheet, 'A2:H2', self::SUSTAINABLE_BG_COLOR_GREEN);
		$currSheet->getStyle('A2:H2')->getFont()->setSize(22)->getColor()->setARGB(\PHPExcel_Style_Color::COLOR_WHITE);
		$currSheet->getRowDimension('2')->setRowHeight(28);

		// Third row - Several grouped cells, background colors
		$headerRecord = array('','','','','','','','','','','SURVEY and CLAIM PROCESSING - Survey Template Sub Projects', '',
			'','','','','', 'Claim Summary', '','TAX REFUND', '', 'FINAL REPORTS', '', '','');
		$currSheet->mergeCells('A3:J3');
		$currSheet->mergeCells('K3:Q3');
		$currSheet->mergeCells('R3:S3');
		$currSheet->mergeCells('T3:U3');
		$currSheet->mergeCells('V3:Y3');
		$currSheet->getStyle('A3:Y3')->getFont()->setSize(16)->getColor()->setARGB(\PHPExcel_Style_Color::COLOR_WHITE);
		$this->fillRange($currSheet, 'K3:S3', self::SUSTAINABLE_BG_COLOR_DARKBLUE);
		$this->fillRange($currSheet, 'T3:U3', self::SUSTAINABLE_BG_COLOR_ORANGE);
		$this->fillRange($currSheet, 'V3:Y3', self::SUSTAINABLE_BG_COLOR_RED);
		$currSheet->getRowDimension('3')->setRowHeight(49);

		$this->_xlsObject->putRecord($headerRecord);

		// Fourth row - Several background colors
		$headerRecord = ['Column 1','Top Level Project Name',"Portfolio","Source","Tax Deadline","Forecast Claim",
			"Forecast Fee", "Column 2", "Notes", "Team TASK", "Survey Date", "Survey Received",
			"Survey Checked by Project", "Survey Checked by Tech", "Date Survey Report Sent",
			"Date Signed Survey Report Received", "Deadline - 2", "Final Claim Summary Completed", "Claim Total",
			"Deadline - 2", "Tax Return amended or figured passed over - Date Submitted", "Refunds Received / Final Reports Stage",
			"Deadline - 3", "Tech Completed"," Project Completed"];

		$this->_xlsObject->putRecord($headerRecord);
		$currSheet->getStyle('A4:Y4')->getFont()->setSize(11)->getColor()->setARGB(\PHPExcel_Style_Color::COLOR_WHITE);
		$this->fillRange($currSheet, 'A4:H4', self::SUSTAINABLE_BG_COLOR_TEAL);
		$this->fillRange($currSheet, 'I4:J4', self::SUSTAINABLE_BG_COLOR_YELLOW);
		$this->fillRange($currSheet, 'K4:S4', self::SUSTAINABLE_BG_COLOR_DARKBLUE);
		$this->fillRange($currSheet, 'T4:U4', self::SUSTAINABLE_BG_COLOR_ORANGE);
		$this->fillRange($currSheet, 'V4:Y4', self::SUSTAINABLE_BG_COLOR_RED);
		$currSheet->getRowDimension('4')->setRowHeight(20);

	}

	/**
	 * Get a cell range, give pretty background color
	 *
	 * @param $currSheet
	 * @param string $range
	 * @param string $color
	 * @return void
	 */
	public function fillRange(&$currSheet, string $range, string $color)
	{
		$currSheet->getStyle($range)->getFill()->setFillType(\PHPExcel_Style_Fill::FILL_SOLID)
			->getStartColor()->setARGB($color);

	}

	/**
	 * Find child project by template ID
	 *
	 * @param Project $parentProjectModel
	 * @param int $templateId
	 * @return null
	 * @throws \go\core\http\Exception
	 */
	private function findChildProjectByTemplateId(Project $parentProjectModel, int $templateId)
	{
		$childProjectsStmt = Project::model()->find(
			FindParams::newInstance()
				->criteria(FindCriteria::newInstance()
					->addCondition('parent_project_id',$parentProjectModel->id)
					->addCondition('template_id', $templateId)
				)
			->single(true)
		);

		foreach ($childProjectsStmt as $childProjectModel) {
			return $childProjectModel;
		}

		return null;

	}
}