<?php

namespace go\modules\business\mollie;

use DateTime;
use Exception;
use go\core;
use go\core\exception\Forbidden;
use go\core\orm\exception\SaveException;
use go\modules\business\finance\model\FinanceDocument;
use go\modules\business\finance\model\Payment;
use go\modules\business\finance\model\PaymentProviderInterface;
use Mollie\Api\Exceptions\ApiException;
use Mollie\Api\MollieApiClient;


/**
 * @copyright (c) 2020, Intermesh BV https://www.intermesh.nl
 * @author Merijn Schering <mschering@intermesh.nl>
 * @license http://www.gnu.org/licenses/agpl-3.0.html AGPLv3
 */
class Module extends core\Module implements PaymentProviderInterface
{
	use core\event\EventEmitterTrait;

	public function getAuthor() : string
	{
		return "Intermesh BV <info@intermesh.nl>";
	}

    /**
     * The development status of this module
     * @return string
     */
    public function getStatus() : string{
        return self::STATUS_STABLE;
    }

	public function getDependencies() : array
	{
		return ['business/finance'];
	}

	public function requiredLicense(): ?string
	{
		return "billing";
	}


	public function getSettings()
	{
	 return model\Settings::get();
	}

	private $mollie;

	/**
	 * @throws ApiException
	 */
	private function getMollie(): MollieApiClient
	{
		//autoload composer libs for this module
		require_once __DIR__ . '/vendor/autoload.php';

		if(!isset($this->mollie)) {
			$this->mollie = new MollieApiClient();
			$this->mollie->setApiKey(Module::get()->getSettings()->apiKey);
		}

		return $this->mollie;
	}

	/**
	 * @throws Forbidden
	 */
	private function getDocument(int $documentId, string $token) : FinanceDocument {
		$document = FinanceDocument::findById($documentId);
		if ($document->token != $token) {
			throw new Forbidden();
		} else{
			go()->setAuthState(new core\auth\TemporaryState($document->createdBy));
		}

		return $document;
	}

	/**
	 * @param int $documentId
	 * @param string $token
	 * @throws ApiException
	 * @throws Forbidden
	 */
	public function pagePayment(int $documentId, string $token) {

		$document = $this->getDocument($documentId, $token);

		$mollie = $this->getMollie();

		$redirectUrl = go()->getAuthState()->getPageUrl() .	"/business/mollie/return/". $document->id . "/" . $document->token;

		if(!empty($_GET['returnUrl'])) {
			$redirectUrl .= '?returnUrl=' . rawurlencode($_GET['returnUrl']);
		}

		try {

			if($document->getTotalPrice() > Module::get()->getSettings()->maxAmount) {
				throw new Exception("Sorry, the amount is too high. Please pay by bank transfer.");
			}

			$data = [
				"amount" => [
					"currency" => "EUR",
					"value" => number_format($document->getTotalPrice(), 2, ".", "")
				],
				"description" => $document->number,
				"redirectUrl" => $redirectUrl,
				"webhookUrl"  => go()->getAuthState()->getPageUrl() . "/business/mollie/webhook/". $document->id . "/" . $document->token,
				"metadata" => [
					//Unused but might be useful in the future
					"documentId" => $document->id,
				],
			];

			$payment = $mollie->payments->create($data);

			//Make sure this page is not cached by the browser
			header("Cache-Control: no-store, no-cache, must-revalidate, max-age=0");
			header("Cache-Control: post-check=0, pre-check=0", false);
			header("Pragma: no-cache");
			header("Location: " . $payment->getCheckoutUrl(), true, 303);
			exit();

		} catch (Exception $e) {
			$error = htmlspecialchars($e->getMessage());

			$returnUrl = $_GET['returnUrl'] ?? $document->getCustomerUrl();
			require(__DIR__ . '/views/web/error.php');
		}
	}

	/**
	 * @throws Forbidden
	 */
	public function pageReturn(int $documentId, string $token) {

		$document = $this->getDocument($documentId, $token);
		$returnUrl = $_GET['returnUrl'] ?? $document->getCustomerUrl();

		require(__DIR__ . '/views/web/return.php');
	}

	/**
	 * @param int $documentId
	 * @param string $token
	 * @throws Forbidden
	 * @throws SaveException
	 * @throws Exception
	 */
	public function pageWebhook(int $documentId, string $token) {

		/*
		 * How to verify Mollie API Payments in a webhook.
		 *
		 * See: https://docs.mollie.com/guides/webhooks
		 */

		try {

			$document = $this->getDocument($documentId, $token);

			if(!$document->isPaid()) {
				$mollie = $this->getMollie();

				/*
				 * Retrieve the payment's current state.
				 */
				$payment = $mollie->payments->get($_POST["id"]);

				$document->paymentStatus = $payment->status;
				if (!$document->save()) {
					throw new SaveException($document);
				}

				if ($payment->isPaid() && !$payment->hasRefunds() && !$payment->hasChargebacks()) {

					$p = (new Payment())->setValues([
						'businessId' => $document->findBook()->businessId,
						'documentId' => $document->id,
						'customerId' => $document->getCustomerId(),
						'reference' => 'mollie-' . $payment->id,
						'date' => new DateTime(),
						'amount' => $payment->amount->value,
						'description' => "Mollie transaction: " . $payment->description
					]);

					if (!$p->save()) {
						throw new SaveException($p);
					}

				}
			}
		} catch (ApiException $e) {
			echo "API call failed: " . htmlspecialchars($e->getMessage());
		}
	}

	public function getPaymentProviderTitle(): string
	{
		return go()->t("Pay online via Mollie");
	}
}
