<?php /** @noinspection PhpUndefinedFieldInspection */

namespace go\modules\business\finance\install;

use DateTimeZone;
use Exception;
use GO\Base\Db\FindCriteria;
use GO\Base\Db\FindParams;
use GO\Billing\Model\Book;
use GO\Billing\Model\Item;
use GO\Billing\Model\Order;
use GO\Billing\Model\Product;
use go\core\ErrorHandler;
use go\core\exception\NotFound;
use go\core\jmap\Entity;
use go\core\model\Alert;
use go\core\model\Link;
use go\core\orm\EntityType;
use go\core\orm\exception\SaveException;
use go\core\util\DateTime;
use go\modules\business\business\model\Business;
use go\modules\business\business\model\VatRate;
use go\modules\business\catalog\model\Article;
use go\modules\business\contracts\model\Contract;
use go\modules\business\contracts\model\ContractBook;
use go\modules\business\contracts\model\ContractItem;
use go\modules\business\finance\model\FinanceBook;
use go\modules\business\finance\model\FinanceDocument;
use go\modules\business\finance\model\FinanceDocumentItem;
use go\modules\business\finance\model\FinanceDocumentItemGroup;
use go\modules\business\finance\model\FinanceDocumentPayment;
use go\modules\business\finance\model\Payment;
use go\modules\community\addressbook\model\Address;
use go\modules\community\addressbook\model\Contact;
use go\modules\community\addressbook\model\EmailAddress;
use go\modules\community\comments\model\Comment;
use go\modules\community\history\Module;

class MigrateBilling {

	/**
	 * @var FinanceBook
	 *
	 */
	private $financeBook;

	/**
	 * @var \go\core\orm\Entity|ContractBook|null
	 */
	private $contractBook;

	/**
	 * @var \go\core\orm\Entity|Business|null
	 */
	private $business;

	private $cutOffDate = null;
	/**
	 * @var Alert
	 */
	private $alert;

	public function __construct(int $financeBookId, int $contractBookId)
	{
		$this->financeBook = FinanceBook::findById($financeBookId);
		$this->contractBook = ContractBook::findById($contractBookId);
		$this->business = Business::findById($this->financeBook->businessId);

		echo "Working for book " . $this->financeBook->name ."\n";
	}


	public function reset(bool $migrateCatalog = false) {
		if($migrateCatalog) {
			echo "Resetting catalog\n";
			Article::delete(['businessId' => $this->financeBook->businessId]);
		}
		echo "Resetting documents\n";
		FinanceDocument::delete(['bookId' => $this->financeBook->id]);

		echo "Resetting contracts\n";
		Contract::delete([ 'bookId' => $this->contractBook->id]);

		echo "Resetting payments\n";
		Payment::delete([ 'businessId' => $this->financeBook->businessId]);
		echo "Reset done\n\n";
	}


	/**
	 * ./cli.php business/finance/Migrate/importBook --sourceBookId=2 --businessId=1
	 * docker-compose exec groupoffice-finance ./www/cli.php business/finance/Migrate/importBook --sourceBookId=2 --businessId=1
	 *
	 *
	 * @param int $sourceBookId The ID of the billing book to import
	 * @throws NotFound
	 * @throws SaveException
	 * @throws Exception
	 */
	public function run(int $sourceBookId) {

		//for quicker testing
		//$this->cutOffDate = new DateTime("2018-01-01");

		try {

			$this->alert = new Alert();
			$this->alert->setEntity($this->financeBook);
			$this->alert->userId = go()->getUserId();
			$this->alert->triggerAt = new DateTime();
			$this->alert->setData([
					'title' => go()->t("Billing migration", "business", "finance"),
					'body' => go()->t("Your billing migration process has started in the background.", "business", "finance"),
					'progress' => 0
				]
			);
			$this->alert->tag = "migratebilling";

			if (!$this->alert->save()) {
				throw new SaveException($this->alert);
			}

			// Speed things up.
			Entity::$trackChanges = false;
			Module::$enabled = false;


			$this->tz = new DateTimeZone(go()->getSettings()->defaultTimezone);

//		return $this->testDoc($sourceBookId);


			$this->migrateDocs($sourceBookId);

			if ($this->financeBook->type == FinanceBook::TYPE_SALES_INVOICE) {
				$this->migrateContracts($sourceBookId);
			}


			// Speed things up.
			Entity::$trackChanges = true;
			Module::$enabled = true;

			$this->alert->setData([
					'title' => go()->t("Billing migration", "business", "finance"),
					'body' => go()->t("Your billing items have been migrated to the finance module.", "business", "finance"),
					'progress' => 100
				]
			);
			if (!$this->alert->save()) {
				throw new SaveException($this->alert);
			}

			EntityType::push();
		}
		catch(\Throwable $e) {
			$error = ErrorHandler::logException($e);

			echo $error ."\n";

			$this->alert->setData([
					'title' => go()->t("Billing migration", "business", "finance"),
					'body' => go()->t("The migration failed: ", "business", "finance"). $e->getMessage(),
					'progress' => 100
				]
			);
			if (!$this->alert->save()) {
				throw new SaveException($this->alert);
			}
		}

	}

	private function progress(int $progress) {

		$this->alert->setData([
			'progress' => $progress
			]
		);

		Entity::$trackChanges = true;
		if(!$this->alert->save()) {
			throw new SaveException($this->alert);
		}
		EntityType::push();
		Entity::$trackChanges = false;
	}


	/**
	 * @throws Exception
	 * @throws SaveException
	 */
	public function migrateCatalog() {


		$products = Product::model()->find();

		foreach($products as $product) {
			$this->createArticle($product);
		}
	}

	/**
	 * @throws SaveException
	 * @throws Exception
	 */
	private function createArticle(Product $product) {
		if(Article::exists($product->id)) {
			//already exists
			return;
		}

		$article = new Article();
		$article->id = $product->id;



		// todo:

//		* @property int $files_folder_id
//		* @property string $unit_stock
//		* @property string $unit
//		* @property int $stock_min
//		* @property string $required_products
//		* @property int $stock
//		* @property boolean $charge_shipping_costs
//		* @property string $special_total_price
//		* @property string $special_list_price
//		* @property boolean $special
//		* @property boolean $allow_bonus_points
//		* @property string $image
//		* @property int $category_id
//		* @property int $sort_order
//		* @property string $cost_code
//
		$article->number = $product->article_id;

		if($product->supplier_company_id) {
			$contact = Contact::findById($product->supplier_company_id);
			if($contact)
				$article->supplierOrganizationId = $contact->id;
		}
		$article->supplierNumber = (string) $product->supplier_product_id;

		$article->businessId = $this->financeBook->businessId;
		$article->vatRateId = $this->checkVatRate($product->vat)->id;
		$defaultLang =  $product->getLanguage();
		$article->name = $defaultLang->name;
		$article->description = $defaultLang->description;

		$article->price = $product->list_price;
		$article->cost = $product->cost_price;

		$article->cutPropertiesToColumnLength();

		if(!$article->save()) {
			throw new SaveException($article);
		}
	}


	/**
	 * @throws SaveException
	 * @throws Exception
	 * @throws NotFound
	 */
	private function testDoc($sourceBookId) {


		$findParams = new FindParams();
		$findParams->order('order_id', 'ASC');
		$findParams->getCriteria()->addCondition('book_id', $sourceBookId)
			->addRawCondition('order_id = "I2014-004068"');

		$orders = Order::model()->find($findParams);

		foreach($orders as $order) {

			echo $order->order_id;
//			try {
			$this->importOrder($order);
		}
	}

	private $tz;


	/**
	 * Some order mismatched the totals with the items
	 *
	 * @param $sourceBookId
	 * @return void
	 * @throws \GO\Base\Exception\AccessDenied
	 */
	public static function fixBrokenOrders($sourceBookId) {
//		select o.id, o.order_id, o.book_id, ROUND(o.subtotal, 0) as subtotal, ROUND(coalesce(sum(amount * unit_price), 0), 0) as itemsubtotal,customer_name from bs_orders o
//    left join bs_items i on i.order_id = o.id
//
//group by o.id, o.subtotal
//    having ROUND(o.subtotal, 0) != ROUND(coalesce(sum(amount * unit_price), 0), 0);

		$stmt = go()->getDbConnection()
			->select('o.id, o.subtotal, o.total, ROUND(coalesce(sum(amount * unit_price), 0), 0) as itemsubtotal, ROUND(coalesce(sum(amount * unit_total), 0), 0) as itemtotal')
			->from('bs_orders', 'o')
			->join('bs_items', 'i', 'i.order_id = o.id', 'left')
			->where('o.book_id', '=', $sourceBookId)
			->groupBy(['o.id', 'o.subtotal'])
			->having('ROUND(o.subtotal, 0) != ROUND(coalesce(sum(amount * unit_price), 0), 0)');

//		echo $stmt ."\n\n";

		$brokenOrders = $stmt->all();

		if(empty($brokenOrders)) {
			return;
		}

		foreach($brokenOrders as $order) {

			if($order['itemsubtotal'] < $order['subtotal']) {

				$item = new Item();
				$item->amount = 1;
				$item->order_id = $order['id'];
				$item->unit_price = $order['subtotal'] - $order['itemsubtotal'];
				$item->unit_total = $order['total'] - $order['itemtotal'];
				$item->vat = $item->unit_total - $item->unit_price;
				$item->description = 'Corrected broken order on migration';
				if(!$item->save()) {
					var_dump($item->getValidationErrors());
				}
			} else{
				$o = Order::model()->findByPk($order['id']);
				$o->syncItems();
			}

			echo "Fixed " . $order['id'] ."\n";

		}
	}

	/**
	 * @throws Exception
	 * @throws SaveException
	 * @throws NotFound
	 */
	private function migrateDocs($sourceBookId) {

		$book = Book::model()->findByPk($sourceBookId);
		$this->financeBook->nextNumber = $book->next_id + 1;
		$this->financeBook->save();

		self::fixBrokenOrders($sourceBookId);

		$findParams = new FindParams();
		$findParams->order('order_id', 'ASC');
		$findParams->getCriteria()->addCondition('book_id', $sourceBookId)
			->addRawCondition('order_id != ""');

		if(isset($this->cutOffDate)) {
			$findParams->getCriteria()->addCondition('btime', $this->cutOffDate->format("U"), '>');
		}

		$orders = Order::model()->find($findParams);
		$total = $orders->rowCount();
		$i = 0;
		foreach($orders as $order) {

			echo $order->order_id;
//			try {
				$this->importOrder($order);

				$this->progress(floor((++$i / $total) * 100));
				echo ".. done\n";
//			}
//			catch(\Exception $e) {
//				echo "FAILED: " .  $e->getMessage() . "\n";
//
//				echo $e->getTraceAsString();
//
//				echo "\n\n-----\n\n";
//			}
		}


	}

	/**
	 * @throws SaveException
	 * @throws Exception
	 */
	private function createCustomer(Order $order): int
	{
		$addressBookId = go()->getAuthState()->getUser(['addressBookSettings'])->addressBookSettings->getDefaultAddressBookId();

		$org = Contact::find(['id'])->where(['addressBookId' => $addressBookId, 'name' => $order->customer_name])->single();

		if($org) {
			return $org->id;
		}

		$org = new Contact();
		$org->addressBookId = $addressBookId;
		$org->isOrganization = true;
		$org->name = $order->customer_name;

		if(!empty($order->customer_email)) {
			$org->setValue('emailAddresses', [[
				'type' => EmailAddress::TYPE_BILLING,
				'email' => $order->customer_email
			]]);
		}


		if(!empty($order->customer_address) || !empty($order->customer_city)) {
			$org->setValue('addresses', [[
					'address' => $order->customer_address . ' ' . $order->customer_address_no,
					'zipCode' => $order->customer_zip,
					'city' => $order->customer_city,
					'state' => $order->customer_state,
					'country' => $order->customer_country,
					'type' => Address::TYPE_POSTAL
				]]
			);
		}

		$org->vatNo = $order->customer_vat_no;

		if(!$org->save()) {
			throw new SaveException($org);
		}

		return $org->id;

	}

	/**
	 * @throws SaveException
	 * @throws NotFound
	 * @throws Exception
	 * @throws Exception
	 * @throws Exception
	 * @throws Exception
	 * @throws Exception
	 * @throws Exception
	 * @throws Exception
	 */
	private function importOrder(Order $order) {

//		$doc = FinanceDocument::find(['id'])->where('number', '=', $order->order_id)->single();
//		if($doc) {
//			//exists
//			return;
//		}
		$doc = new FinanceDocument();
		$doc->bookId = $this->financeBook->id;

		switch($this->financeBook->type) {
			case FinanceBook::TYPE_SALES_INVOICE:
				$doc->completedAt = $order->ptime > 0 ? new DateTime('@' . $order->ptime) : null;
				break;

			case FinanceBook::TYPE_SALES_ORDER:
				//$doc->completedAt = $order->ptime > 0 ? new DateTime('@' . $order->ptime) : null;\
				//todo
				break;
		}

		$doc->createdAt = new DateTime('@' . $order->ctime);
		$doc->modifiedAt = new DateTime('@' . $order->mtime);
		$doc->createdBy = \go\core\model\User::exists($order->user_id) ? $order->user_id : 1;
		$doc->modifiedBy = \go\core\model\User::exists($order->muser_id) ? $order->muser_id : 1;
		$doc->number = $order->order_id;

		$doc->reference = $order->reference;

		if(!empty($order->contact_id)){
			try {
				$doc->setContactId($order->contact_id);
			}catch(NotFound $e) {
				//ignore
			}
		}

		if(!empty($order->company_id)) {
			try {
				$doc->setCustomerId($order->company_id);
			}catch(NotFound $e) {
				//ignore
			}
		}

		if($doc->getCustomerId() == null) {
			$doc->setCustomerId($this->createCustomer($order));
		}

		$doc->date = new DateTime("@" . $order->btime);
		$doc->date->setTimezone( $this->tz);
		$doc->greeting = $order->frontpage_text;

		$map = [];
		foreach($order->itemGroups as $itemGroup) {
			$group = new FinanceDocumentItemGroup($doc);
			$group->title = $itemGroup->name;
			$group->showTotals = $itemGroup->summarize;

			//todo
			//$group->showContents = $itemGroup->show_individual_prices;

			//$doc->groups[] = $group;
			$map[$itemGroup->id] = $group;
		}

		$defaultGroup = new FinanceDocumentItemGroup($doc);

		foreach($order->items as $item) {
			$docItem = new FinanceDocumentItem($defaultGroup);
			if(Article::exists($item->product_id)) {
				$docItem->articleId = $item->product_id;
			}
			$docItem->description = $item->description ? $item->description : "-";
			$docItem->unitCost = $item->unit_cost;
			$docItem->unitPrice = $item->unit_price;
			$docItem->vatRate = $item->vat;
			$docItem->vatRateId = $this->checkVatRate($item->vat)->id;
			$docItem->unit = $item->unit;
			$docItem->quantity = $item->amount;

			if(!isset($map[$item->item_group_id])){
				$defaultGroup->items[] = $docItem;
			} else {
				$map[$item->item_group_id]->items[] = $docItem;
			}
		}

		if(!empty($defaultGroup->items)) {
			array_unshift($doc->itemGroups, $defaultGroup);
		}

		foreach($map as $group) {
			$doc->itemGroups[] = $group;
		}

		$doc->cutPropertiesToColumnLength();

		if(!$doc->save()) {
			throw new SaveException($doc);
		}

		foreach($order->payments as $payment) {
			$p = new Payment();
			$p->description = $payment->description;
			$p->date = new DateTime('@' . $payment->date);
			$p->amount = $payment->amount;
			$p->documentId = $doc->id;
			$p->customerId = $doc->getCustomerId();
			$p->businessId = $this->financeBook->businessId;
			$p->checked = true;
			if(!$p->save()) {
				throw new SaveException($p);
			}
		}

		Comment::copyTo($order, $doc);

		Link::copyTo($order, $doc);

// TODO Test when files are present
//		$sourceFolder = $order->getFilesFolder(false);
//		if($sourceFolder) {
//			$destFolder = Folder::model()->findForEntity($doc, true);
//			$destFolder->copyContentsFrom($sourceFolder);
//		}

	}


	/**
	 * @throws SaveException
	 * @throws Exception
	 */
	private function checkVatRate($rate): VatRate
	{
		foreach($this->business->vatRates as $vatRate) {
			if($vatRate->rate == $rate) {
				return $vatRate;
			}
		}

		$vatRate = new VatRate($this->business);
		$this->business->vatRates[] = $vatRate->setValues(['rate' => $rate, 'name' => $rate . '%']);

		if(!$this->business->save()) {
			throw new SaveException($this->business);
		}

		return $vatRate;
	}

	/**
	 * @throws Exception
	 * @throws SaveException
	 * @throws NotFound
	 */
	private function migrateContracts($sourceBookId) {

		//fix broken recurrence records
		go()->getDbConnection()->exec("update bs_orders set recurred_order_id = 0 where recurred_order_id not in (select id from bs_orders);");

		//find where recurtype is not empty and create contract.
		//recurse back in time to link older orders using recurring_order_id
		$fp = FindParams::newInstance();
		$fp->getCriteria()->addCondition('book_id', $sourceBookId)
			->mergeWith(
				(new FindCriteria())
					->addCondition('recur_type', '', '!=')
					->addCondition('recurred_order_id', '0', '>','t', false)
			)
			->addRawCondition('order_id != ""');

//		$fp->getCriteria()->addRawCondition('customer_name = "Cormed Ltd."');

		if(isset($this->cutOffDate)) {
			$fp->getCriteria()->addCondition('btime', $this->cutOffDate->format("U"), '>');
		}

		$fp->order('btime', 'ASC');

		$orders = Order::model()->find($fp);
		foreach($orders as $order) {
			$this->createContract($order);
		}
	}

	private function findFirst(Order $order): Order
	{
		while($previous = Order::model()->findSingleByAttribute('recurred_order_id', $order->id)) {
			$order = $previous;
		}

		return $order;
	}

	private function findLast(Order $order): Order
	{

		while(!empty($order->recurred_order_id) && ($last = Order::model()->findSingleByAttribute('id', $order->recurred_order_id))) {
			$order = $last;
		}

		return $order;
	}

	private function getRecurringMonths(Order $last) {
		if($last->isRecurring()) {
			$previousTime = $last->btime;
			$nextTime = $last->getNextRecurrenceTime();
		} else{
			$previous = Order::model()->findSingleByAttribute('recurred_order_id', $last->id);
			if(!$previous) {
				return 0;
			}

			$previousTime = $previous->btime;
			$nextTime = $last->btime;
		}

		$previousYear = date('Y', $previousTime);
		$previousMonth = date('m', $previousTime);

		$thisYear = date('Y', $nextTime);
		$thisMonth = date('m', $nextTime);

		$yearDiff = $thisYear - $previousYear;
		$thisMonth += $yearDiff * 12;

		return $thisMonth - $previousMonth;
	}

	/**
	 * @param Order $order
	 * @return void
	 * @throws NotFound
	 * @throws SaveException
	 * @throws Exception
	 * @throws Exception
	 * @throws Exception
	 * @throws Exception
	 * @throws Exception
	 * @throws Exception
	 */
	private function createContract(Order $order) : void
	{

		$first = $this->findFirst($order);
		$last = $this->findLast($order);

		if($last->id == $first->id) {
			echo "WARNING: Check invoice " . $first->order_id ."\n\n";
			return;
		}

		$number = 'C' . $first->order_id;

		echo $number .' - '.$order->order_id ."\n";

		/**
		 * @var FinanceDocument $migratedInvoice
		 */
		$migratedInvoice = FinanceDocument::find()->where(['bookId' => $this->financeBook->id, 'number' => $order->order_id])->single();

		if(!$migratedInvoice) {
			throw new Exception("Could not find invoice " . $order->order_id);
		}

		$contract = Contract::find()->where('number', '=', $number)->single();
		if(!$contract) {
			$contract = new Contract();
			$contract->number = $number;

			$contract->startsAt = new DateTime('@' . $first->btime);
			$contract->startsAt->setTimezone( $this->tz);
			$contract->bookId = $this->contractBook->id;

			try {
				if($order->company_id)
					$contract->setCustomerId($order->company_id);
			}catch(NotFound $e) {
				//ignore
			}

			if($contract->getCustomerId() == null) {
				$contract->setCustomerId($this->createCustomer($order));
			}

			if(!$contract->contactId && !empty($order->contact_id)) {

				$contact = Contact::findById($order->contact_id, ['id']);
				if($contact) {
					$contract->contactId = $contact->id;
				}
			}
		}

		$nextAt = new DateTime('@' . $last->btime);
		$nextAt->setTimezone($this->tz);

		$contract->migrateNextAt($nextAt);
		if($last->isRecurring()) {
			//contract is still running
			$contract->endsAt = null;
		} else{
			$contract->endsAt = new DateTime('@' . $last->btime);
			$contract->endsAt->setTimezone( $this->tz);
		}

		$contract->items = [];


		$contract->intervalMonths = $this->getRecurringMonths($last);

		if(!$contract->intervalMonths) {
			$contract->description = "WARNING: Check recurrence";
		}

		foreach($migratedInvoice->itemGroups as $ig) {
			foreach($ig->items as $item) {
				$itemValue = $item->toArray([
					"quantity",
					"description",
					"unitPrice",
					"vatRateId",
					"articleId"
				]);

				if(empty($contract->description)) {
					$contract->description = $item->description;
				}

				$contract->items[] = (new ContractItem($contract))->setValues($itemValue);
			}
		}

		$contract->cutPropertiesToColumnLength();

		if(empty($contract->description )) {
			$contract->description = "No description";
		}

		if(!$contract->save()){
			throw new SaveException($contract);
		}


		Link::create($contract, $migratedInvoice);

	}

}