import { ArrayUtil } from "../util/ArrayUtil.js";
import { Collection } from "../util/Collection.js";
import { createComponent } from "../component/Component.js";
/**
 * Generic data store used by components
 *
 * @category Data
 */
export class Store extends Collection {
    constructor() {
        // private static stores: Record<string, Store> = {};
        //
        // /**
        //  * If the ID is given the store can be fetched with
        //  */
        // public id?: string;
        super(...arguments);
        this._loading = false;
        /**
         * Sort the data on field and direction
         */
        this.sort = [];
    }
    /**
     * True when the store is loading
     */
    get loading() {
        return this._loading;
    }
    /**
     * Load a set of records
     *
     * @param records
     * @param append
     */
    loadData(records, append = true) {
        append ? this.add(...records) : this.replace(...records);
        this.fire("load", this, records, append);
    }
    set data(records) {
        this.loadData(records);
    }
    get data() {
        return this.getArray();
    }
    /**
     * Reload the data from the source
     */
    async reload() {
        return this.load();
    }
    /**
     * Returns the loaded records. If append is true it only returns the new records.
     * Override this function for new store types.
     *
     * @param append
     * @protected
     */
    internalLoad(append) {
        this.loadData(ArrayUtil.multiSort(this.items, this.sort), append);
        return Promise.resolve(this.items);
    }
    /**
     * Loads the data from source into the store
     *
     * @param append
     */
    load(append = false) {
        this._loading = true;
        this.fire("beforeload", this, append);
        return this.internalLoad(append)
            .catch(reason => {
            console.error(reason);
            this.fire("loadexception", this, reason);
            throw reason;
        })
            .finally(() => {
            this._loading = false;
        });
    }
    /**
     * Load the next set of records when paging.
     * Doesn't do anything in the array store but can be implemented in async stores.
     */
    loadNext(append = false) {
        return Promise.resolve([]);
    }
    /**
     * Load the next set of records when paging.
     * Doesn't do anything in the array store but can be implemented in async stores.
     */
    loadPrevious() {
        return Promise.resolve([]);
    }
    hasNext() {
        return false;
    }
    hasPrevious() {
        return false;
    }
}
/**
 * Shorthand function to create a {@see Store}
 *
 * @param config
 */
export const store = (config) => createComponent(new Store(), config);
//# sourceMappingURL=Store.js.map