/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { Config, Listener, Observable, ObservableEventMap, ObservableListenerOpts } from "../Observable.js";
import { List } from "../List.js";
export interface RowSelectEventMap<Type extends Observable> extends ObservableEventMap<Type> {
    /**
     * Fires when selection changes. When holding arrow on keyboard it will only fire once at key up to prevent
     * flooding the server with requests
     *
     * @example get store record in list config
     * ```
     * rowSelection: {
     * 				multiSelect: true,
     * 				listeners: {
     * 					selectionchange: (rowSelect) => {
     * 						if(rowSelect.getSelected().length == 1) {
     * 							const table = rowSelect.getTable();
     * 							const record = table.getStore().getRecordAt(rowSelect.getSelected()[0]);
     * 						}
     * 					}
     * 				}
     * 			}
     * ```
     * @param rowSelect
     */
    selectionchange: (rowSelect: Type) => void;
    /**
     * Fires when a row is selected
     * @param rowSelect
     * @param storeIndex
     */
    rowselect: (rowSelect: Type, storeIndex: number) => void;
    /**
     * Fires when a row is deselected
     *
     * @param rowSelect
     * @param storeIndex
     */
    rowdeselect: (rowSelect: Type, storeIndex: number) => void;
}
export interface RowSelect {
    on<K extends keyof RowSelectEventMap<this>, L extends Listener>(eventName: K, listener: Partial<RowSelectEventMap<this>>[K], options?: ObservableListenerOpts): L;
    fire<K extends keyof RowSelectEventMap<this>>(eventName: K, ...args: Parameters<RowSelectEventMap<any>[K]>): boolean;
}
/**
 * Row selection model
 */
export declare class RowSelect extends Observable {
    readonly list: List;
    private _selected;
    /**
     * Last selected index used for multi selection with shift
     * @private
     */
    private lastIndex;
    private shiftStartIndex?;
    multiSelect: boolean;
    private hasKeyUpListener;
    private fireSelectionChange;
    constructor(list: List);
    clear(): void;
    selectAll(): void;
    /**
     * Get selected indexes
     *
     * Note that this is a copy and can't be edited directly. You have to set the selected property with a changed array.
     */
    get selected(): number[];
    set selected(newSelection: number[]);
    private onRowMouseDown;
    add(index: number): void;
    /**
     * Remove an index from the selection
     *
     * @param index
     * @param silent Don't fire events
     */
    remove(index: number, silent?: boolean): void;
    /**
     * Set selected indexes
     *
     * @param newSelection
     * @param silent Suspends 'selectionchange' event
     */
    private setSelected;
    private onKeyDown;
}
export type RowSelectConfig = Config<RowSelect, RowSelectEventMap<RowSelect>, "list">;
/**
 * Shorthand function to create {@see RowSelect}
 *
 * @param config
 */
export declare const rowselect: (config: RowSelectConfig) => RowSelect;
//# sourceMappingURL=RowSelect.d.ts.map