/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { Field, FieldEventMap } from "./Field.js";
import { Toolbar } from "../Toolbar.js";
import { Config, Listener, ObservableListenerOpts } from "../Observable.js";
import { Component } from "../Component.js";
/**
 * @inheritDoc
 */
export interface HtmlFieldEventMap<Type> extends FieldEventMap<Type> {
    /**
     * Fires before adding an item. Return false to abort.
     *
     * @param container
     * @param item
     * @param index
     */
    updatetoolbar: (htmlfield: Type) => void;
    /**
     * Fires when an image is selected, pasted or dropped into the field
     *
     * @param htmlfield
     * @param file
     * @param img The img element in the editor
     */
    insertimage: (htmlfield: Type, file: File, img: HTMLImageElement) => void;
    /**
     * Fires when a non image is pasted or dropped into the field
     *
     * @param htmlfield
     * @param file
     * @param img
     */
    attach: (htmlfield: Type, file: File) => void;
}
export interface HtmlField extends Field {
    on<K extends keyof HtmlFieldEventMap<HtmlField>, L extends Listener>(eventName: K, listener: Partial<HtmlFieldEventMap<HtmlField>>[K], options?: ObservableListenerOpts): L;
    un<K extends keyof HtmlFieldEventMap<this>>(eventName: K, listener: Partial<HtmlFieldEventMap<this>>[K]): boolean;
    fire<K extends keyof HtmlFieldEventMap<HtmlField>>(eventName: K, ...args: Parameters<HtmlFieldEventMap<Component>[K]>): boolean;
}
/**
 * Available toolbar items
 */
type ToolbarItems = "-" | "bold" | "italic" | "underline" | "strikeThrough" | "foreColor" | "backColor" | "removeFormat" | "justifyLeft" | "justifyCenter" | "justifyRight" | "insertOrderedList" | "insertUnorderedList" | "indent" | "outdent" | "image" | "createLink";
/**
 * A HTML editor field component
 *
 * @see Form
 */
export declare class HtmlField extends Field {
    protected baseCls: string;
    /**
     * When the field is empty this will be dispklayed inside the field
     */
    placeholder: string | undefined;
    private editor;
    private tbar?;
    constructor();
    getEditor(): HTMLDivElement;
    private closestStyle;
    /**
     * Toolbar items to enable.
     *
     * If you can't use inline css then use:
     *
     * ```
     * [
     * 		"bold", "italic", "underline",
     * 		"-",
     * 		"foreColor", "removeFormat",
     * 		"-",
     * 		"insertOrderedList",
     * 		"insertUnorderedList",
     * 		"-",
     * 		"indent",
     * 		"outdent",
     * 		"-",
     * 		"image"
     * 	]
     * ```
     */
    toolbarItems: ToolbarItems[];
    private commands;
    private updateToolbar;
    private execCmd;
    /**
     * Inserts an HTML string at the insertion point (deletes selection). Requires a valid HTML string as a value argument.
     *
     * @param html
     */
    insertHtml(html: string): void;
    protected internalRemove(): void;
    protected getToolbar(): Toolbar;
    protected showToolbar(): void;
    protected hideToolbar(): void;
    protected createControl(): undefined | HTMLElement;
    set value(v: string);
    get value(): string;
    focus(o?: FocusOptions): void;
    private lineIndex;
    private lineSequence;
    private static removeCharsFromCursorPos;
    private onKeyDown;
    private onDrop;
    private static _uid;
    /**
     * Generate unique ID
     */
    private static imgUID;
    private handleImage;
    private onPaste;
}
/**
 * Shorthand function to create {@see HtmlField}
 *
 * @param config
 */
export declare const htmlfield: (config?: Config<HtmlField, HtmlFieldEventMap<HtmlField>>) => HtmlField;
export {};
//# sourceMappingURL=HtmlField.d.ts.map