/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { Observable, } from "./Observable.js";
import { State } from "../State.js";
import { browser, Collection } from "../util";
const html = document.querySelector('html');
export const REM_UNIT_SIZE = parseFloat(window.getComputedStyle(html).fontSize);
/**
 * Component
 *
 * A component in its simplest form.
 *
 * @example
 *
 * ```typescript
 * Component.create({
 *   tagName: "hr",
 *   cls: "special"
 * })
 * ```
 */
export class Component extends Observable {
    /**
     * Component constructor
     *
     * @param tagName The tagname used for the root HTMLElement of this component
     */
    constructor(tagName = "div") {
        super();
        this.tagName = tagName;
        /**
         * A base class not configurable. cls can be used to add extra classes leaving this class alone
         *
         * @protected
         */
        this.baseCls = "";
        /**
         * True when added to the DOM tree
         *
         * @private
         */
        this._rendered = false;
        /**
         * Component item ID that can be used to lookup the Component inside a Component with Component.findItem() and
         * Component.findItemIndex();
         *
         * if stateId is given it will also be used as itemId
         */
        this._itemId = "";
        /**
         * Set arbitrary data on a component.
         *
         * Should be used with caution as this data is not typed.
         */
        this.dataSet = {};
    }
    /**
     * Component item ID that can be used to lookup the Component inside a Component with Component.findItem() and
     * Component.findItemIndex();
     *
     * if stateId is given it will also be used as itemId
     */
    get itemId() {
        return this._itemId || this.stateId || this.el.id || "";
    }
    set itemId(itemId) {
        this._itemId = itemId;
    }
    initItems() {
        this.items.on("add", (_collection, item, index) => {
            item.parent = this;
            // fires before render! Menu uses this to modify item.parent
            item.fire("added", item, index, this);
            if (this.rendered) {
                this.renderItem(item);
            }
        });
        this.items.on("remove", (_collection, item) => {
            if (item.parent) {
                item.parent = undefined;
                item.remove();
            }
        });
    }
    getState() {
        return State.get().getItem(this.stateId);
    }
    hasState() {
        return this.stateId && State.get().hasItem(this.stateId);
    }
    /**
     * Restore state of the component in this function. It's called before render in init().
     *
     * @see saveState();
     * @param state
     *
     * @protected
     */
    restoreState(state) {
    }
    /**
     * Call save start when something relevant to the state changes.
     * Implement buildState() to save relevant state properties and restore it in restoreState()
     *
     * stateId must be set on components to be stateful
     *
     * @protected
     */
    saveState() {
        if (this.stateId) {
            State.get().setItem(this.stateId, this.buildState());
        }
    }
    /**
     * Build state for the component
     *
     * @see saveState();
     * @protected
     */
    buildState() {
        return {};
    }
    /**
     * Title of the dom element
     */
    set title(title) {
        this.el.title = title;
    }
    get title() {
        return this.el.title;
    }
    /**
     * Check if the component has been rendered and added to the DOM tree
     */
    get rendered() {
        return this._rendered;
    }
    get el() {
        if (!this._el) {
            this._el = document.createElement(this.tagName);
        }
        return this._el;
    }
    /**
     * Class name to add to element
     *
     * Some common classes to add for layout:
     *
     * - hbox: Set's flex layout to horizontal boxes. Use flex: n to stretch columns
     * - vbox: As above but vertical
     * - fit: Fit the parent's size
     * - scroll: Set's autoscroll: true
     * - pad: Set common padding on the element
     * - border-(top|bottom|left|right) to add a border
     *
     * Other:
     *
     * - goui-fade-in: The component will fade in when show() is used.
     * - goui-fade-out: The component will fade out when hide is used.
     *
     */
    set cls(cls) {
        //remove previously set
        if (this._cls) {
            this._cls.split(/\s+/).forEach(cls => {
                this.el.classList.remove(cls);
            });
        }
        this._cls = cls;
        this.initClassName();
        this.el.className += " " + cls;
    }
    initClassName() {
        if (this.el.classList.contains("goui")) {
            return;
        }
        this.el.classList.add("goui");
        if (this.baseCls) {
            this.el.className += " " + this.baseCls;
        }
    }
    /**
     * Renders the component and it's children
     */
    internalRender() {
        this.initClassName();
        this.renderItems();
        if (this.stateId) {
            this.restoreState(this.getState());
        }
        return this.el;
    }
    /**
     * The tabindex attribute specifies the tab order of an element (when the "tab" button is used for navigating).
     */
    set tabIndex(tabIndex) {
        this.el.tabIndex = tabIndex;
    }
    get tabIndex() {
        return this.el.tabIndex;
    }
    /**
     * CSS flex value
     */
    set flex(flex) {
        this.el.style.flex = flex + "";
    }
    /**
     * CSS flex value
     */
    get flex() {
        return this.el.style.flex;
    }
    /**
     * Make it resizable
     */
    set resizable(resizable) {
        if (resizable) {
            this.el.classList.add("goui-resizable");
        }
        else {
            this.el.classList.remove("goui-resizable");
        }
    }
    get resizable() {
        return this.el.classList.contains("goui-resizable");
    }
    /**
     * Render the component
     *
     * @param parentEl The element this componennt will render into
     * @param insertBefore If given, the element will be inserted before this child
     */
    render(parentEl, insertBefore) {
        if (this._rendered) {
            throw new Error("Already rendered");
        }
        this.fire("beforerender", this);
        // If parent is already rendered then we must determine the DOM index of this child item
        // if parent is rendering then we can simply add it
        if (!parentEl) {
            if (this.renderTo) {
                parentEl = this.renderTo;
            }
            else {
                if (!this.parent) {
                    throw new Error("No parent set for " + (typeof this));
                }
                parentEl = this.parent.itemContainerEl;
                insertBefore = this.getInsertBefore();
            }
        }
        if (!insertBefore) {
            parentEl.appendChild(this.el);
        }
        else {
            parentEl.insertBefore(this.el, insertBefore);
        }
        this.internalRender();
        this._rendered = true;
        this.fire("render", this);
        return this.el;
    }
    /**
     * Finds the DOM node in the parent's children to insert before when rendering a child component
     *
     * @protected
     */
    getInsertBefore() {
        if (!this.parent.rendered) {
            return undefined;
        }
        const index = this.parent.items.indexOf(this);
        let refItem = undefined;
        //find nearest rendered item
        for (let i = index + 1, l = this.parent.items.count(); i < l; i++) {
            if (this.parent.items.get(i).rendered) {
                refItem = this.parent.items.get(i).el;
                break;
            }
        }
        return refItem;
    }
    /**
     * Remove component from the component tree
     */
    remove() {
        if (!this.fire("beforeremove", this)) {
            return false;
        }
        this.internalRemove();
        this.fire("remove", this);
        return true;
    }
    internalRemove() {
        this.items.clear();
        // remove this item from parent the Component
        if (this.parent) {
            // detach from parent here so parent won't call this.remove() again
            const p = this.parent;
            this.parent = undefined;
            p.items.remove(this);
        }
        //remove it from the DOM
        if (this.el) {
            this.el.remove();
        }
    }
    /**
     * Hide element
     */
    set hidden(hidden) {
        if (this.el.hidden == hidden) {
            return;
        }
        const eventName = hidden ? "hide" : "show";
        // noinspection PointlessBooleanExpressionJS
        if (this.fire("before" + eventName, this) === false) {
            return;
        }
        this.internalSetHidden(hidden);
        this.fire(eventName, this);
    }
    /**
     * Overridable method so that the (before)show/hide event fire before and after.
     *
     * @param hidden
     * @protected
     */
    internalSetHidden(hidden) {
        this.el.hidden = hidden;
    }
    get hidden() {
        return this.el.hidden;
    }
    /**
     * Hide this component
     *
     * This sets the "hidden" attribute on the DOM element which set's CSS display:none.
     * You can change this to fade with css class "goui-fade-in" and "goui-fade-out"
     *
     * If you want to override this function, please override internalSetHidden instead so the beforeshow and show event
     * fire at the right time.
     */
    hide() {
        this.hidden = true;
        return this.hidden;
    }
    /**
     * Show the component
     *
     * If you want to override this function, please override internalSetHidden instead so the beforeshow and show event
     * fire at the right time.
     */
    show() {
        this.hidden = false;
        return !this.hidden;
    }
    /**
     * Disable component
     */
    set disabled(disabled) {
        if (!disabled) {
            this.el.removeAttribute("disabled");
        }
        else {
            this.el.setAttribute("disabled", "");
        }
    }
    get disabled() {
        return this.el.hasAttribute('disabled');
    }
    /**
     * Set the HTML contents of the component (innerHTML)
     */
    set html(html) {
        this.el.innerHTML = html;
    }
    get html() {
        return this.el.innerHTML;
    }
    /**
     * Set the innerText
     */
    set text(text) {
        this.el.innerText = text;
    }
    get text() {
        return this.el.innerText;
    }
    /**
     * Set the width in scalable pixels
     *
     * The width is applied in rem units divided by 10. Because the font-size of the html
     * element has a font-size of 62.5% this is equals the amount of pixels, but it can be
     * scaled easily for different themes.
     *
     */
    set width(width) {
        this.el.style.width = (width / 10) + "rem";
    }
    get width() {
        const px = this.el.offsetWidth;
        if (px) {
            return (px / REM_UNIT_SIZE) * 10;
        }
        const styleWidth = this.el.style.width;
        if (styleWidth.substring(styleWidth.length - 3) == "rem") {
            return parseFloat(styleWidth);
        }
        else if (styleWidth.substring(styleWidth.length - 2) == "px") {
            return (parseFloat(styleWidth) / REM_UNIT_SIZE) * 10;
        }
        return 0;
    }
    /**
     * Set inline style
     */
    set style(style) {
        Object.assign(this.el.style, style);
    }
    get style() {
        return this.el.style;
    }
    computeZIndex() {
        const z = parseInt(window.getComputedStyle(this.el).getPropertyValue('z-index'));
        if (!z) {
            return this.parent ? this.parent.computeZIndex() : 0;
        }
        else {
            return z;
        }
    }
    ;
    /**
     * The height in scalable pixels
     *
     * @see width
     */
    set height(height) {
        this.el.style.height = (height / 10) + "rem";
    }
    get height() {
        const px = this.el.offsetHeight;
        if (px) {
            return (px / REM_UNIT_SIZE) * 10;
        }
        const styleHeight = this.el.style.height;
        if (styleHeight.substring(styleHeight.length - 3) == "rem") {
            return parseFloat(styleHeight);
        }
        else if (styleHeight.substring(styleHeight.length - 2) == "px") {
            return (parseFloat(styleHeight) / REM_UNIT_SIZE) * 10;
        }
        return 0;
    }
    /**
     * Element ID
     */
    set id(id) {
        this.el.id = id;
    }
    get id() {
        return this.el.id;
    }
    /**
     * Focus the component
     *
     * @param o
     */
    focus(o) {
        // setTimeout needed for chrome :(
        // setTimeout(() => {
        this.el.focus(o);
        this.fire("focus", this, o);
        // });
    }
    isFocusable() {
        return this.el && !this.hidden && (this.el.tagName == "BUTTON" ||
            this.el.tagName == "INPUT" ||
            this.el.tagName == "A" ||
            this.el.tagName == "AREA" ||
            this.el.tabIndex > -1);
    }
    /**
     * Get the component that's next to this one
     */
    nextSibling() {
        if (!this.parent) {
            return undefined;
        }
        const index = this.parent.items.indexOf(this);
        if (index == -1) {
            return undefined;
        }
        return this.parent.items.get(index + 1);
    }
    /**
     * Get the component that's previous to this one
     */
    previousSibling() {
        if (!this.parent) {
            return undefined;
        }
        const index = this.parent.items.indexOf(this);
        if (index < 1) {
            return undefined;
        }
        return this.parent.items.get(index - 1);
    }
    /**
     * Find ancestor
     *
     * The method traverses the Component's ancestors (heading toward the document root) until it finds
     * one where the given function returns true.
     *
     * @param fn When the function returns true the item will be returned. Otherwise it will move up to the next parent.
     */
    findAncestor(fn) {
        let p = this.parent;
        while (p != undefined) {
            if (fn(p)) {
                return p;
            }
            else {
                p = p.parent;
            }
        }
        return undefined;
    }
    /**
     * Find parent by instance type of the parent
     *
     * @example
     * ```
     * const form = textField.findAncestorByType(Form);
     * ```
     * @param cls
     */
    findAncestorByType(cls) {
        const p = this.findAncestor(cmp => cmp instanceof cls);
        if (p) {
            return p;
        }
        else {
            return undefined;
        }
    }
    /**
     * The child components of this component
     */
    get items() {
        if (!this._items) {
            this._items = new Collection();
            this.initItems();
        }
        return this._items;
    }
    renderItems() {
        if (this._items) {
            this._items.forEach((item) => {
                this.renderItem(item);
            });
        }
    }
    get itemContainerEl() {
        return this.el;
    }
    /**
     * Can be overriden to wrap the component
     *
     * @param item
     * @protected
     */
    renderItem(item) {
        // getInsertBefore check for ext components. They don't have it.
        item.render(this.itemContainerEl, item.getInsertBefore ? item.getInsertBefore() : undefined);
    }
    /**
     * Find the item by element ID, itemId property, Component instance or custom function
     */
    findItemIndex(predicate) {
        let fn = this.createFindPredicateFunction(predicate);
        return this.items.findIndex(fn);
    }
    /**
     * Find the item by element ID, itemId property, Component instance or custom function.
     *
     * If you want to search the component tree hierarchy use {@see findChild()}
     *
     */
    findItem(predicate) {
        let fn = this.createFindPredicateFunction(predicate);
        return this.items.find(fn);
    }
    /**
     * Cascade down the component hierarchy
     *
     * @param fn When the function returns false then the cascading will be stopped. The current Component will be finished!
     */
    cascade(fn) {
        if (fn(this) === false) {
            return this;
        }
        if (this.items) {
            for (let cmp of this.items) {
                cmp.cascade && cmp.cascade(fn);
            }
        }
        return this;
    }
    createFindPredicateFunction(predicate) {
        if (predicate instanceof Function) {
            return predicate;
        }
        else {
            return (item) => {
                return item === predicate || item.itemId === predicate || item.id === predicate;
            };
        }
    }
    /**
     * Find a child at any level by element ID, itemId property, Component instance or custom function.
     *
     * It cascades down the component hierarchy. See also {@see findChildByType}
     *
     */
    findChild(predicate) {
        let fn = this.createFindPredicateFunction(predicate);
        let child;
        this.cascade((item) => {
            if (fn(item)) {
                child = item;
                return false;
            }
        });
        return child;
    }
    /**
     * Find children at any level by element ID, itemId property, Component instance or custom function.
     *
     * It cascades down the component hierarchy. See also {@see findChildByType}
     *
     */
    findChildren(predicate) {
        let fn = this.createFindPredicateFunction(predicate);
        const children = [];
        this.cascade((item) => {
            if (fn(item)) {
                children.push(item);
            }
        });
        return children;
    }
    /**
     * Find child by instance type of the parent
     *
     * @example
     * ```
     * const form = textField.findAncestorByType(Form);
     * ```
     * @param cls
     */
    findChildByType(cls) {
        const p = this.findChild(cmp => cmp instanceof cls);
        if (p) {
            return p;
        }
        else {
            return undefined;
        }
    }
    /**
     * Find children by instance type of the parent
     *
     * @example
     * ```
     * const form = textField.findAncestorByType(Form);
     * ```
     * @param cls
     */
    findChildrenByType(cls) {
        return this.findChildren(Component => Component instanceof cls);
    }
    /**
     * Set attributes of the DOM element
     *
     * @param attr
     */
    set attr(attr) {
        for (let name in attr) {
            this.el.setAttribute(name, attr[name]);
        }
    }
    /**
     * Mask the component to disable user interaction
     * It creates an absolute positioned Mask
     * component. This component should have a non-static position style for this to work.
     */
    mask(delay = 300) {
        if (this.maskTimeout || (this._mask && this._mask.hidden == false)) {
            return;
        }
        this.maskTimeout = setTimeout(() => {
            if (!this._mask) {
                this._mask = mask({ spinner: true });
                this.items.add(this._mask);
            }
            this.el.classList.add("masked");
            this._mask.show();
            this.maskTimeout = undefined;
        }, delay);
    }
    /**
     * Unmask the body
     */
    unmask() {
        if (this.maskTimeout) {
            clearTimeout(this.maskTimeout);
            this.maskTimeout = undefined;
        }
        if (this._mask) {
            this._mask.hide();
        }
        this.el.classList.remove("masked");
    }
    static uniqueID() {
        return "goui-" + (++Component._uniqueID);
    }
    /**
     * Print this component. Everything else will be left out.
     */
    print() {
        // this.el.classList.add("goui-print");
        // window.print();
        //
        // window.addEventListener("afterprint" , () => {
        // 	// document.title = oldTitle;
        // 	// this.el.classList.remove("goui-print");
        // }, {once: true});
        let paper = document.getElementById('paper');
        if (!paper) {
            paper = document.createElement("div");
            paper.id = "paper";
            document.body.appendChild(paper);
        }
        const style = window.getComputedStyle(this.el);
        paper.style.cssText = style.cssText;
        const size = this.el.getBoundingClientRect();
        paper.style.width = size.width + "px";
        paper.innerHTML = this.el.innerHTML;
        const oldTitle = document.title;
        if (this.title) {
            //replace chars not valid for filenames
            document.title = this.title.replace(':', '.').replace(/[/\\?%*|"<>]+/g, '-');
            ;
        }
        if (!browser.isFirefox()) {
            Promise.all(Array.from(document.images).filter(img => !img.complete).map(img => new Promise(resolve => { img.onload = img.onerror = resolve; }))).then(() => {
                browser.isSafari() ? document.execCommand('print') : window.print();
            });
        }
        else {
            // this is not needed in firefox and somehow it also fails to resolve the promises above.
            window.print();
        }
        window.addEventListener("afterprint", () => {
            if (oldTitle) {
                document.title = oldTitle;
            }
            if (paper) {
                paper.innerHTML = "";
            }
        }, { once: true });
    }
}
Component._uniqueID = 0;
/**
 * Mask element
 *
 * Shows a mask over the entire (position:relative) element it's in.
 *
 * Used in {@see Body.mask()}
 */
export class Mask extends Component {
    constructor() {
        super(...arguments);
        this.baseCls = "goui-mask";
    }
    /**
     * Show loading spinner
     */
    set spinner(spinner) {
        if (spinner) {
            this.html = '<div class="spinner"><div></div><div></div><div></div><div></div><div></div><div></div><div></div><div></div></div>';
        }
    }
}
/**
 * Shorthand function to create a {@see Mask} component
 *
 * @param config
 */
export const mask = (config) => createComponent(new Mask(), config);
/**
 * Shorthand function to create {@see Component}
 */
export const comp = (config, ...items) => createComponent(new Component(config === null || config === void 0 ? void 0 : config.tagName), config, items);
export const p = (config, ...items) => createComponent(new Component("p"), typeof config == 'string' ? { html: config } : config, items);
export const small = (config, ...items) => createComponent(new Component("small"), typeof config == 'string' ? { html: config } : config, items);
export const h1 = (config, ...items) => createComponent(new Component("h1"), typeof config == 'string' ? { html: config } : config, items);
export const h2 = (config, ...items) => createComponent(new Component("h2"), typeof config == 'string' ? { html: config } : config, items);
export const h3 = (config, ...items) => createComponent(new Component("h3"), typeof config == 'string' ? { html: config } : config, items);
export const h4 = (config, ...items) => createComponent(new Component("h4"), typeof config == 'string' ? { html: config } : config, items);
export const code = (config, ...items) => createComponent(new Component("code"), typeof config == 'string' ? { html: config } : config, items);
export const section = (config, ...items) => createComponent(new Component("section"), typeof config == 'string' ? { html: config } : config, items);
export const hr = (config) => createComponent(new Component("hr"), config);
export const progress = (config) => createComponent(new Component("progress"), config);
export const createComponent = (comp, config, items) => {
    if (config) {
        if (config.listeners) {
            for (let key in config.listeners) {
                const eventName = key;
                if (typeof config.listeners[eventName] == 'function') {
                    comp.on(eventName, config.listeners[eventName]);
                }
                else {
                    const o = config.listeners[eventName];
                    const fn = o.fn;
                    delete o.fn;
                    comp.on(eventName, fn, o);
                }
            }
            delete config.listeners;
        }
        Object.assign(comp, config);
    }
    if (items && items.length) {
        comp.items.add(...items);
    }
    return comp;
};
//# sourceMappingURL=Component.js.map