/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { Listener, Observable, ObservableEventMap, ObservableListenerOpts } from "./component/Observable.js";
export interface Route {
    re: RegExp;
    handler: Function;
}
/**
 * @inheritDoc
 */
export interface RouterEventMap<Type extends Observable> extends ObservableEventMap<Type> {
    change: (path: string, oldPath: string) => void;
}
export interface Router extends Observable {
    on<K extends keyof RouterEventMap<this>, L extends Listener>(eventName: K, listener: RouterEventMap<this>[K], options?: ObservableListenerOpts): L;
    un<K extends keyof RouterEventMap<this>>(eventName: K, listener: RouterEventMap<this>[K]): boolean;
    fire<K extends keyof RouterEventMap<this>>(eventName: K, ...args: Parameters<RouterEventMap<any>[K]>): boolean;
}
export type RouterMethod = (...args: string[]) => Promise<any> | any;
/**
 * Router class
 *
 * @see router
 */
export declare class Router extends Observable {
    private routes;
    private suspendEvent;
    private loadedPath;
    private defaultRoute?;
    private debug;
    private params;
    constructor();
    /**
     * Get the router path
     */
    getPath(): string;
    /**
     * Get the parameters evaluated from the router path
     */
    getParams(): string[];
    /**
     * Set route path without executing matching routes.
     *
     * @param path
     */
    setPath(path: string): void;
    /**
     * Add a route
     *
     * The first matching route will be executed
     *
     * @example
     * ```
     * go.Router.add(/^([a-zA-Z0-9]*)\/([\d]*)$/, (entity:string, id:string) => {
     *
     * });
     * ```
     *
     * @param re eg. /^notes/(.*)$/
     * @param handler Is called with the arguments matched in the route regexp. May return Promise so the router start()
     *  promise will resolve when this promise is resolved.
     */
    add(re: RegExp | RouterMethod, handler?: RouterMethod): this;
    /**
     * Start the router and run the matching route handlers
     */
    start(): Promise<void>;
    private handleRoute;
    /**
     * Reload current page.
     */
    reload(): void;
    /**
     * Go to the give router path
     *
     * @param path
     * @return Promise<Router>
     */
    goto(path: string): Promise<unknown>;
}
/**
 * Router instance
 *
 * The router will execute all matching route methods!
 *
 * so if the route is /home then both routes will execute
 *
 * @example Dynamic loading and promises
 * ```
 * router.add('/home', () => {})
 *  .add('/home/test', () => {
 *    //may return promise to load a new  module for example and add new sub routes
 *    router.add(/home\/test\/sub/, () => {});
 *
 *    return Promise.resolve();
 *  });
 *
 * ```
 *
 * @example With parameters
 * ```
 * router.
 *  .add(/about/,  () => {
 * 					console.log('about');
 * 				})
 *  .add(/products\/(.*)\/edit\/(.*)/, () => {
 * 					console.log('products', arguments);
 * 				})
 *  .add( () => {
 * 					console.log('default');
 * 				});
 * ```
 */
export declare const router: Router;
//# sourceMappingURL=Router.d.ts.map