var __async = (__this, __arguments, generator) => {
  return new Promise((resolve, reject) => {
    var fulfilled = (value) => {
      try {
        step(generator.next(value));
      } catch (e) {
        reject(e);
      }
    };
    var rejected = (value) => {
      try {
        step(generator.throw(value));
      } catch (e) {
        reject(e);
      }
    };
    var step = (x) => x.done ? resolve(x.value) : Promise.resolve(x.value).then(fulfilled, rejected);
    step((generator = generator.apply(__this, __arguments)).next());
  });
};

// goui/script/util/FunctionUtil.ts
var BufferedFunction = class {
  /**
   * Constructor
   * @param delay Delay to execute function
   * @param fn Function to buffer
   */
  constructor(delay, fn) {
    this.delay = delay;
    this.fn = fn;
    this.delay = delay;
    this.fn = fn;
  }
  /**
   * Bugger the function with the delay set on this class
   *
   * @param args
   */
  buffer(args = []) {
    this.cancel();
    return new Promise((resolve) => {
      this.id = window.setTimeout(() => {
        this.cancel();
        resolve(this.fn.apply(null, args));
      }, this.delay);
    });
  }
  /**
   * Cancel the function call
   */
  cancel() {
    if (this.id) {
      clearTimeout(this.id);
      this.id = void 0;
    }
  }
  /**
   * Check if it's still pending for execution
   */
  pending() {
    return this.id !== void 0;
  }
};
var FunctionUtil = class {
  /**
   * Buffer the function with this number of milliseconds.
   * When the function is called multiple times within the delay period only the last call will execute.
   *
   * @param delay
   * @param fn
   */
  static buffer(delay, fn) {
    const bf = new BufferedFunction(delay, fn);
    return (...args) => {
      return bf.buffer(args);
    };
  }
  /**
   *
   * Delay function execution
   *
   * @param delay
   * @param fn
   */
  static delay(delay, fn) {
    return (...args) => {
      const bf = new BufferedFunction(delay, fn);
      bf.buffer(args);
    };
  }
  /**
   * Execute on the next repaint
   *
   * The onRepaint method tells the browser that you wish to perform an animation and requests
   * that the browser calls a specified function to update an animation before the next repaint.
   *
   * @link https://developer.mozilla.org/en-US/docs/Web/API/window/requestAnimationFrame
   * @param fn
   */
  static onRepaint(fn) {
    let frame = -1;
    return (...args) => {
      if (frame > -1) {
        cancelAnimationFrame(frame);
      }
      frame = window.requestAnimationFrame(() => {
        fn.apply(null, args);
      });
    };
  }
  /**
   * Will combine the given functions into one.
   *
   * The newFn function will be called with the return value of origFn function + the parameters of origFn
   * and function will return the value of newFn
   *
   * @param origFn
   * @param newFn
   *
   * @example
   * ```
   * function test(a:number, b:string) : string {
   * 	return b;
   * }
   *
   * FunctionUtil.createSequence(test, function(r, a, b) {
   * 	return r + "b";
   * })
   * ```
   *
   */
  static createSequence(origFn, newFn) {
    return function(...args) {
      const r = origFn.apply(this, args);
      return newFn.call(this, r, ...args);
    };
  }
  /**
   * Create a combined function of an orignal and new function. The new function will be called
   * before the original,
   *
   * @param origFn
   * @param newFn
   *
   * @example
   * ```
   * export function playgroundTableOverride() {
   * 	PlaygroundTable.prototype.render = FunctionUtil.createInterceptor(
   * 		PlaygroundTable.prototype.render,
   * 		function(this:PlaygroundTable) {
   * 			this.el.classList.add("cls-added-by-override");
   * 		}
   * 	)
   * }
   * ```
   */
  static createInterceptor(origFn, newFn) {
    return function(...args) {
      newFn.apply(this, args);
      return origFn.apply(this, args);
    };
  }
};

// goui/script/component/Observable.ts
var Observable = class {
  /**
   * Add a listener
   *
   * @param eventName
   * @param listener
   * @param options
   */
  on(eventName, listener, options) {
    const unbindkey = listener;
    if (options) {
      if (options.buffer !== void 0) {
        listener = FunctionUtil.buffer(options.buffer, listener);
      }
      if (options.once) {
        listener = this.once(eventName, listener);
      }
      if (options.delay) {
        listener = FunctionUtil.delay(options.delay, listener);
      }
    }
    this.lisnrs = this.lisnrs || {};
    if (!this.lisnrs[eventName]) {
      this.lisnrs[eventName] = [];
      this.onFirstListenerAdded(eventName);
    }
    if (options == null ? void 0 : options.unshift) {
      this.lisnrs[eventName].unshift({ listener, options, unbindkey });
    } else {
      this.lisnrs[eventName].push({ listener, options, unbindkey });
    }
    return unbindkey;
  }
  /**
   * Override this function to lazily initialize events
   * @param eventName
   * @protected
   */
  onFirstListenerAdded(eventName) {
  }
  once(eventName, listener) {
    const newfn = (...args) => {
      listener.apply(null, args);
      setTimeout(() => {
        this.un(eventName, listener);
      });
    };
    return newfn;
  }
  /**
   * Remove listener
   *
   * @param eventName
   * @param listener
   */
  un(eventName, listener) {
    if (!this.lisnrs || !this.lisnrs[eventName]) {
      return false;
    }
    for (let i = 0, l = this.lisnrs[eventName].length; i < l; i++) {
      const l2 = this.lisnrs[eventName][i];
      if (l2.unbindkey === listener) {
        this.lisnrs[eventName].splice(i, 1);
        return true;
      }
    }
    return false;
  }
  /**
   * Fire an event
   *
   * When a listener returns false this function will return false too.
   *
   * @param eventName
   * @param args
   */
  fire(eventName, ...args) {
    if (!this.lisnrs || !this.lisnrs[eventName]) {
      return true;
    }
    let ret = true;
    for (let l of this.lisnrs[eventName]) {
      if (l.listener.apply(null, args) === false) {
        ret = false;
      }
    }
    return ret;
  }
  relayEvent(comp2, type) {
    comp2.on(type, (...args) => {
      this.fire(type, ...args);
    });
  }
};

// goui/script/State.ts
var State = class _State {
  static get() {
    return new _State();
  }
  hasItem(id) {
    return localStorage.getItem("state-" + id) !== null;
  }
  getItem(id) {
    const json = localStorage.getItem("state-" + id);
    if (!json) {
      return {};
    }
    return JSON.parse(json);
  }
  setItem(id, state) {
    localStorage.setItem("state-" + id, JSON.stringify(state));
  }
};

// goui/script/util/ArrayUtil.ts
var ArrayUtil = class {
  /**
   * Get array of all values of array1 that are NOT present in array2
   *
   * Opposite of intersect()
   *
   * @param array1
   * @param array2
   */
  static diff(array1, array2) {
    return array1.filter((i) => {
      return array2.indexOf(i) === -1;
    });
  }
  /**
   * Get array of all values of array1 that are ALSO present in array2
   *
   * @param array1
   * @param array2
   */
  static intersect(array1, array2) {
    return array1.filter((i) => {
      return array2.indexOf(i) !== -1;
    });
  }
  /**
   * Sort an array by multiple fields
   *
   * @param array
   * @param comparators
   */
  static multiSort(array, comparators) {
    if (!comparators.length) {
      return array;
    }
    const keySort = (a2, b, isAscending) => {
      const direction = isAscending ? 1 : -1;
      if (a2 === b) {
        return 0;
      }
      return a2 > b ? direction : -1 * direction;
    };
    return array.sort((a2, b) => {
      let sorted = 0;
      let index = 0;
      while (sorted === 0 && index < comparators.length) {
        const key = comparators[index].property;
        if (key) {
          sorted = keySort(a2[key], b[key], comparators[index].isAscending || comparators[index].isAscending === void 0);
          index++;
        }
      }
      return sorted;
    });
  }
};

// goui/script/util/Browser.ts
var Browser = class {
  constructor() {
    this.cache = {};
    this.ua = navigator.userAgent.toLowerCase();
  }
  check(r, cacheKey) {
    if (!this.cache[cacheKey]) {
      this.cache[cacheKey] = r.test(this.ua);
    }
    return this.cache[cacheKey];
  }
  isWebkit() {
    return this.check(/webkit/, "isWebkit");
  }
  isChrome() {
    return this.check(/\bchrome\b/, "isChrome");
  }
  isSafari() {
    return !this.isChrome() && this.check(/safari/, "isSafari");
  }
  isFirefox() {
    return this.check(/gecko/, "isFirefox");
  }
  isMac() {
    return this.check(/macintosh|mac os x/, "isMac");
  }
  isLinux() {
    return this.check(/linux/, "isLinux");
  }
  isWindows() {
    return this.check(/windows|win32/, "isWindows");
  }
  /**
   * Open file upload dialog to get local files
   *
   * @example
   *
   * ```
   * btn({
   * 	type: "button",
   * 	text: t("Attach files"),
   * 	icon: "attach_file",
   * 	handler: async () => {
   *
   * 		const files = await browser.pickLocalFiles(true);
   * 		this.mask();
   * 		const blobs = await client.uploadMultiple(files);
   * 		this.unmask();
   * 	  console.warn(blobs);
   *
   * 	}
   * })
   * ```
   *
   * @param accept eg. "image/*"
   * @param directory Allow directory upload
   * @param multiple Allow multiple files
   */
  pickLocalFiles(multiple = false, directory = false, accept = void 0) {
    return new Promise((resolve, reject) => {
      const input = document.createElement("input");
      input.style.display = "none";
      input.setAttribute("type", "file");
      input.addEventListener("change", (e) => {
        if (input.files) {
          resolve(Array.from(input.files));
        }
        input.value = "";
      });
      document.body.appendChild(input);
      if (accept) {
        input.setAttribute("accept", accept);
      }
      if (directory) {
        input.setAttribute("webkitdirectory", "true");
        input.setAttribute("directory", "true");
      }
      if (multiple) {
        input.setAttribute("multiple", "true");
      }
      input.click();
      input.remove();
    });
  }
  /**
   * Copy text to clip board
   *
   * @param {string} text
   */
  copyTextToClipboard(text) {
    const al = document.activeElement;
    if (!navigator.clipboard) {
      const textArea = document.createElement("textarea");
      textArea.value = text;
      document.body.appendChild(textArea);
      textArea.focus();
      textArea.select();
      try {
        const successful = document.execCommand("copy");
        const msg = successful ? "successful" : "unsuccessful";
        console.log("Fallback: Copying text command was " + msg);
      } catch (err) {
        console.error("Fallback: Oops, unable to copy", err);
      }
      document.body.removeChild(textArea);
      if (al && al.focus) {
        al.focus();
      }
      return;
    }
    navigator.clipboard.writeText(text).then(function() {
      console.log("Async: Copying to clipboard was successful!");
      if (al && al.focus) {
        al.focus();
      }
    }, function(err) {
      console.error("Async: Could not copy text: ", err);
    });
  }
};
var browser = new Browser();

// goui/script/util/Collection.ts
var Collection = class extends Observable {
  constructor(items = []) {
    super();
    this.items = items;
  }
  [Symbol.iterator]() {
    return this.items[Symbol.iterator]();
  }
  /**
   * Add items at the end
   *
   * @returns the index of the last added item
   */
  add(...items) {
    let index = -1;
    items.forEach((item) => {
      index = this.items.length;
      if (!this.fire("beforeadd", this, item, index)) {
        return -1;
      }
      this.items.push(item);
      this.fire("add", this, item, index);
      this.fire("datachanged", this);
    });
    return index;
  }
  /**
   * Insert items at the given index
   *
   * @param index Use negative indexes to insert from the end. For example -1 inserts before the last item.
   */
  insert(index, ...items) {
    if (index < 0) {
      index = this.count() + index;
    }
    items.forEach((item) => {
      if (!this.fire("beforeadd", this, item, index)) {
        return -1;
      }
      this.items.splice(index, 0, item);
      this.fire("add", this, item, index);
      this.fire("datachanged", this);
      index++;
    });
    return index;
  }
  /**
   * Get an item at the given index
   * @param index
   */
  get(index) {
    return this.items[index];
  }
  /**
   * Return first item
   */
  first() {
    return this.get(0);
  }
  /**
   * return the last item
   */
  last() {
    return this.get(this.count() - 1);
  }
  /**
   * Find the index of an item. Returns -1 if not found.
   * @param item
   */
  indexOf(item) {
    return this.items.indexOf(item);
  }
  /**
   * Remove items
   */
  remove(...items) {
    items.forEach((item) => {
      const index = this.indexOf(item);
      if (index == -1) {
        return false;
      } else {
        return this.removeAt(index);
      }
    });
  }
  /**
   * Remove an item
   *
   * @param index Item index
   */
  removeAt(index) {
    const item = this.get(index);
    if (!item) {
      return false;
    }
    if (!this.fire("beforeremove", this, item, index)) {
      return false;
    }
    this.items.splice(index, 1);
    this.fire("remove", this, item, index);
    this.fire("datachanged", this);
    return true;
  }
  /**
   * Count the number of items
   */
  count() {
    return this.items.length;
  }
  /**
   * Clears all items
   */
  clear() {
    while (this.count()) {
      this.removeAt(0);
    }
    return this;
  }
  /**
   * Replace the collection with an array
   *
   * @param items
   */
  replace(...items) {
    this.clear().add(...items);
  }
  /**
   * Performs the specified action for each element in an array.
   * @param callbackfn  A function that accepts up to three arguments. forEach calls the callbackfn function one time for each element in the array.
   * @param thisArg  An object to which the this keyword can refer in the callbackfn function. If thisArg is omitted, undefined is used as the this value.
   */
  forEach(callbackfn, thisArg) {
    return this.items.forEach(callbackfn, thisArg);
  }
  getArray() {
    return this.items;
  }
  /**
   * Returns the value of the first element in the array where predicate is true, and undefined
   * otherwise.
   * @param predicate find calls predicate once for each element of the array, in ascending
   * order, until it finds one where predicate returns true. If such an element is found, find
   * immediately returns that element value. Otherwise, find returns undefined.
   *
   * @returns CollectionItem | undefined
   */
  find(predicate) {
    return this.items.find(predicate);
  }
  /**
   * The filter() method creates a new array with all elements that pass the test implemented by the provided function.
   *
   * @param predicate find calls predicate once for each element of the array, in ascending
   * order, until it finds one where predicate returns true. If such an element is found, find
   * immediately returns that element value. Otherwise, find returns undefined.
   *
   * @returns CollectionItem | undefined
   */
  filter(predicate) {
    return this.items.filter(predicate);
  }
  /**
   * Check if the given item is present
   *
   * @param item
   */
  has(item) {
    return this.findIndex((v) => v == item) > -1;
  }
  /**
   * Returns the index of the first element in the array where predicate is true, and -1
   * otherwise.
   * @param predicate find calls predicate once for each element of the array, in ascending
   * order, until it finds one where predicate returns true. If such an element is found,
   * findIndex immediately returns that element index. Otherwise, findIndex returns -1.
   */
  findIndex(predicate) {
    return this.items.findIndex(predicate);
  }
};

// goui/script/util/Cookies.ts
var Cookies = class {
  /**
   * Set a cookie
   *
   * @param {string} name
   * @param {string} value
   * @param {int} maxAge Maximum age in seconds. Leave empty to clear on browser close
   * @param {string} path
   * @returns {void}
   */
  set(name, value, maxAge, path) {
    path = path || document.location.pathname;
    let maxAgeStr = "";
    if (maxAge) {
      maxAgeStr = ";Max-Age=" + maxAge;
    }
    const secure = location.protocol === "https:" ? ";secure" : "";
    document.cookie = name + "=" + encodeURIComponent(value + "") + maxAgeStr + secure + ";path=" + path + ";SameSite=Strict";
  }
  /**
   * Get a cookie
   *
   * @param {string} name
   * @returns {string}
   */
  get(name) {
    const nameEQ = name + "=";
    const ca = document.cookie.split(";");
    for (let i = 0; i < ca.length; i++) {
      let c = ca[i];
      while (c.charAt(0) == " ")
        c = c.substring(1, c.length);
      if (c.indexOf(nameEQ) == 0)
        return decodeURIComponent(c.substring(nameEQ.length, c.length));
    }
    return void 0;
  }
  /**
   * Unset a cookie
   *
   * @param {string} name
   * @returns {void}
   */
  unset(name, path) {
    path = path || document.location.pathname;
    document.cookie = name + "=; expires=Thu, 01 Jan 1970 00:00:00 UTC; path=" + path + ";SameSite=Strict";
  }
};
var cookies = new Cookies();

// goui/script/util/DateInterval.ts
function pad(n) {
  return n.toString().padStart(2, "0");
}
var _DateInterval = class _DateInterval {
  /**
   * Constructor
   *
   * @link https://en.wikipedia.org/wiki/ISO_8601#Durations
   * @param duration ISO 8601 Duration
   */
  constructor(duration) {
    /**
     * Number of microseconds.
     */
    this.microSeconds = 0;
    /**
     * Number of seconds
     */
    this.seconds = 0;
    /**
     * Number of minutes
     */
    this.minutes = 0;
    /**
     * Number of hours
     */
    this.hours = 0;
    /**
     * Number of days
     */
    this.days = 0;
    /**
     * NUmber of months
     */
    this.months = 0;
    /**
     * Number of years
     */
    this.years = 0;
    /**
     * True if it represents a negative period
     */
    this.invert = false;
    if (duration && !this.setDuration(duration)) {
      throw "Invalid duration: " + duration;
    }
  }
  /**
   * Set interval from the time elapsed bewtween to datetime objects
   *
   * @param start
   * @param end
   */
  setFromDates(start, end) {
    this._start = start;
    this._end = end;
    let monthDays = end.clone().setDate(0).getDate(), sihdmy = [0, 0, 0, 0, 0, end.getYear() - start.getYear()], it = 0, map = { getSeconds: 60, getMinutes: 60, getHours: 24, getDate: monthDays, getMonth: 12 };
    for (let i in map) {
      let fn = i;
      if (sihdmy[it] + end[fn]() < start[fn]()) {
        sihdmy[it + 1]--;
        sihdmy[it] += map[fn] - start[fn]() + end[fn]();
      } else if (sihdmy[it] + end[fn]() > start[fn]()) {
        sihdmy[it] += end[fn]() - start[fn]();
      }
      it++;
    }
    this.seconds = sihdmy[0];
    this.minutes = sihdmy[1];
    this.hours = sihdmy[2];
    this.days = sihdmy[3];
    this.months = sihdmy[4];
    this.years = sihdmy[5];
  }
  /**
   * Calculates total number of days that have elapsed between two dates.
   *
   * Only available if this diff was created using {@link DateTime.diff}
   */
  getTotalDays() {
    if (!this._start || !this._end) {
      return void 0;
    }
    return Math.floor((Date.UTC(this._end.getYear(), this._end.date.getMonth(), this._end.getDate()) - Date.UTC(this._start.getYear(), this._start.date.getMonth(), this._start.getDate())) / 864e5);
  }
  /**
   * Calculates total number of minutes that have elapsed between two dates.
   *
   * Only available if this diff was created using {@link DateTime.diff}
   */
  getTotalMinutes() {
    if (this._start && this._end) {
      return Math.abs(this._start.getTime() - this._end.getTime()) / 6e4;
    }
    if (this.days || this.months || this.years) {
      return void 0;
    }
    return this.hours * 60 + this.minutes;
  }
  /**
   * Set the interval from an ISO8601 duration
   * @link https://en.wikipedia.org/wiki/ISO_8601#Durations
   * @param duration
   */
  setDuration(duration) {
    const iso8601DurationRegex = /(-)?P(?:([.,\d]+)Y)?(?:([.,\d]+)M)?(?:([.,\d]+)W)?(?:([.,\d]+)D)?(?:T(?:([.,\d]+)H)?(?:([.,\d]+)M)?(?:([.,\d]+)S)?)?/;
    const matches = duration.match(iso8601DurationRegex);
    if (!matches) {
      return false;
    }
    this.invert = matches[1] !== void 0;
    this.years = matches[2] ? parseInt(matches[2]) : 0;
    this.months = matches[3] ? parseInt(matches[3]) : 0;
    this.days = matches[4] ? parseInt(matches[4]) * 7 : 0;
    if (!matches[4]) {
      this.days = matches[5] ? parseInt(matches[5]) : 0;
    }
    this.hours = matches[6] ? parseInt(matches[6]) : 0;
    this.minutes = matches[7] ? parseInt(matches[7]) : 0;
    this.seconds = matches[8] ? parseInt(matches[8]) : 0;
    return true;
  }
  /**
   * Build an Iso8601 duration string
   *
   * @link https://en.wikipedia.org/wiki/ISO_8601#Durations
   */
  toIso8601() {
    return "P" + (this.years > 0 ? this.years + "Y" : "") + (this.months > 0 ? this.months + "M" : "") + (this.days > 0 ? this.days + "D" : "") + (this.hours || this.minutes || this.seconds ? "T" + (this.hours > 0 ? this.hours + "H" : "") + (this.minutes > 0 ? this.minutes + "M" : "") + (this.seconds > 0 ? this.seconds + "S" : "") : "");
  }
  /**
   * Format the interval to a string.
   *
   * You can use the following characters. You can escape a character with a \ to output it as given:
   *
   * Y	Years, numeric, at least 2 digits with leading 0, eg.	01, 03
   * y	Years, numeric	1, 3
   * M	Months, numeric, at least 2 digits with leading 0, eg.	01, 03, 12
   * m	Months, numeric, eg.	1, 3, 12
   * D	Days, numeric, at least 2 digits with leading 0, eg.	01, 03, 31
   * d	Days, numeric, eg.	1, 3, 31
   * a	Total number of days as a result of a {@link DateTime.diff} or (unknown) otherwise, eg.	4, 18, 8123
   * H	Hours, numeric, at least 2 digits with leading 0, eg.	01, 03, 23
   * h	Hours, numeric, eg.	1, 3, 23
   * I	Minutes, numeric, at least 2 digits with leading 0, eg.	01, 03, 59
   * i	Minutes, numeric, eg.	1, 3, 59
   * j  The total number of minutes as a result of a {@link DateTime.diff} or (unknown) if this duration holds more than hours and minutes and seconds, eg.	4, 18, 8123
   * S	Seconds, numeric, at least 2 digits with leading 0, eg.	01, 03, 57
   * s	Seconds, numeric, eg.	1, 3, 57
   * F	Microseconds, numeric, at least 6 digits with leading 0, eg.	007701, 052738, 428291
   * f	Microseconds, numeric, eg.	7701, 52738, 428291
   * R	Sign "-" when negative, "+" when positive, eg.	-, +
   * r	Sign "-" when negative, empty when positive, eg.	-,
   *
   * @param format
   */
  format(format) {
    const chars = format.split("");
    let output = "";
    for (let i = 0, l = chars.length; i < l; i++) {
      let char = chars[i];
      if (char == "\\") {
        i++;
        if (chars.length > i + 1) {
          char += chars[i];
        }
      } else if (char in _DateInterval.converters) {
        char = _DateInterval.converters[char](this) + "";
      }
      output += char;
    }
    return output;
  }
  static createFormatRegex(format) {
    const chars = format.split("");
    let output = "";
    for (let i = 0, l = chars.length; i < l; i++) {
      let char = chars[i];
      switch (char) {
        case "Y":
          char = "(?<Y>\\d{2,4})";
          break;
        case "I":
        case "M":
        case "S":
          char = "(?<" + char + ">\\d{2})";
          break;
        case "y":
        case "s":
        case "i":
        case "H":
        case "m":
        case "d":
        case "h":
          char = "(?<" + char + ">\\d{1,2})";
          break;
        case "R":
          char = "(?<" + char + ">[-+]{1})";
          break;
        case "r":
          char = "(?<" + char + ">-?)";
          break;
        default:
          break;
      }
      output += char;
    }
    return output;
  }
  /**
   * Create date by given format. See {@link DateInterval.format}.
   *
   * Does not support "a". "j" can only be used alone.
   *
   * @example
   * ```
   * const dateInterval = DateInterval.createFromFormat("21:09", "h:I"));
   *
   * const dateInterval = DateInterval.createFromFormat("315", "j"));
   * ```
   */
  static createFromFormat(dateStr, format = "h:I") {
    if (format == "j") {
      const date2 = new _DateInterval();
      const mins = parseInt(dateStr);
      date2.minutes = mins % 60;
      date2.hours = Math.floor(mins / 60);
      return date2;
    }
    const regex = new RegExp(_DateInterval.createFormatRegex(format), "u");
    const result = regex.exec(dateStr);
    if (!result) {
      return void 0;
    }
    const date = new _DateInterval();
    for (let key in result.groups) {
      switch (key) {
        case "Y":
        case "y":
          date.years = parseInt(result.groups[key]);
          break;
        case "M":
        case "m":
          date.months = parseInt(result.groups[key]);
          break;
        case "D":
        case "d":
          date.days = parseInt(result.groups[key]);
          break;
        case "H":
        case "h":
          date.hours = parseInt(result.groups[key]);
          break;
        case "I":
        case "i":
          date.minutes = parseInt(result.groups[key]);
          break;
        case "S":
        case "s":
          date.seconds = parseInt(result.groups[key]);
          break;
        case "F":
        case "f":
          date.microSeconds = parseInt(result.groups[key]);
          break;
        case "R":
        case "r":
          date.invert = result.groups[key] == "-";
          break;
      }
    }
    return date;
  }
  /**
   * Compare with given interval
   *
   * Returns:
   *
   * - -1 if this date is before the given date
   * - 0 if dates are equal
   * - 1 if this date is after the given date
   *
   * @param date
   * @return number
   */
  compare(other) {
    const thisDuration = parseInt(this.format("rYMDHISF"));
    const otherDuration = parseInt(other.format("rYMDHISF"));
    const result = (thisDuration > otherDuration ? 1 : 0) - (thisDuration < otherDuration ? 1 : 0);
    return result;
  }
};
_DateInterval.converters = {
  "Y": (dateInterval) => pad(dateInterval.years),
  "y": (dateInterval) => dateInterval.years.toString(),
  "M": (dateInterval) => pad(dateInterval.months),
  "m": (dateInterval) => dateInterval.months.toString(),
  "D": (dateInterval) => pad(dateInterval.days),
  "d": (dateInterval) => dateInterval.days.toString(),
  "a": (dateInterval) => {
    var _a;
    return ((_a = dateInterval.getTotalDays()) != null ? _a : "") + "";
  },
  "H": (dateInterval) => pad(dateInterval.hours),
  "h": (dateInterval) => dateInterval.hours.toString(),
  "I": (dateInterval) => pad(dateInterval.minutes),
  "i": (dateInterval) => dateInterval.minutes.toString(),
  "j": (dateInterval) => {
    var _a;
    return ((_a = dateInterval.getTotalMinutes()) != null ? _a : "") + "";
  },
  "S": (dateInterval) => pad(dateInterval.seconds),
  "s": (dateInterval) => dateInterval.seconds.toString(),
  "F": (dateInterval) => pad(dateInterval.microSeconds),
  "f": (dateInterval) => dateInterval.microSeconds.toString(),
  "R": (dateInterval) => dateInterval.invert ? "-" : "+",
  "r": (dateInterval) => dateInterval.invert ? "-" : ""
};
var DateInterval = _DateInterval;

// goui/script/util/DateTime.ts
var SystemTimeZone = Intl.DateTimeFormat().resolvedOptions().timeZone.toLowerCase();
function pad2(n) {
  return n.toString().padStart(2, "0");
}
var _DateTime = class _DateTime {
  /**
   * Constructor
   *
   * @param date Can be a date object, a unix timestamp or date string (see https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Date#date_time_string_format)
   */
  constructor(date) {
    /**
     * The timezone of the date
     */
    this.timezone = SystemTimeZone;
    this.date = date instanceof Date ? date : date ? new Date(date) : /* @__PURE__ */ new Date();
  }
  static staticInit(lang) {
    const locale = new Intl.Locale(lang), dayList = ["su", "mo", "tu", "we", "th", "fr", "sa"];
    if ("weekInfo" in locale) {
      _DateTime.firstWeekDay = locale.weekInfo.firstDay;
    }
    let tmp = /* @__PURE__ */ new Date("1970-01-01"), intlDays = new Intl.DateTimeFormat(lang, { weekday: "long" });
    for (let i = 0; i < 7; i++) {
      tmp.setDate(i + 4 + _DateTime.firstWeekDay);
      const d = dayList[tmp.getDay()];
      _DateTime.dayMap.push(d);
      _DateTime.dayNames[d] = intlDays.format(tmp);
    }
    let intlMonth = new Intl.DateTimeFormat(lang, { month: "long" });
    for (let i = 0; i < 12; i++) {
      tmp.setMonth(i);
      _DateTime.monthNames[i] = intlMonth.format(tmp);
    }
  }
  /**
   * User timezone offset in minutes
   *
   * eg.
   *
   * ```
   * const browser = new Date("2021-10-21T16:00:00+00:00");
   *
   * const amsterdam = utc.toTimezone("europe/amsterdam");
   *
   * console.log(amsterdam.getUserTimezoneOffset()); // 120 (in daylight savings time)
   * ```
   */
  getTimezoneOffset() {
    const systemTZOffset = this.date.getTimezoneOffset();
    return systemTZOffset + this.getSystemTimezoneDiff();
  }
  /**
   * Convert date to timezone
   *
   * @example On computer with Amsterdam timezone
   *
   * ```
   * const browser = new Date("2021-10-21T16:00:00+00:00");
   * console.log(browser.format("c"), browser.getTimeZone());
   * const riga = browser.toTimezone("europe/riga");
   *
   * console.log(riga.format("c"), riga.getTimeZone());
   *
   * const utc = riga.toUTC();
   *
   * console.log(utc.format("c"), utc.getTimeZone());
   *
   * ```
   *
   * Output:
   * 2021-10-21T18:00:00+02:00 Europe/Amsterdam
   * 2021-10-21T19:00:00+03:00 europe/riga
   * 2021-10-21T16:00:00+00:00 UTC
   *
   *
   * @param timezone eg. europe/amsterdam
   */
  toTimezone(timezone) {
    if (this.timezone == timezone) {
      return this.clone();
    }
    const offset = this.getTimezoneOffset();
    const d = this.clone();
    d.timezone = timezone;
    const newOffset = d.getTimezoneOffset();
    d.setMinutes(d.getMinutes() - newOffset + offset);
    return d;
  }
  /**
   * Calculate difference between this and the given date
   *
   * @param end
   */
  diff(end) {
    const di = new DateInterval();
    di.setFromDates(this, end);
    return di;
  }
  /**
   * Create a copy of this object without reference
   */
  clone() {
    return new _DateTime(new Date(this.date));
  }
  /**
   * Convert to UTC timezone
   */
  toUTC() {
    return this.toTimezone("UTC");
  }
  static getFormatter(timezone) {
    if (!_DateTime.cache[timezone]) {
      _DateTime.cache[timezone] = new Intl.DateTimeFormat("en-US", {
        timeZone: timezone,
        dateStyle: "short",
        timeStyle: "short"
      });
    }
    return _DateTime.cache[timezone];
  }
  /**
   * Get the offset between the system timezone and the date timezone in minutes.
   *
   * For example if the computer has europe/amsterdam and the date UTF it's 60 minutes in winter time.
   *
   * @private
   */
  getSystemTimezoneDiff() {
    if (this.timezone == SystemTimeZone) {
      return 0;
    }
    const local = _DateTime.getFormatter(this.timezone).format(this.date);
    const d = new Date(local);
    const u = Math.floor(this.date.getTime() / 1e3 / 60), localU = Math.floor(d.getTime() / 1e3 / 60);
    return u - localU;
  }
  /**
   * The ISO-8601 week number of year (weeks starting on Monday)
   */
  getWeekOfYear() {
    const ms1d = 864e5, ms7d = 7 * ms1d;
    const DC3 = Date.UTC(this.getYear(), this.getMonth() - 1, this.getMonthDay() + 3) / ms1d, AWN = Math.floor(DC3 / 7), Wyr = new Date(AWN * ms7d).getUTCFullYear();
    return AWN - Math.floor(Date.UTC(Wyr, 0, 7) / ms7d) + 1;
  }
  /**
   * Get the numeric day number of the year
   */
  getDayOfYear() {
    let num = 0, d = new _DateTime(this.date.getTime()), m = this.getMonth(), i;
    for (i = 1, d.setMonthDay(1), d.setMonth(1); i < m; d.setMonth(++i)) {
      num += d.getDaysInMonth();
    }
    return num + this.getMonthDay() - 1;
  }
  /**
   * Get the year
   */
  getYear() {
    return this.date.getFullYear();
  }
  /** Gets the month. Unlike JS it's 1 - 12 */
  getMonth() {
    return this.date.getMonth() + 1;
  }
  getDate() {
    return this.date.getDate();
  }
  getMonthDay() {
    return this.getDate();
  }
  /** 0 for sunday, 1 for monday, 2 for tuesday */
  getDay() {
    return this.date.getDay();
  }
  /**
   * Like getDay but take firstWeekDay of the week into account
   * 0 = first day of the week, 6 = last day
   */
  getWeekDay() {
    if (_DateTime.firstWeekDay == 1) {
      return (this.date.getDay() || 7) - 1;
    }
    return this.date.getDay();
  }
  getHours() {
    return this.date.getHours();
  }
  getMinutes() {
    return this.date.getMinutes();
  }
  getSeconds() {
    return this.date.getSeconds();
  }
  getMilliseconds() {
    return this.date.getMilliseconds();
  }
  /** Gets the time value in milliseconds. */
  getTime() {
    return this.date.getTime();
  }
  getMinuteOfDay() {
    return this.date.getHours() * 60 + this.date.getMinutes();
  }
  /**
   * Sets the hour value in the Date object
   * @param hours A numeric value equal to the hours value.
   * @params min A numeric value equal to the minutes value.
   * @param sec A numeric value equal to the seconds value.
   */
  setHours(hours, min = this.date.getMinutes(), sec = this.date.getSeconds(), ms = this.date.getMilliseconds()) {
    this.date.setHours(hours, min, sec, ms);
    return this;
  }
  setMinutes(min) {
    this.date.setMinutes(min);
    return this;
  }
  setSeconds(s) {
    this.date.setSeconds(s);
    return this;
  }
  setMilliseconds(ms) {
    this.date.setMilliseconds(ms);
    return this;
  }
  setYear(year) {
    this.date.setFullYear(year);
    return this;
  }
  setMonth(month) {
    this.date.setMonth(month - 1);
    return this;
  }
  setDate(date) {
    this.date.setDate(date);
    return this;
  }
  setMonthDay(date) {
    return this.setDate(date);
  }
  setWeekDay(day) {
    this.date.setDate(this.date.getDate() - this.getWeekDay() + day);
    return this;
  }
  /** Jump to day in current week */
  setDay(day) {
    this.date.setDate(this.date.getDate() - this.date.getDay() + day);
    return this;
  }
  addYears(years) {
    this.date.setFullYear(this.date.getFullYear() + years);
    return this;
  }
  addMonths(months) {
    this.date.setMonth(this.date.getMonth() + months);
    return this;
  }
  addDays(days) {
    this.date.setDate(this.date.getDate() + days);
    return this;
  }
  addHours(hours) {
    this.date.setHours(this.date.getHours() + hours);
    return this;
  }
  addMinutes(minutes) {
    this.date.setMinutes(this.date.getMinutes() + minutes);
    return this;
  }
  addSeconds(seconds) {
    this.date.setSeconds(this.date.getSeconds() + seconds);
    return this;
  }
  /**
   * Add a date interval
   *
   * @param dateInterval
   */
  add(dateInterval) {
    const inv = dateInterval.invert ? -1 : 1;
    this.setYear(this.getYear() + dateInterval.years * inv);
    this.setMonth(this.getMonth() + dateInterval.months * inv);
    this.setDay(this.getDay() + dateInterval.days * inv);
    this.setHours(this.getHours() + dateInterval.hours * inv);
    this.setMinutes(this.getMinutes() + dateInterval.minutes * inv);
    this.setSeconds(this.getSeconds() + dateInterval.seconds * inv);
    return this;
  }
  /**
   * Check if current date is in a leap year
   */
  isLeapYear() {
    const year = this.getYear();
    return !!((year & 3) == 0 && (year % 100 || year % 400 == 0 && year));
  }
  /**
   * Get the number of days in the current month
   */
  getDaysInMonth() {
    const daysInMonth = [31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];
    const m = this.getMonth() - 1;
    return m == 1 && this.isLeapYear() ? 29 : daysInMonth[m];
  }
  getGMTOffset(colon = ":") {
    const tzo = this.getTimezoneOffset();
    return (tzo > 0 ? "-" : "+") + pad2(Math.floor(Math.abs(tzo) / 60)) + colon + pad2(Math.abs(tzo % 60));
  }
  /**
   * Format the date into a string
   *
   * You can use the following characters. You can escape a character with a \ to output it as given.
   *
   * d - The day of the month (from 01 to 31)
   * D - A textual representation of a day (three letters)
   * j - The day of the month without leading zeros (1 to 31)
   * l (lowercase 'L') - A full textual representation of a day
   * N - The ISO-8601 numeric representation of a day (1 for Monday, 7 for Sunday)
   * S - The English ordinal suffix for the day of the month (2 characters st, nd, rd or th. Works well with j)
   * w - A numeric representation of the day (0 for Sunday, 6 for Saturday)
   * z - The day of the year (from 0 through 365)
   * W - The ISO-8601 week number of year (weeks starting on Monday)
   * F - A full textual representation of a month (January through December)
   * m - A numeric representation of a month (from 01 to 12)
   * M - A short textual representation of a month (three letters)
   * n - A numeric representation of a month, without leading zeros (1 to 12)
   * t - The number of days in the given month
   * L - Whether it's a leap year (1 if it is a leap year, 0 otherwise)
   * o - The ISO-8601 year number
   * Y - A four digit representation of a year
   * y - A two digit representation of a year
   * a - Lowercase am or pm
   * A - Uppercase AM or PM
   * B - Swatch Internet time (000 to 999)
   * g - 12-hour format of an hour (1 to 12)
   * G - 24-hour format of an hour (0 to 23)
   * h - 12-hour format of an hour (01 to 12)
   * H - 24-hour format of an hour (00 to 23)
   * i - Minutes with leading zeros (00 to 59)
   * s - Seconds, with leading zeros (00 to 59)
   * u - Microseconds (added in PHP 5.2.2)
   * e - The timezone identifier (Examples: UTC, GMT, Atlantic/Azores)
   * I  (capital i) - Whether the date is in daylights savings time (1 if Daylight Savings Time, 0 otherwise)
   * O - Difference to Greenwich time (GMT) in hours (Example: +0100)
   * P - Difference to Greenwich time (GMT) in hours:minutes (added in PHP 5.1.3)
   * T - Timezone abbreviations (Examples: EST, MDT)
   * Z - Timezone offset in seconds. The offset for timezones west of UTC is negative (-43200 to 50400)
   * c - The ISO-8601 date (e.g. 2013-05-05T16:34:42+00:00)
   * r - The RFC 2822 formatted date (e.g. Fri, 12 Apr 2013 12:01:05 +0200)
   * U - The seconds since the Unix Epoch (January 1 1970 00:00:00 GMT)
   *
   */
  format(format) {
    const chars = format.split("");
    let output = "";
    for (let i = 0, l = chars.length; i < l; i++) {
      let char = chars[i];
      if (char == "\\") {
        i++;
        if (chars.length > i + 1) {
          char += chars[i];
        }
      } else if (char in _DateTime.converters) {
        char = _DateTime.converters[char](this) + "";
      }
      output += char;
    }
    return output;
  }
  static createFormatRegex(format) {
    const chars = format.split("");
    let output = "";
    for (let i = 0, l = chars.length; i < l; i++) {
      let char = chars[i];
      switch (char) {
        case "Y":
          char = "(?<Y>\\d{4})";
          break;
        case "s":
        case "i":
        case "y":
          char = "(?<" + char + ">\\d{2})";
          break;
        case "G":
        case "H":
        case "m":
        case "d":
        case "g":
        case "h":
        case "j":
        case "n":
          char = "(?<" + char + ">\\d{1,2})";
          break;
        case "a":
          char = "(?<a>am|pm)";
          break;
        case "A":
          char = "(?<a>AM|PM)";
          break;
        case "P":
          char = "(?<P>[+-](\\d{2}):(\\d{2}))";
          break;
        case "c":
          char = _DateTime.createFormatRegex("Y-m-dTH:i:sP");
          break;
        default:
          break;
      }
      output += char;
    }
    return output;
  }
  /**
   * Create date by given format. See {@link DateTime.format}.
   * Supports:
   * Y, y, m, n, d, j, H, h, G, g, i, s, a, A
   *
   * @example
   * ```
   * const date = Date.createFromFormat("2021-10-21 21:09", "Y-m-d H:i"));
   *
   * const date = console.log(Date.createFromFormat("10/12/2021 9:09am", "m/d/Y g:ia", "America/New_York));
   * ```
   */
  static createFromFormat(dateStr, format = "c", timezone) {
    const regex = new RegExp(_DateTime.createFormatRegex(format), "u");
    const result = regex.exec(dateStr);
    if (!result) {
      return void 0;
    }
    const date = new _DateTime("1970-01-01");
    date.setHours(0, 0, 0, 0);
    if (timezone) {
      date.timezone = timezone;
    }
    if (result.groups["Y"]) {
      date.setYear(parseInt(result.groups["Y"]));
      delete result.groups["Y"];
    } else if (result.groups["y"]) {
      date.setYear(parseInt(2e3 + result.groups["y"]));
      delete result.groups["y"];
    }
    for (let key of ["n", "m"]) {
      if (result.groups[key]) {
        date.setMonth(parseInt(result.groups[key]));
        delete result.groups[key];
      }
    }
    for (let key in result.groups) {
      switch (key) {
        case "j":
        case "d":
          date.setMonthDay(parseInt(result.groups[key]));
          break;
        case "G":
        case "H":
          date.setHours(parseInt(result.groups[key]));
          break;
        case "h":
        case "g":
          const pm = result.groups.a ? result.groups.a == "pm" : result.groups.A ? result.groups.A == "PM" : false;
          const h = pm ? parseInt(result.groups[key]) : parseInt(result.groups[key]) + 12;
          date.setHours(h);
          break;
        case "i":
          date.setMinutes(parseInt(result.groups[key]));
          break;
        case "s":
          date.setSeconds(parseInt(result.groups[key]));
          break;
      }
    }
    return date;
  }
  /**
   * Compare with given date
   *
   * Returns:
   *
   * - -1 if this interval is smaller than the other
   * - 0 if intervals are equal
   * - 1 if this interval is greater than the other
   *
   * @param date
   * @return number
   */
  compare(date) {
    return (this.date > date.date ? 1 : 0) - (this.date < date.date ? 1 : 0);
  }
};
_DateTime.firstWeekDay = 1;
// 1 = monday, 7 = sunday
_DateTime.dayNames = {};
// {mo: t('Monday'), tu: t('Tuesday'), ...}
_DateTime.dayMap = [];
// ['mo','tu',...] if week starts on monday else index 0 = 'su'
_DateTime.monthNames = [];
_DateTime.cache = {};
_DateTime.converters = {
  "d": (date) => pad2(date.getMonthDay()),
  "D": (date) => _DateTime.dayNames[_DateTime.dayMap[date.getWeekDay()]].substring(0, 3),
  "j": (date) => date.getMonthDay().toString(),
  "l": (date) => _DateTime.dayNames[_DateTime.dayMap[date.getWeekDay()]],
  "S": (date) => ["st", "nd", "rd"][((date.getMonthDay() + 90) % 100 - 10) % 10 - 1] || "th",
  "w": (date) => date.getDay().toString(),
  "z": (date) => date.getDayOfYear().toString(),
  "W": (date) => date.getWeekOfYear().toString(),
  "F": (date) => _DateTime.monthNames[date.getMonth() - 1],
  "m": (date) => pad2(date.getMonth()),
  "M": (date) => _DateTime.monthNames[date.getMonth() - 1].substring(0, 3),
  "n": (date) => date.getMonth().toString(),
  "Y": (date) => date.getYear().toString(),
  "y": (date) => (date.getYear() + "").substr(-2),
  "a": (date) => date.getHours() > 12 ? "pm" : "am",
  "A": (date) => date.getHours() > 12 ? "PM" : "AM",
  "g": (date) => (date.getHours() % 12).toString(),
  "G": (date) => date.getHours().toString(),
  "h": (date) => pad2(date.getHours() % 12),
  "H": (date) => pad2(date.getHours()),
  "i": (date) => pad2(date.getMinutes()),
  "s": (date) => pad2(date.getSeconds()),
  "O": (date) => date.getGMTOffset(""),
  "P": (date) => date.getGMTOffset(),
  "U": (date) => Math.floor(date.getTime() / 1e3).toString(),
  "c": (date) => date.format("Y-m-dTH:i:sP")
};
var DateTime = _DateTime;
DateTime.staticInit(navigator.language);

// goui/script/util/Element.ts
function E(tag, ...items) {
  const el = document.createElement(tag);
  el.append(...items);
  return el;
}
Object.assign(Element.prototype, {
  on(event, listener, options) {
    this.addEventListener(event, listener, options);
    return this;
  },
  un(event, listener, options) {
    this.removeEventListener(event, listener, options);
    return this;
  },
  cls(name, enable) {
    if (!name)
      return this;
    if (Array.isArray(name)) {
      name.map((n) => {
        this.cls(n, enable);
      });
      return this;
    }
    name = name;
    if (enable !== void 0) {
      name = (enable ? "+" : "-") + name;
    }
    switch (name.substring(0, 1)) {
      case "+":
        this.classList.add(name.substring(1));
        break;
      case "-":
        this.classList.remove(name.substring(1));
        break;
      case "!":
        this.classList.toggle(name.substring(1));
        break;
      default:
        this.className = name;
    }
    return this;
  },
  css(style) {
    Object.assign(this.style, style);
    return this;
  },
  attr(name, value) {
    if (value === void 0) {
      return this.getAttribute(name);
    }
    this.setAttribute(name, value);
    return this;
  },
  has(clsOrAttr) {
    if (clsOrAttr.substring(0, 1) === ".") {
      return this.classList.contains(clsOrAttr.substring(1));
    }
    return this.hasAttribute(clsOrAttr);
  },
  isA(tagName) {
    return this.tagName.toLowerCase() === tagName.toLowerCase();
  },
  up(expression, until) {
    if (!expression)
      return this.parentElement;
    let curr = this;
    do {
      if (curr === until)
        return null;
      if (curr.matches(expression))
        return curr;
    } while (curr = curr.parentElement);
  },
  empty() {
    while (this.firstChild) {
      this.removeChild(this.firstChild);
    }
  }
});

// goui/script/util/Format.ts
var _Format = class _Format {
  /**
   * Format a date to a string
   *
   * @param date
   */
  static date(date) {
    if (!date) {
      return "";
    }
    if (!(date instanceof DateTime)) {
      date = new DateTime(date);
    }
    return date.toTimezone(this.timezone).format(_Format.dateFormat);
  }
  /**
   * Format a time to a string
   * @param date
   */
  static time(date) {
    if (!(date instanceof DateTime)) {
      date = new DateTime(date);
    }
    return date.toTimezone(this.timezone).format(_Format.timeFormat);
  }
  /**
   * Formats a date and time to the default format
   *
   * @see Format.dateFormat
   * @see Format.timeFormat
   * @param date
   */
  static dateTime(date) {
    if (!(date instanceof DateTime)) {
      date = new DateTime(date);
    }
    return date.toTimezone(this.timezone).format(_Format.dateFormat + " " + _Format.timeFormat.replace("g", "h").replace("G", "H"));
  }
  /**
   * Format a number to money
   *
   * @param amount
   */
  static money(amount) {
    return this.currency + " " + this.number(amount, 2);
  }
  /**
   * Format a number to a localized string
   *
   * @param value
   * @param decimals
   */
  static number(value, decimals = 2) {
    const neg = value < 0;
    if (neg) {
      value *= -1;
    }
    const dec = this.decimalSeparator, tho = this.thousandsSeparator;
    const no = value.toFixed(decimals), parts = no.split(".");
    let formatted = "";
    const length = parts[0].length;
    for (let i = length - 1, l = 0; i >= l; i--) {
      formatted = parts[0][i] + formatted;
      if (i > 0 && (length - i) % 3 == 0) {
        formatted = tho + formatted;
      }
    }
    if (decimals) {
      formatted += dec + parts[1];
    }
    if (neg) {
      formatted = "-" + formatted;
    }
    return formatted;
  }
  static duration(value, zeroPad = false) {
    if (value < 0) {
      return "";
    }
    let retStr = "";
    const hours = Math.floor(value / 60), minutes = value % 60;
    ;
    if (zeroPad && hours < 10) {
      retStr += "0";
    }
    retStr += hours + ":";
    retStr += minutes < 10 ? "0" + minutes : minutes;
    return retStr;
  }
  static shortTime(v) {
    let arV = v.split(":");
    if (arV.length < 2 || arV.length > 3) {
      return "";
    }
    return arV[0] + ":" + arV[1];
  }
  static minutes(timeStr) {
    const parts = timeStr.split(":");
    return parseInt(parts[0]) * 60 + parseInt(parts[1]);
  }
  // /**
  //  * Pass a javascript template as literals to compile at runtime
  //  *
  //  * @param template
  //  * @param vars
  //  */
  // public static template(template:string, vars: Record<string, any> = {}): string {
  //
  // 	const fn = [
  // 		'const tagged = ( ' + Object.keys(vars).join(', ') + ' ) =>',
  // 		'`' + template + '`',
  // 		'return tagged(...Object.values(vars))'
  // 	].join('\n');
  //
  // 	console.log(fn);
  //
  // 	const handler = new Function('vars', fn)
  //
  // 	return handler(vars);
  // }
};
/**
 * Date format when using date formatting functions
 */
_Format.dateFormat = "d-m-Y";
/**
 * Time format when using time formatting functions
 */
_Format.timeFormat = "G:i";
/**
 * Timezone when using time format functions
 */
_Format.timezone = Intl.DateTimeFormat().resolvedOptions().timeZone.toLowerCase();
/**
 * Currency to use
 */
_Format.currency = "\u20AC";
/**
 * Decimal separator when using number formatting functions
 */
_Format.decimalSeparator = ".";
/**
 * Thousands separator when using number formatting functions
 */
_Format.thousandsSeparator = ",";
/**
 * Escape a HTML string for safely inserting it in the DOM tree
 * @param str
 */
_Format.escapeHTML = function(str) {
  const p2 = document.createElement("p");
  p2.innerText = str;
  return p2.innerHTML;
};
var Format = _Format;

// goui/script/util/ObjectUtil.ts
var _ObjectUtil = class _ObjectUtil {
  /**
   * Simple JSON path function
   *
   * eg.
   * const obj = {
   *   foo : {
   *     bar: {
   *       test: 1
   *     }
   *   }
   * }
   *
   * Object.path(obj, 'foo.bar.test'); // 1
   *
   * @param obj
   * @param path
   * @return The value from the path or undefined if not found
   */
  static path(obj, path) {
    const dotPos = path.indexOf(".");
    const prop = dotPos > -1 ? path.substr(0, dotPos) : path;
    const next = obj[prop];
    if (!next) {
      return next;
    }
    if (dotPos == -1) {
      return next;
    }
    if (!_ObjectUtil.isObject(next)) {
      return void 0;
    }
    return this.path(next, path.substr(dotPos + 1));
  }
  /**
   * Deep merge two key value objects
   * @param o1
   * @param o2
   */
  static merge(o1, o2) {
    if (!this.isObject(o2)) {
      return o2;
    }
    for (let key in o2) {
      if (key in o1 && this.isObject(o1[key])) {
        o1[key] = this.merge(o1[key], o2[key]);
      } else {
        o1[key] = o2[key];
      }
    }
    return o1;
  }
  /**
   * Clone an object
   *
   * @param source
   */
  static clone(source) {
    return structuredClone(source);
  }
  //
  // private static deepClone(source: any, hash = new WeakMap()): any {
  // 	if(Array.isArray(source)) {
  // 		return source.map(item => this.deepClone(item));
  // 	}
  //
  // 	if(source instanceof Date) {
  // 		return new Date(source.getTime());
  // 	}
  //
  // 	if(source && typeof source === 'object') {
  //
  // 		if (hash.has(source)) return hash.get(source);
  //
  //
  // 		const copy = Object.getOwnPropertyNames(source).reduce((o, prop) => {
  // 			Object.defineProperty(o, prop, Object.getOwnPropertyDescriptor(source, prop)!);
  // 			o[prop] = this.deepClone((source as { [key: string]: any })[prop]);
  // 			return o;
  // 		}, Object.create(Object.getPrototypeOf(source)));
  //
  // 		hash.set(source, copy);
  //
  // 		return copy;
  // 	}
  //
  // 	return source;
  // }
};
_ObjectUtil.isObject = (obj) => {
  return Object.prototype.toString.call(obj) === "[object Object]";
};
var ObjectUtil = _ObjectUtil;

// goui/script/util/QuoteStripper.ts
var QuoteStripper = class {
  constructor(body) {
    this.body = body;
  }
  getBodyWithoutQuote() {
    if (this.quote === void 0) {
      this.split();
    }
    return this.bodyWithoutQuote;
  }
  getQuote() {
    if (this.quote === void 0) {
      this.split();
    }
    return this.quote;
  }
  split() {
    const blockQuoteIndex = this.findByBlockQuote(), headerBlockIndex = this.findQuoteByHeaderBlock();
    let quoteIndex = blockQuoteIndex;
    if (blockQuoteIndex < 1 || headerBlockIndex > 0 && headerBlockIndex < blockQuoteIndex) {
      quoteIndex = headerBlockIndex;
    }
    if (quoteIndex > 0) {
      this.bodyWithoutQuote = this.body.substring(0, quoteIndex);
      this.quote = this.body.substring(quoteIndex);
    } else {
      this.bodyWithoutQuote = this.body;
      this.quote = "";
    }
  }
  findByBlockQuote() {
    this.quoteIndex = this.body.indexOf("<blockquote");
    return this.quoteIndex;
  }
  splitLines() {
    if (!this.lines) {
      const br = "|BR|";
      const html2 = this.body.replace(/<\/(p|div|ul|ol|table|dl)>/ig, "$&" + br).replace(/<br[^>]*>/ig, "$&" + br);
      this.lines = html2.split(br);
    }
    return this.lines;
  }
  /**
   * eg
   *
   * Van: Merijn Schering [mailto:mschering@intermesh.nl]
    Verzonden: donderdag 20 november 2014 16:40
    Aan: Someone
    Onderwerp: Subject
   *
   * @return int|boolean
   */
  findQuoteByHeaderBlock() {
    const lines = this.splitLines();
    const greaterThan = /^&gt;(\s|&nbsp;)/;
    const header = /[a-z]+:.*[a-z0-9._\-+&]+@[a-z0-9.\-_]+/i;
    const maybeHeader = /[a-z]+:.*/i;
    let pos = 0, maybePos = 0;
    for (let i = 0, c = lines.length; i < c; i++) {
      const plain = lines[i].replace(/(<([^>]+)>)/ig, "");
      if (plain.match(maybeHeader)) {
        if (!maybePos) {
          maybePos = pos;
        }
      } else {
        maybePos = 0;
      }
      if (plain.match(header)) {
        return maybePos || pos;
      }
      if (plain.match(greaterThan)) {
        return pos;
      }
      pos += lines[i].length;
    }
    return -1;
  }
};

// goui/script/util/Recurrence.ts
var _Recurrence = class _Recurrence {
  constructor(config) {
    this.occurrence = 0;
    this.indices = {
      "BYSECOND": 0,
      "BYMINUTE": 0,
      "BYHOUR": 0,
      "BYDAY": 0,
      "BYMONTH": 0,
      "BYWEEKNO": 0,
      "BYMONTHDAY": 0
    };
    var _a;
    (_a = config.rule).interval || (_a.interval = 1);
    this.rule = config.rule;
    this.dtstart = config.dtstart;
    this.current = new DateTime(+this.dtstart);
    if (config.rule.until) {
      if (config.rule.until.length == 10) {
        this.until = /* @__PURE__ */ new Date(config.rule.until + " 23:59:59");
      } else {
        this.until = new Date(config.rule.until.replace("T", " "));
      }
    }
    this.occurrence++;
    this.validate(this.rule);
    if (config.ff) {
      while (this.current.date < config.ff && this.next()) {
      }
    }
  }
  dayNb(shortName) {
    return { "mo": 1, "tu": 2, "we": 3, "th": 4, "fr": 5, "sa": 6, "su": 0 }[shortName];
  }
  next() {
    let previous = this.last ? this.last.clone() : null;
    if (this.rule.count && this.occurrence >= this.rule.count || this.until && this.current.date > this.until) {
      return null;
    }
    switch (this.rule.frequency) {
      case "daily":
        this.nextDaily();
        break;
      case "weekly":
        this.nextWeekly();
        break;
      case "monthly":
        this.nextMonthly();
        break;
      case "yearly":
        this.nextYearly();
        break;
      default:
        return null;
    }
    if (this.current == previous) {
      throw new Error("Recursion isn't going anywhere");
    }
    this.last = this.current;
    if (this.until && this.current.date > this.until) {
      return null;
    } else {
      this.occurrence++;
      return this.current;
    }
  }
  nDayHas(date) {
    for (const d of this.rule.byDay) {
      if (this.dayNb(d.day) === date.getDay())
        return true;
    }
    return false;
  }
  // private nextHourly() {
  // 	this.current.setHours(this.current.getHours() + this.rule.interval);
  // }
  nextDaily() {
    if (!this.rule.byDay) {
      this.current.addDays(this.rule.interval);
      return;
    }
    do {
      this.current.addDays(this.rule.interval);
    } while (!this.nDayHas(this.current) || !this.rule.byMonth.includes("" + this.current.getMonth()));
  }
  nextWeekly() {
    if (!this.rule.byDay) {
      this.current.addDays(this.rule.interval * 7);
      return;
    }
    do {
      this.current.addDays(1);
      if (_Recurrence.dayMap[this.current.getDay()].toLowerCase() === this.rule.firstDayOfWeek) {
        this.current.addDays(this.rule.interval * 7);
        this.current.setDay(this.dayNb(this.rule.firstDayOfWeek));
      }
    } while (!this.nDayHas(this.current));
  }
  nextMonthly() {
    if (!this.rule.byDay) {
      this.current.addMonths(this.rule.interval);
    } else {
      throw new Error("Not yet supported");
    }
  }
  nextYearly() {
    if (!this.rule.byMonth) {
      this.current.addYears(this.rule.interval);
      return;
    } else {
      throw new Error("Not yet supported");
    }
  }
  validate(p2) {
    if ("byYearDay" in p2 && ("byMonth" in p2 || "byWeek" in p2 || "byMonthDay" in p2 || "byDay" in p2)) {
      throw new Error("Invalid byYearday rule");
    }
    if ("byWeekNo" in p2 && "byMonthDay" in p2) {
      throw new Error("byWeekNo does not fit to byMonthDay");
    }
    if (["daily", "weekly", "monthly", "yearly"].indexOf(p2.frequency) === -1) {
      throw new Error("Invalid frequency rule");
    }
    if (p2.frequency == "monthly" && ("byYearDay" in p2 || "byWeekNo" in p2)) {
      throw new Error("Invalid monthly rule");
    }
    if (p2.frequency == "weekly" && ("byYearDay" in p2 || "byMonthDay" in p2)) {
      throw new Error("Invalid weekly rule");
    }
    if (p2.frequency != "yearly" && "byYearDay" in p2) {
      throw new Error("byYearDay may only appear in yearly rules");
    }
  }
};
_Recurrence.dayMap = ["SU", "MO", "TU", "WE", "TH", "FR", "SA"];
var Recurrence = _Recurrence;

// goui/script/util/BrowserStorage.ts
var Connection = class {
  constructor(dbName = "goui") {
    this.dbName = dbName;
    this.enabled = true;
    this.upgrading = false;
  }
  /**
   * Workaround safari / webkit bug:
   *
   *  https://bugs.webkit.org/show_bug.cgi?id=226547
   * @return {*|Promise}
   */
  idbReady() {
    const isSafari = /Safari\//.test(navigator.userAgent) && !/Chrom(e|ium)\//.test(navigator.userAgent);
    if (!isSafari || !window.indexedDB.databases) {
      return Promise.resolve(this);
    }
    let intervalId;
    return new Promise((resolve) => {
      const tryIdb = () => {
        window.indexedDB.databases().finally(() => resolve(this));
      };
      intervalId = setInterval(tryIdb, 100);
      tryIdb();
    }).finally(() => clearInterval(intervalId));
  }
  /**
   * If upgrade is pending this promise resolves when done.
   */
  upgradeReady() {
    let intervalId;
    return new Promise((resolve) => {
      const checkUpgrading = () => {
        console.log("Check upgrade");
        if (!this.upgrading) {
          resolve(this);
        }
      };
      intervalId = setInterval(checkUpgrading, 100);
      checkUpgrading();
    }).finally(() => clearInterval(intervalId));
  }
  /**
   * Connect to the database
   */
  connect() {
    return __async(this, null, function* () {
      if (this.upgrading) {
        yield this.upgradeReady();
      }
      if (!this.conn) {
        this.conn = yield this.idbReady().then(() => {
          return new Promise((resolve, reject) => {
            const openreq = window.indexedDB.open(this.dbName);
            openreq.onerror = () => {
              this.enabled = false;
              console.warn("Disabling browser storage in indexedDB because browser doesn't support it.");
              reject(openreq.error);
            };
            openreq.onsuccess = () => {
              this.version = openreq.result.version;
              openreq.result.onversionchange = (e) => {
                console.warn("Version change");
                this.conn = void 0;
                openreq.result.close();
              };
              resolve(openreq.result);
            };
            openreq.onblocked = function() {
              console.log("IndexedDB upgrade blocked");
              reject("blocked");
            };
          });
        });
      }
      return this.conn;
    });
  }
  /**
   * Prepare database for upgrade.
   *
   * When resolved it MUST set conn.upgrading to false.
   */
  upgrade() {
    return __async(this, null, function* () {
      this.upgrading = true;
      this.disconnect();
      return new Promise((resolve, reject) => {
        let version = this.version + 1;
        const openreq = window.indexedDB.open(this.dbName, version);
        openreq.onerror = () => {
          this.enabled = false;
          console.warn("Disabling browser storage in indexedDB because browser doesn't support it.");
          reject(openreq.error);
        };
        openreq.onsuccess = () => {
          this.version = openreq.result.version;
          openreq.result.onversionchange = (e) => {
            console.warn("Version change");
            this.conn = void 0;
            openreq.result.close();
          };
          this.conn = openreq.result;
        };
        openreq.onblocked = (e) => {
          console.warn("IndexedDB upgrade blocked.");
        };
        openreq.onupgradeneeded = (e) => {
          resolve(e);
        };
      });
    });
  }
  /**
   * Disconnect from the database
   */
  disconnect() {
    if (this.conn) {
      this.conn.close();
      this.conn = void 0;
    }
  }
  /**
   * Delete the entire database
   */
  deleteDatabase() {
    return __async(this, null, function* () {
      if (!this.enabled) {
        return Promise.resolve(null);
      }
      return new Promise((resolve, reject) => {
        const openreq = indexedDB.deleteDatabase(this.dbName);
        openreq.onerror = () => reject(openreq.error);
        openreq.onsuccess = () => resolve(openreq.result);
      });
    });
  }
};
var browserStoreConnection = new Connection();
var BrowserStore = class {
  constructor(storeName) {
    this.storeName = storeName;
  }
  getStore(mode) {
    return __async(this, null, function* () {
      let db = yield browserStoreConnection.connect();
      if (!db.objectStoreNames.contains(this.storeName)) {
        db = yield this.createStore();
      }
      return db.transaction(this.storeName, mode).objectStore(this.storeName);
    });
  }
  createStore() {
    return __async(this, null, function* () {
      if (browserStoreConnection.upgrading) {
        yield browserStoreConnection.upgradeReady();
        let db = yield browserStoreConnection.connect();
        if (db.objectStoreNames.contains(this.storeName)) {
          return db;
        }
      }
      return new Promise((resolve, reject) => __async(this, null, function* () {
        const e = yield browserStoreConnection.upgrade(), db = e.target.result, t2 = e.target.transaction;
        db.createObjectStore(this.storeName);
        t2.oncomplete = () => {
          browserStoreConnection.upgrading = false;
          resolve(db);
        };
        t2.onerror = (e2) => {
          browserStoreConnection.upgrading = false;
          reject(e2);
        };
      }));
    });
  }
  requestPromise(req) {
    return new Promise((resolve, reject) => {
      req.onerror = (e) => {
        reject(e);
      };
      req.onsuccess = (e) => {
        resolve(req.result);
      };
    });
  }
  /**
   * Get an item from the store
   * returns undefined if not found.
   *
   * @param key
   */
  getItem(key) {
    return __async(this, null, function* () {
      if (!browserStoreConnection.enabled) {
        return Promise.resolve(void 0);
      }
      const store3 = yield this.getStore("readonly"), req = store3.get(key + "");
      return this.requestPromise(req);
    });
  }
  /**
   * Set an item
   *
   * @param key
   * @param value
   */
  setItem(key, value) {
    return __async(this, null, function* () {
      if (!browserStoreConnection.enabled) {
        return Promise.resolve(true);
      }
      const store3 = yield this.getStore("readwrite"), req = store3.put(value, key + "");
      return this.requestPromise(req).then(() => true).catch(() => false);
    });
  }
  /**
   * Remove an item
   *
   * @param key
   */
  removeItem(key) {
    return __async(this, null, function* () {
      if (!browserStoreConnection.enabled) {
        return Promise.resolve(key);
      }
      const store3 = yield this.getStore("readwrite"), req = store3.delete("" + key);
      return this.requestPromise(req).then(() => key);
    });
  }
  /**
   * Clear all data from this store
   */
  clear() {
    return __async(this, null, function* () {
      if (!browserStoreConnection.enabled) {
        return Promise.resolve(null);
      }
      const store3 = yield this.getStore("readwrite"), req = store3.clear();
      return this.requestPromise(req);
    });
  }
  keys() {
    return __async(this, null, function* () {
      if (!browserStoreConnection.enabled) {
        return Promise.resolve([]);
      }
      const keys = [];
      const store3 = yield this.getStore("readonly");
      return new Promise((resolve, reject) => {
        (store3.openKeyCursor || store3.openCursor).call(store3).onsuccess = function() {
          if (!this.result) {
            resolve(keys);
            return;
          }
          keys.push(this.result.key);
          this.result.continue();
        };
      });
    });
  }
};

// goui/script/component/Component.ts
var html = document.querySelector("html");
var REM_UNIT_SIZE = parseFloat(window.getComputedStyle(html).fontSize);
var _Component = class _Component extends Observable {
  /**
   * Component constructor
   *
   * @param tagName The tagname used for the root HTMLElement of this component
   */
  constructor(tagName = "div") {
    super();
    this.tagName = tagName;
    /**
     * A base class not configurable. cls can be used to add extra classes leaving this class alone
     *
     * @protected
     */
    this.baseCls = "";
    /**
     * True when added to the DOM tree
     *
     * @private
     */
    this._rendered = false;
    /**
     * Component item ID that can be used to lookup the Component inside a Component with Component.findItem() and
     * Component.findItemIndex();
     *
     * if stateId is given it will also be used as itemId
     */
    this._itemId = "";
    /**
     * Set arbitrary data on a component.
     *
     * Should be used with caution as this data is not typed.
     */
    this.dataSet = {};
  }
  /**
   * Component item ID that can be used to lookup the Component inside a Component with Component.findItem() and
   * Component.findItemIndex();
   *
   * if stateId is given it will also be used as itemId
   */
  get itemId() {
    return this._itemId || this.stateId || this.el.id || "";
  }
  set itemId(itemId) {
    this._itemId = itemId;
  }
  initItems() {
    this.items.on("add", (_collection, item, index) => {
      item.parent = this;
      item.fire("added", item, index, this);
      if (this.rendered) {
        this.renderItem(item);
      }
    });
    this.items.on("remove", (_collection, item) => {
      if (item.parent) {
        item.parent = void 0;
        item.remove();
      }
    });
  }
  getState() {
    return State.get().getItem(this.stateId);
  }
  hasState() {
    return this.stateId && State.get().hasItem(this.stateId);
  }
  /**
   * Restore state of the component in this function. It's called before render in init().
   *
   * @see saveState();
   * @param state
   *
   * @protected
   */
  restoreState(state) {
  }
  /**
   * Call save start when something relevant to the state changes.
   * Implement buildState() to save relevant state properties and restore it in restoreState()
   *
   * stateId must be set on components to be stateful
   *
   * @protected
   */
  saveState() {
    if (this.stateId) {
      State.get().setItem(this.stateId, this.buildState());
    }
  }
  /**
   * Build state for the component
   *
   * @see saveState();
   * @protected
   */
  buildState() {
    return {};
  }
  /**
   * Title of the dom element
   */
  set title(title) {
    this.el.title = title;
  }
  get title() {
    return this.el.title;
  }
  /**
   * Check if the component has been rendered and added to the DOM tree
   */
  get rendered() {
    return this._rendered;
  }
  get el() {
    if (!this._el) {
      this._el = document.createElement(this.tagName);
    }
    return this._el;
  }
  /**
   * Class name to add to element
   *
   * Some common classes to add for layout:
   *
   * - hbox: Set's flex layout to horizontal boxes. Use flex: n to stretch columns
   * - vbox: As above but vertical
   * - fit: Fit the parent's size
   * - scroll: Set's autoscroll: true
   * - pad: Set common padding on the element
   * - border-(top|bottom|left|right) to add a border
   *
   * Other:
   *
   * - goui-fade-in: The component will fade in when show() is used.
   * - goui-fade-out: The component will fade out when hide is used.
   *
   */
  set cls(cls) {
    if (this._cls) {
      this._cls.split(/\s+/).forEach((cls2) => {
        this.el.classList.remove(cls2);
      });
    }
    this._cls = cls;
    this.initClassName();
    this.el.className += " " + cls;
  }
  get cls() {
    var _a;
    return (_a = this._cls) != null ? _a : "";
  }
  initClassName() {
    if (this.el.classList.contains("goui")) {
      return;
    }
    this.el.classList.add("goui");
    if (this.baseCls) {
      this.el.className += " " + this.baseCls;
    }
  }
  /**
   * Renders the component and it's children
   */
  internalRender() {
    this.initClassName();
    this.renderItems();
    if (this.stateId) {
      this.restoreState(this.getState());
    }
    return this.el;
  }
  /**
   * The tabindex attribute specifies the tab order of an element (when the "tab" button is used for navigating).
   */
  set tabIndex(tabIndex) {
    this.el.tabIndex = tabIndex;
  }
  get tabIndex() {
    return this.el.tabIndex;
  }
  /**
   * CSS flex value
   */
  set flex(flex) {
    this.el.style.flex = flex + "";
  }
  /**
   * CSS flex value
   */
  get flex() {
    return this.el.style.flex;
  }
  /**
   * Make it resizable
   */
  set resizable(resizable) {
    if (resizable) {
      this.el.classList.add("goui-resizable");
    } else {
      this.el.classList.remove("goui-resizable");
    }
  }
  get resizable() {
    return this.el.classList.contains("goui-resizable");
  }
  /**
   * Render the component
   *
   * @param parentEl The element this componennt will render into
   * @param insertBefore If given, the element will be inserted before this child
   */
  render(parentEl, insertBefore) {
    if (this._rendered) {
      throw new Error("Already rendered");
    }
    this.fire("beforerender", this);
    if (!parentEl) {
      if (this.renderTo) {
        parentEl = this.renderTo;
      } else {
        if (!this.parent) {
          throw new Error("No parent set for " + typeof this);
        }
        parentEl = this.parent.itemContainerEl;
        insertBefore = this.getInsertBefore();
      }
    }
    if (!insertBefore) {
      parentEl.appendChild(this.el);
    } else {
      parentEl.insertBefore(this.el, insertBefore);
    }
    this.internalRender();
    this._rendered = true;
    this.fire("render", this);
    return this.el;
  }
  /**
   * Finds the DOM node in the parent's children to insert before when rendering a child component
   *
   * @protected
   */
  getInsertBefore() {
    if (!this.parent.rendered) {
      return void 0;
    }
    const index = this.parent.items.indexOf(this);
    let refItem = void 0;
    for (let i = index + 1, l = this.parent.items.count(); i < l; i++) {
      if (this.parent.items.get(i).rendered) {
        refItem = this.parent.items.get(i).el;
        break;
      }
    }
    return refItem;
  }
  /**
   * Remove component from the component tree
   */
  remove() {
    if (!this.fire("beforeremove", this)) {
      return false;
    }
    this.internalRemove();
    this.fire("remove", this);
    return true;
  }
  internalRemove() {
    this.items.clear();
    if (this.parent) {
      const p2 = this.parent;
      this.parent = void 0;
      p2.items.remove(this);
    }
    if (this.el) {
      this.el.remove();
    }
  }
  /**
   * Hide element
   */
  set hidden(hidden) {
    if (this.el.hidden == hidden) {
      return;
    }
    const eventName = hidden ? "hide" : "show";
    if (this.fire("before" + eventName, this) === false) {
      return;
    }
    this.internalSetHidden(hidden);
    this.fire(eventName, this);
  }
  /**
   * Overridable method so that the (before)show/hide event fire before and after.
   *
   * @param hidden
   * @protected
   */
  internalSetHidden(hidden) {
    this.el.hidden = hidden;
  }
  get hidden() {
    return this.el.hidden;
  }
  /**
   * Hide this component
   *
   * This sets the "hidden" attribute on the DOM element which set's CSS display:none.
   * You can change this to fade with css class "goui-fade-in" and "goui-fade-out"
   *
   * If you want to override this function, please override internalSetHidden instead so the beforeshow and show event
   * fire at the right time.
   */
  hide() {
    this.hidden = true;
    return this.hidden;
  }
  /**
   * Show the component
   *
   * If you want to override this function, please override internalSetHidden instead so the beforeshow and show event
   * fire at the right time.
   */
  show() {
    this.hidden = false;
    return !this.hidden;
  }
  /**
   * Disable component
   */
  set disabled(disabled) {
    if (!disabled) {
      this.el.removeAttribute("disabled");
    } else {
      this.el.setAttribute("disabled", "");
    }
  }
  get disabled() {
    return this.el.hasAttribute("disabled");
  }
  /**
   * Set the HTML contents of the component (innerHTML)
   */
  set html(html2) {
    this.el.innerHTML = html2;
  }
  get html() {
    return this.el.innerHTML;
  }
  /**
   * Set the innerText
   */
  set text(text) {
    this.el.innerText = text;
  }
  get text() {
    return this.el.innerText;
  }
  /**
   * Set the width in scalable pixels
   *
   * The width is applied in rem units divided by 10. Because the font-size of the html
   * element has a font-size of 62.5% this is equals the amount of pixels, but it can be
   * scaled easily for different themes.
   *
   */
  set width(width) {
    this.el.style.width = width / 10 + "rem";
  }
  get width() {
    const px = this.el.offsetWidth;
    if (px) {
      return px / REM_UNIT_SIZE * 10;
    }
    const styleWidth = this.el.style.width;
    if (styleWidth.substring(styleWidth.length - 3) == "rem") {
      return parseFloat(styleWidth);
    } else if (styleWidth.substring(styleWidth.length - 2) == "px") {
      return parseFloat(styleWidth) / REM_UNIT_SIZE * 10;
    }
    return 0;
  }
  /**
   * Set inline style
   */
  set style(style) {
    Object.assign(this.el.style, style);
  }
  get style() {
    return this.el.style;
  }
  computeZIndex() {
    const z = parseInt(window.getComputedStyle(this.el).getPropertyValue("z-index"));
    if (!z) {
      return this.parent ? this.parent.computeZIndex() : 0;
    } else {
      return z;
    }
  }
  /**
   * The height in scalable pixels
   *
   * @see width
   */
  set height(height) {
    this.el.style.height = height / 10 + "rem";
  }
  get height() {
    const px = this.el.offsetHeight;
    if (px) {
      return px / REM_UNIT_SIZE * 10;
    }
    const styleHeight = this.el.style.height;
    if (styleHeight.substring(styleHeight.length - 3) == "rem") {
      return parseFloat(styleHeight);
    } else if (styleHeight.substring(styleHeight.length - 2) == "px") {
      return parseFloat(styleHeight) / REM_UNIT_SIZE * 10;
    }
    return 0;
  }
  /**
   * Element ID
   */
  set id(id) {
    this.el.id = id;
  }
  get id() {
    return this.el.id;
  }
  /**
   * Focus the component
   *
   * @param o
   */
  focus(o) {
    this.el.focus(o);
    this.fire("focus", this, o);
  }
  isFocusable() {
    return this.el && !this.hidden && (this.el.tagName == "BUTTON" || this.el.tagName == "INPUT" || this.el.tagName == "A" || this.el.tagName == "AREA" || this.el.tabIndex > -1);
  }
  /**
   * Get the component that's next to this one
   */
  nextSibling() {
    if (!this.parent) {
      return void 0;
    }
    const index = this.parent.items.indexOf(this);
    if (index == -1) {
      return void 0;
    }
    return this.parent.items.get(index + 1);
  }
  /**
   * Get the component that's previous to this one
   */
  previousSibling() {
    if (!this.parent) {
      return void 0;
    }
    const index = this.parent.items.indexOf(this);
    if (index < 1) {
      return void 0;
    }
    return this.parent.items.get(index - 1);
  }
  /**
   * Find ancestor
   *
   * The method traverses the Component's ancestors (heading toward the document root) until it finds
   * one where the given function returns true.
   *
   * @param fn When the function returns true the item will be returned. Otherwise it will move up to the next parent.
   */
  findAncestor(fn) {
    let p2 = this.parent;
    while (p2 != void 0) {
      if (fn(p2)) {
        return p2;
      } else {
        p2 = p2.parent;
      }
    }
    return void 0;
  }
  /**
   * Find parent by instance type of the parent
   *
   * @example
   * ```
   * const form = textField.findAncestorByType(Form);
   * ```
   * @param cls
   */
  findAncestorByType(cls) {
    const p2 = this.findAncestor((cmp) => cmp instanceof cls);
    if (p2) {
      return p2;
    } else {
      return void 0;
    }
  }
  /**
   * The child components of this component
   */
  get items() {
    if (!this._items) {
      this._items = new Collection();
      this.initItems();
    }
    return this._items;
  }
  renderItems() {
    if (this._items) {
      this._items.forEach((item) => {
        this.renderItem(item);
      });
    }
  }
  get itemContainerEl() {
    return this.el;
  }
  /**
   * Can be overriden to wrap the component
   *
   * @param item
   * @protected
   */
  renderItem(item) {
    item.render(this.itemContainerEl, item.getInsertBefore ? item.getInsertBefore() : void 0);
  }
  /**
   * Find the item by element ID, itemId property, Component instance or custom function
   */
  findItemIndex(predicate) {
    let fn = this.createFindPredicateFunction(predicate);
    return this.items.findIndex(fn);
  }
  /**
   * Find the item by element ID, itemId property, Component instance or custom function.
   *
   * If you want to search the component tree hierarchy use {@see findChild()}
   *
   */
  findItem(predicate) {
    let fn = this.createFindPredicateFunction(predicate);
    return this.items.find(fn);
  }
  /**
   * Cascade down the component hierarchy
   *
   * @param fn When the function returns false then the cascading will be stopped. The current Component will be finished!
   */
  cascade(fn) {
    if (fn(this) === false) {
      return this;
    }
    if (this.items) {
      for (let cmp of this.items) {
        cmp.cascade && cmp.cascade(fn);
      }
    }
    return this;
  }
  createFindPredicateFunction(predicate) {
    if (predicate instanceof Function) {
      return predicate;
    } else {
      return (item) => {
        return item === predicate || item.itemId === predicate || item.id === predicate;
      };
    }
  }
  /**
   * Find a child at any level by element ID, itemId property, Component instance or custom function.
   *
   * It cascades down the component hierarchy. See also {@see findChildByType}
   *
   */
  findChild(predicate) {
    let fn = this.createFindPredicateFunction(predicate);
    let child;
    this.cascade((item) => {
      if (fn(item)) {
        child = item;
        return false;
      }
    });
    return child;
  }
  /**
   * Find children at any level by element ID, itemId property, Component instance or custom function.
   *
   * It cascades down the component hierarchy. See also {@see findChildByType}
   *
   */
  findChildren(predicate) {
    let fn = this.createFindPredicateFunction(predicate);
    const children = [];
    this.cascade((item) => {
      if (fn(item)) {
        children.push(item);
      }
    });
    return children;
  }
  /**
   * Find child by instance type of the parent
   *
   * @example
   * ```
   * const form = textField.findAncestorByType(Form);
   * ```
   * @param cls
   */
  findChildByType(cls) {
    const p2 = this.findChild((cmp) => cmp instanceof cls);
    if (p2) {
      return p2;
    } else {
      return void 0;
    }
  }
  /**
   * Find children by instance type of the parent
   *
   * @example
   * ```
   * const form = textField.findAncestorByType(Form);
   * ```
   * @param cls
   */
  findChildrenByType(cls) {
    return this.findChildren((Component18) => Component18 instanceof cls);
  }
  /**
   * Set attributes of the DOM element
   *
   * @param attr
   */
  set attr(attr) {
    for (let name in attr) {
      this.el.setAttribute(name, attr[name]);
    }
  }
  /**
   * Mask the component to disable user interaction
   * It creates an absolute positioned Mask
   * component. This component should have a non-static position style for this to work.
   */
  mask(delay = 300) {
    if (this.maskTimeout || this._mask && this._mask.hidden == false) {
      return;
    }
    this.maskTimeout = setTimeout(() => {
      if (!this._mask) {
        this._mask = mask({ spinner: true });
        this.items.add(this._mask);
      }
      this.el.classList.add("masked");
      this._mask.show();
      this.maskTimeout = void 0;
    }, delay);
  }
  /**
   * Unmask the body
   */
  unmask() {
    if (this.maskTimeout) {
      clearTimeout(this.maskTimeout);
      this.maskTimeout = void 0;
    }
    if (this._mask) {
      this._mask.hide();
    }
    this.el.classList.remove("masked");
  }
  static uniqueID() {
    return "goui-" + ++_Component._uniqueID;
  }
  /**
   * Print this component. Everything else will be left out.
   */
  print() {
    let paper = document.getElementById("paper");
    if (!paper) {
      paper = document.createElement("div");
      paper.id = "paper";
      document.body.appendChild(paper);
    }
    const style = window.getComputedStyle(this.el);
    paper.style.cssText = style.cssText;
    const size = this.el.getBoundingClientRect();
    paper.style.width = size.width + "px";
    paper.innerHTML = this.el.innerHTML;
    const oldTitle = document.title;
    if (this.title) {
      document.title = this.title.replace(":", ".").replace(/[/\\?%*|"<>]+/g, "-");
      ;
    }
    if (!browser.isFirefox()) {
      Promise.all(Array.from(document.images).filter((img2) => !img2.complete).map((img2) => new Promise((resolve) => {
        img2.onload = img2.onerror = resolve;
      }))).then(() => {
        browser.isSafari() ? document.execCommand("print") : window.print();
      });
    } else {
      window.print();
    }
    window.addEventListener("afterprint", () => {
      if (oldTitle) {
        document.title = oldTitle;
      }
      if (paper) {
        paper.innerHTML = "";
      }
    }, { once: true });
  }
};
_Component._uniqueID = 0;
var Component = _Component;
var Mask = class extends Component {
  constructor() {
    super(...arguments);
    this.baseCls = "goui-mask";
  }
  /**
   * Show loading spinner
   */
  set spinner(spinner) {
    if (spinner) {
      this.html = '<div class="spinner"><div></div><div></div><div></div><div></div><div></div><div></div><div></div><div></div></div>';
    }
  }
};
var mask = (config) => createComponent(new Mask(), config);
var comp = (config, ...items) => createComponent(new Component(config == null ? void 0 : config.tagName), config, items);
var p = (config, ...items) => createComponent(new Component("p"), typeof config == "string" ? { html: config } : config, items);
var small = (config, ...items) => createComponent(new Component("small"), typeof config == "string" ? { html: config } : config, items);
var h1 = (config, ...items) => createComponent(new Component("h1"), typeof config == "string" ? { html: config } : config, items);
var h2 = (config, ...items) => createComponent(new Component("h2"), typeof config == "string" ? { html: config } : config, items);
var h3 = (config, ...items) => createComponent(new Component("h3"), typeof config == "string" ? { html: config } : config, items);
var h4 = (config, ...items) => createComponent(new Component("h4"), typeof config == "string" ? { html: config } : config, items);
var code = (config, ...items) => createComponent(new Component("code"), typeof config == "string" ? { html: config } : config, items);
var section = (config, ...items) => createComponent(new Component("section"), typeof config == "string" ? { html: config } : config, items);
var hr = (config) => createComponent(new Component("hr"), config);
var img = (config) => {
  var _a;
  const img2 = createComponent(new Component("img"), config);
  img2.attr = {
    src: config.src,
    alt: (_a = config.alt) != null ? _a : ""
  };
  return img2;
};
var a = (config, ...items) => {
  var _a, _b;
  const a2 = createComponent(new Component("a"), config, items);
  a2.attr = {
    href: (_a = config.href) != null ? _a : "",
    target: (_b = config.target) != null ? _b : ""
  };
  return a2;
};
var progress = (config) => createComponent(new Component("progress"), config);
var createComponent = (comp2, config, items) => {
  if (config) {
    if (config.listeners) {
      for (let key in config.listeners) {
        const eventName = key;
        if (typeof config.listeners[eventName] == "function") {
          comp2.on(eventName, config.listeners[eventName]);
        } else {
          const o = config.listeners[eventName];
          const fn = o.fn;
          delete o.fn;
          comp2.on(eventName, fn, o);
        }
      }
      delete config.listeners;
    }
    Object.assign(comp2, config);
  }
  if (items && items.length) {
    comp2.items.add(...items);
  }
  return comp2;
};

// goui/script/component/Toolbar.ts
var Toolbar = class _Toolbar extends Component {
  constructor() {
    super("menu");
    this.baseCls = "goui-toolbar";
    /**
     * Used by keyboard nav
     * @protected
     */
    this.orientation = "horizontal";
    this.focusedItemIndex = -1;
    this.setupKeyboardNav();
  }
  /**
   * Find the first menu in the tree of submenu's
   */
  findToolbar() {
    let parent = this.parent;
    if (!parent || !(parent instanceof _Toolbar)) {
      return void 0;
    }
    if (parent.orientation == "horizontal") {
      return parent;
    } else {
      return parent.findToolbar();
    }
  }
  setupKeyboardNav() {
    this.items.on("add", (collection, item, index) => {
      item.el.addEventListener("click", () => {
        this.focusedItemIndex = index;
      });
    });
    this.on("focus", () => {
      if (this.focusedItemIndex > -1) {
        this.items.get(this.focusedItemIndex).focus();
      }
    });
    this.on("hide", () => {
      this.focusedItemIndex = -1;
    });
    this.el.addEventListener("keydown", (ev) => {
      switch (ev.key) {
        case "ArrowRight":
          if (this.orientation == "vertical") {
            if (!this.focusChild()) {
              const tb = this.findToolbar();
              if (tb) {
                tb.focusNext();
              }
            }
          } else {
            this.focusNext();
          }
          ev.stopPropagation();
          ev.preventDefault();
          break;
        case "ArrowDown":
          if (this.orientation == "vertical") {
            this.focusNext();
          } else {
            this.focusChild();
          }
          ev.stopPropagation();
          ev.preventDefault();
          break;
        case "ArrowLeft":
          if (this.orientation == "vertical") {
            if (!this.focusParent()) {
              const tb = this.findToolbar();
              if (tb) {
                tb.focusNext(-1);
              }
            }
          } else {
            this.focusNext(-1);
          }
          ev.stopPropagation();
          ev.preventDefault();
          break;
        case "ArrowUp":
          if (this.orientation == "vertical") {
            this.focusNext(-1);
          } else {
            this.focusParent();
          }
          ev.stopPropagation();
          ev.preventDefault();
          break;
      }
    });
  }
  focusNext(inc = 1) {
    const nextIndex = this.focusedItemIndex + inc;
    this.focusedItemIndex = Math.min(Math.max(nextIndex, 0), this.items.count() - 1);
    if (nextIndex != this.focusedItemIndex) {
      return false;
    }
    const cmp = this.items.get(this.focusedItemIndex);
    if (!cmp.isFocusable()) {
      return this.focusNext(inc);
    } else {
      cmp.focus();
      if (this.orientation == "horizontal") {
        cmp.el.click();
      }
      return true;
    }
  }
  focusChild() {
    const child = this.items.get(this.focusedItemIndex);
    if (!child || !child.menu) {
      return false;
    }
    child.menu.focusNext();
    return true;
  }
  focusParent() {
    const child = this.items.get(this.focusedItemIndex);
    if (!child) {
      return false;
    }
    if (child.parent) {
      child.parent.focus();
    }
    return true;
  }
};
var tbar = (config, ...items) => {
  const c = new Toolbar();
  if (config) {
    Object.assign(c, config);
  }
  c.items.add(...tbarItems(items));
  return c;
};
var tbarItems = (items) => {
  const l = items.length;
  if (l) {
    for (let i = 0; i < l; i++) {
      switch (items[i]) {
        case "->":
          items[i] = comp({
            flex: 1
          });
          break;
        case "-":
          items[i] = comp({ tagName: "hr" });
          break;
      }
    }
  }
  return items;
};

// goui/script/Translate.ts
var Translate = class {
  constructor() {
    this.lang = {};
    this.missing = {};
    this.defaultPkg = "core";
    this.defaultModule = "core";
  }
  setDefaultModule(pkg, module) {
    this.defaultModule = module;
    this.defaultPkg = pkg;
  }
  load(lang, pkg = "core", module = "core") {
    if (!this.lang[pkg]) {
      this.lang[pkg] = {};
    }
    if (!this.lang[pkg][module]) {
      this.lang[pkg][module] = {};
    }
    ObjectUtil.merge(this.lang[pkg][module], lang);
    this.setDefaultModule(pkg, module);
  }
  /**
   * Translate a string
   *
   * @param key Translate key. Usually the english text.
   * @param pkg The module package
   * @param module The module name
   */
  static t(key, pkg, module) {
    var _a, _b, _c;
    if (!pkg) {
      pkg = translate.defaultPkg;
    }
    if (!module) {
      module = translate.defaultModule;
    }
    if ((_c = (_b = (_a = translate.lang) == null ? void 0 : _a[pkg]) == null ? void 0 : _b[module]) == null ? void 0 : _c[key]) {
      return translate.lang[pkg][module][key];
    } else if (pkg == "core" && module == "core") {
      if (!translate.missing[pkg]) {
        translate.missing[pkg] = {};
      }
      if (!translate.missing[pkg][module]) {
        translate.missing[pkg][module] = {};
      }
      translate.missing[pkg][module][key] = key;
      return key;
    } else {
      return t(key, "core", "core");
    }
  }
};
var translate = new Translate();
var t = Translate.t;

// goui/script/component/form/Field.ts
var Field = class extends Component {
  constructor(tagName = "label") {
    super(tagName);
    this.isFormField = true;
    /**
     * Adds standard style. You may want to remove this if you don't want the standard
     * look of a form field.
     *
     * @protected
     */
    this.baseCls = "goui-form-field";
    this._required = false;
    this._readOnly = false;
    this._label = "";
    this.invalidMsg = "";
    this._hint = "";
    /**
     * Validate the field on blur
     */
    this.validateOnBlur = true;
    /**
     * Fires a change event if the field's value is different since it got focus
     * @protected
     */
    this.fireChangeOnBlur = true;
    this.onAdded = (comp2, index, parent) => {
      this.trackReset();
      this.defaultValue = this.value;
    };
    this.on("added", this.onAdded, { once: true });
    this.control = this.createControl();
  }
  onFocusOut(e) {
    if (e.relatedTarget instanceof HTMLElement && this.el.contains(e.relatedTarget)) {
      return;
    }
    if (this.validateOnBlur) {
      this.validate();
    }
    if (this.fireChangeOnBlur && this.isChangedSinceFocus()) {
      this.fireChange();
    }
  }
  /**
   * Return true if the field was modified
   */
  isChangedSinceFocus() {
    const v = this.value;
    if (typeof v == "object") {
      return JSON.stringify(this.valueOnFocus) != JSON.stringify(v);
    } else {
      return this.valueOnFocus != v;
    }
  }
  onFocusIn(e) {
    if (e.relatedTarget instanceof HTMLElement && this.el.contains(e.relatedTarget)) {
      return;
    }
    if (this.fireChangeOnBlur) {
      this.captureValueForChange();
    }
  }
  captureValueForChange() {
    const v = this.value;
    this.valueOnFocus = typeof v == "object" ? structuredClone(v) : v;
  }
  internalRender() {
    const el = super.internalRender();
    this.renderControl();
    this.el.addEventListener("focusin", this.onFocusIn.bind(this));
    this.el.addEventListener("focusout", this.onFocusOut.bind(this));
    return el;
  }
  isFocusable() {
    return !this.hidden;
  }
  // get el(): HTMLElement {
  // 	const el = super.el;
  //
  // 	// if(!this.wrap) {
  // 	// 	el.append(this.wrap = E("div").cls('+wrap'));
  // 	// }
  // 	return el;
  // }
  /**
   * A wrapper DIV element that contains input and toolbar for input buttons like an expand button for a drop down
   */
  get wrap() {
    if (!this._wrap) {
      this.el.append(this._wrap = E("div").cls("+wrap"));
    }
    return this._wrap;
  }
  renderControl() {
    const label = this.createLabel();
    if (label) {
      this.wrap.append(label);
    }
    if (this.control) {
      this.wrap.append(this.control.cls("+control"));
      if (this.title) {
        this.control.title = this.title;
      }
    }
    this.renderButtons();
    const hint = this.createHint();
    if (hint) {
      this.el.appendChild(hint);
    }
  }
  renderButtons() {
    if (this._buttons) {
      this.toolbar = tbar({}, ...this._buttons);
      this.toolbar.parent = this;
      this.toolbar.render(this.wrap);
      this._buttons.forEach((btn2) => {
        if (btn2.menu) {
          this.setupMenu(btn2.menu);
        }
      });
    } else {
      if (this.toolbar) {
        this.toolbar.remove();
      }
    }
  }
  /**
   * When buttons with menus are added it is handy to delay the validation on blur.
   * Because when the user will click something in the menu it will blur the field and you probably don't
   * want it to validate at that point. It's important that the menu will return focus to the field
   * and sets the value afterward.
   *
   * @param menu
   * @protected
   */
  setupMenu(menu2) {
    let origValidateOnBlur = false;
    menu2.on("beforeshow", () => {
      origValidateOnBlur = this.validateOnBlur;
      this.validateOnBlur = false;
    });
    menu2.on("hide", () => {
      this.validateOnBlur = origValidateOnBlur;
      this.focus();
    });
  }
  createControl() {
    return void 0;
  }
  /**
   * Render buttons inside the text field
   *
   * @example
   * ```
   * buttons: [
   * 				 		btn({icon: "clear", handler:(btn) => (btn.parent!.parent! as Field).value = ""})
   * 					]
   * ```
   * @param buttons
   */
  set buttons(buttons) {
    this._buttons = buttons;
    if (this.rendered) {
      this.renderButtons();
    }
  }
  get buttons() {
    return this._buttons;
  }
  createHint() {
    this.hintEl = E("div", this._hint).cls("hint");
    return this.hintEl;
  }
  createLabel() {
    this._labelEl = E("div", this.getLabelText()).cls("label");
    return this._labelEl;
  }
  getLabelText() {
    if (!this._label) {
      return "";
    }
    let labelText = this._label;
    if (this._required) {
      labelText += " *";
    }
    return labelText;
  }
  /**
   * Form element name which will be the key in values
   * If omitted the field won't be included in the form values.
   */
  get name() {
    return this._name || "";
  }
  /**
   * The field's name
   */
  set name(name) {
    this._name = name;
  }
  get required() {
    return this._required;
  }
  /**
   * Required or not
   */
  set required(required) {
    this._required = required;
    if (this._labelEl) {
      this._labelEl.innerHTML = this.getLabelText();
    }
    if (this.rendered) {
      this.clearInvalid();
    }
  }
  get label() {
    return this._label + "";
  }
  /**
   * The field's label
   */
  set label(label) {
    this._label = label;
    if (this._labelEl) {
      this._labelEl.innerHTML = this.getLabelText();
    }
  }
  get icon() {
    return this._icon;
  }
  /**
   * The field's icon rendered at the left inside the field
   */
  set icon(icon) {
    this._icon = icon;
    this.createIcon();
  }
  get hint() {
    return this._hint + "";
  }
  /**
   * The field's hint text
   */
  set hint(hint) {
    this._hint = hint;
    if (this.rendered) {
      this.applyInvalidMsg();
    }
  }
  get readOnly() {
    return this._readOnly;
  }
  /**
   * Make the field read only
   */
  set readOnly(readOnly) {
    this.el.classList.toggle("readonly", readOnly);
    this._readOnly = readOnly;
  }
  /**
   * Check if the field was modified since create or when a form was loaded and @see trackReset() was called.
   */
  isModified() {
    return JSON.stringify(this.resetValue) !== JSON.stringify(this.value);
  }
  /**
   * Copies the current value to the reset value. Typically happens when this component was added to a parent and
   * when the form it belongs too loads.
   *
   * @see Form in the trackModifications method
   */
  trackReset() {
    this.resetValue = structuredClone(this.value);
  }
  /**
   * Set the field value
   */
  set value(v) {
    const old = this._value;
    this._value = v;
    this.internalSetValue(v);
    this.checkHasValue();
    this.fire("setvalue", this, this._value, old);
  }
  checkHasValue() {
    this.el.classList.toggle("has-value", !this.isEmpty());
  }
  /**
   * Applies set value to the control.
   *
   * This is also called when the control is rendered. Note that this.rendered is still false when that happens.
   *
   * @param v
   * @protected
   */
  internalSetValue(v) {
  }
  /**
   * Helper to fire "change" event. Child classes must implement this.
   */
  fireChange() {
    const v = this.value;
    this.fire("setvalue", this, v, this.valueOnFocus);
    this.fire("change", this, v, this.valueOnFocus);
    this.valueOnFocus = void 0;
    this.checkHasValue();
  }
  get value() {
    return this._value;
  }
  /**
   * Reset the field value to the value that was given to the field's constructor
   * @see setValue()
   */
  reset() {
    const old = this.value;
    this.value = this.resetValue;
    this.clearInvalid();
    this.fire("reset", this, this.resetValue, old);
    this.fire("setvalue", this, this.resetValue, old);
    this.fire("change", this, this.resetValue, old);
  }
  /**
   * Set the field as invalid and set a message
   *
   * @param msg
   */
  setInvalid(msg) {
    this.invalidMsg = msg;
    if (this.rendered) {
      this.applyInvalidMsg();
    }
    this.fire("invalid", this);
  }
  /**
   * Check if the field is empty
   */
  isEmpty() {
    const v = this.value;
    return v === void 0 || v === null || v === "";
  }
  validate() {
    this.clearInvalid();
    if (this._required && this.isEmpty()) {
      this.setInvalid(t("This field is required"));
    }
    this.fire("validate", this);
  }
  /*
  
  			badInput
  :
  false
  customError
  :
  false
  patternMismatch
  :
  false
  rangeOverflow
  :
  false
  rangeUnderflow
  :
  false
  stepMismatch
  :
  false
  tooLong
  :
  false
  tooShort
  :
  false
  typeMismatch
  :
  false
  
  			 */
  setValidityState(input) {
    const validity = input.validity;
    if (validity.valid) {
      return;
    }
    if (validity.typeMismatch) {
      switch (input.type) {
        case "email":
          this.setInvalid(t("Please enter a valid e-mail address"));
          return;
        default:
          this.setInvalid(t("Please enter a valid value"));
          return;
      }
    } else if (validity.valueMissing) {
      this.setInvalid(t("This field is required"));
    } else if (validity.tooLong) {
      this.setInvalid(t("The maximum length for this field is {max}").replace("{max}", input.maxLength));
    } else if (validity.tooShort) {
      this.setInvalid(t("The minimum length for this field is {max}").replace("{max}", input.minLength));
    } else if (validity.patternMismatch) {
      this.setInvalid(t("Please match the format requested").replace("{max}", input.minLength));
    } else {
      console.warn("TODO: Implement translated message");
      this.setInvalid(input.validationMessage);
    }
  }
  applyInvalidMsg() {
    if (this.invalidMsg) {
      this.el.classList.add("invalid");
      if (this.hintEl)
        this.hintEl.innerHTML = this.invalidMsg;
    } else {
      this.el.classList.remove("invalid");
      if (this.hintEl)
        this.hintEl.innerHTML = this._hint || "";
    }
  }
  /**
   * Clears the invalid state
   */
  clearInvalid() {
    if (!this.invalidMsg) {
      return;
    }
    this.invalidMsg = "";
    if (this.rendered) {
      this.applyInvalidMsg();
    }
  }
  /**
   * Checks if the field is valid
   */
  isValid() {
    if (this.invalidMsg != "") {
      return false;
    }
    this.validate();
    if (this.invalidMsg != "") {
      console.warn("Field '" + this.name + "' is invalid: " + this.invalidMsg, this);
    }
    return this.invalidMsg == "";
  }
  createIcon() {
    if (this._icon) {
      if (!this.iconEl) {
        this.iconEl = E("i").cls("icon");
      }
      this.iconEl.innerText = this._icon;
      this.el.classList.add("with-icon");
      if (this.wrap) {
        this.wrap.insertBefore(this.iconEl, this.wrap.firstChild);
      }
      return this.iconEl;
    } else {
      if (this.iconEl) {
        this.iconEl.remove();
        this.el.classList.remove("with-icon");
      }
    }
  }
};

// goui/script/component/form/ArrayField.ts
var ArrayField = class extends Field {
  /**
   *
   * @param buildField Function that returns a new form field for an array item
   */
  constructor(buildField) {
    super("div");
    this.buildField = buildField;
    this.enableChangeEvent = true;
    this.baseCls = "flow";
    this.items.on("datachanged", () => {
      if (this.enableChangeEvent) {
        this.fireChange();
      }
    });
  }
  renderControl() {
  }
  set value(v) {
    super.value = v;
    this.enableChangeEvent = false;
    this.items.clear();
    if (v) {
      v.forEach((item) => {
        this.internalAddValue(item);
      });
    }
    this.enableChangeEvent = true;
  }
  get value() {
    const v = [];
    this.items.forEach((item) => {
      if (item instanceof Field) {
        v.push(item.value);
      }
    });
    return v;
  }
  /**
   * Add value to the values array. Also fires change event
   *
   * @param value
   */
  addValue(value) {
    if (!this.valueOnFocus) {
      this.captureValueForChange();
    }
    return this.internalAddValue(value);
  }
  internalAddValue(value) {
    const field = this.buildField(value);
    field.value = value;
    this.items.add(field);
    return field;
  }
  reset() {
    super.reset();
    if (this.items.count()) {
      this.enableChangeEvent = false;
      this.items.clear();
      this.enableChangeEvent = true;
      this.fireChange();
    }
  }
  isEmpty() {
    return this.items.count() > 0;
  }
};
var arrayfield = (config, ...items) => createComponent(new ArrayField(config.buildField), config, items);

// goui/script/component/form/InputField.ts
var InputField = class extends Field {
  constructor() {
    super();
  }
  /**
   * Get the DOM HTMLInputElement
   */
  get input() {
    return this._input;
  }
  set title(title) {
    super.title = title;
    if (this._input) {
      this._input.title = this.title;
    }
  }
  focus(o) {
    var _a;
    if (!this._input) {
      super.focus(o);
    }
    (_a = this._input) == null ? void 0 : _a.focus(o);
  }
  createControl() {
    this._input = this.createInput();
    return this._input;
  }
  createInput() {
    const control = document.createElement("input");
    if (this.invalidMsg) {
      this.applyInvalidMsg();
    }
    return control;
  }
  setInvalid(msg) {
    super.setInvalid(msg);
    if (this.rendered) {
      this.applyInvalidMsg();
    }
  }
  clearInvalid() {
    super.clearInvalid();
    this.applyInvalidMsg();
  }
  internalSetValue(v) {
    this._input.value = v !== void 0 && v !== null ? v.toString() : "";
  }
  set value(v) {
    super.value = v;
  }
  get value() {
    return this._input.value;
  }
  set name(name) {
    super.name = name;
    this._input.name = this.name;
  }
  get name() {
    return super.name;
  }
  set type(type) {
    if (this._input instanceof HTMLInputElement)
      this._input.type = type;
  }
  get type() {
    return this._input.type;
  }
  /**
   * Autocomplete value
   *
   * @link https://developer.mozilla.org/en-US/docs/Web/HTML/Attributes/autocomplete
   *
   * @param autocomplete
   */
  set autocomplete(autocomplete2) {
    this._input.autocomplete = this.autocomplete;
  }
  get autocomplete() {
    return this._input.autocomplete;
  }
  /**
   * When the field is empty this will be displayed inside the field
   *
   * @link https://developer.mozilla.org/en-US/docs/Web/HTML/Attributes/placeholder
   *
   * @param placeholder
   */
  set placeholder(placeholder) {
    if (!(this._input instanceof HTMLSelectElement))
      this._input.placeholder = placeholder;
    if (this.placeholder !== " ") {
      this.el.classList.add("no-floating-label");
    }
  }
  get placeholder() {
    if (!(this._input instanceof HTMLSelectElement))
      return this._input.placeholder;
    else
      return "";
  }
  /**
   * Make field read only
   *
   * @link https://developer.mozilla.org/en-US/docs/Web/HTML/Attributes/readonly
   * @param readOnly
   */
  set readOnly(readOnly) {
    super.readOnly = readOnly;
    if (!(this._input instanceof HTMLSelectElement))
      this._input.readOnly = this.readOnly;
  }
  get readOnly() {
    return super.readOnly;
  }
  /**
   * Make field required
   */
  get required() {
    return super.required;
  }
  set required(required) {
    super.required = required;
    this._input.required = this.required;
  }
  validate() {
    super.validate();
    if (this._input) {
      this.setValidityState(this._input);
    }
  }
};

// goui/script/component/form/TextField.ts
var TextField = class extends InputField {
  constructor() {
    super();
    this.baseCls = "goui-form-field text";
    this.type = "text";
  }
  /**
   * The input type
   *
   * @link https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input/text
   * @param type
   */
  set type(type) {
    super.type = type;
  }
  get type() {
    return super.type;
  }
  /**
   * Pattern regex for validation
   *
   * @link https://developer.mozilla.org/en-US/docs/Web/HTML/Attributes/pattern
   *
   * @param pattern
   */
  set pattern(pattern) {
    this.input.pattern = pattern;
  }
  get pattern() {
    return this.input.pattern;
  }
};
var textfield = (config) => createComponent(new TextField(), config);

// goui/script/component/Root.ts
var Root = class extends Component {
  constructor() {
    super();
    this.items.on("beforeadd", () => {
      this.el.classList.add("goui");
      this.el.classList.add("root");
    }, {
      once: true
    });
  }
  internalRender() {
    this.renderItems();
    return this.el;
  }
  get el() {
    if (!this._rootEl) {
      let rootEl = document.getElementById("goui");
      if (!rootEl) {
        rootEl = document.createElement("div");
        rootEl.id = "goui";
        document.body.append(rootEl);
      }
      this._rootEl = rootEl;
    }
    return this._rootEl;
  }
  get rendered() {
    return true;
  }
};
var root = new Root();

// goui/script/component/menu/Menu.ts
var Menu = class _Menu extends Toolbar {
  constructor() {
    super();
    /**
     * Make the menu at least as wide as the component it aligns too.
     */
    this.alignToInheritWidth = false;
    /**
     * The element it renders to. By default it's rendered to the root element of GOUI.
     */
    this.renderTo = root.el;
    /**
     * Remove menu when closed
     */
    this.removeOnClose = true;
    this.baseCls = "";
    this.orientation = "vertical";
  }
  internalRender() {
    const el = super.internalRender();
    this.el.addEventListener("keydown", (ev) => {
      switch (ev.key) {
        case "Escape":
          this.close();
          ev.stopPropagation();
          ev.preventDefault();
          break;
      }
    });
    const onClose = () => {
      if (_Menu.openedMenu == this) {
        _Menu.openedMenu = void 0;
      }
    };
    this.on("hide", onClose);
    this.on("remove", onClose);
    this.el.addEventListener("click", (e) => {
      e.stopPropagation();
    });
    return el;
  }
  /**
   * Menu can be rendered as a component in the normal flow or as a
   * floating dropdown.
   *
   * @param value
   */
  set isDropdown(value) {
    this.el.classList.toggle("goui-dropdown", value);
    this.el.classList.toggle("goui-fade-out", value);
    this.hidden = true;
  }
  get isDropdown() {
    return this.el.classList.contains("goui-dropdown");
  }
  /**
   * Expand menu on the left side of the parent button.
   * If not set then it will automatically detect that it goes outside the right side off the screen and set this to true
   * @param expandLeft
   */
  set expandLeft(expandLeft) {
    this.el.classList.add("expand-left");
  }
  get expandLeft() {
    return this.el.classList.contains("expand-left");
  }
  renderItem(item) {
    const insertBefore = this.getInsertBefore();
    if (!insertBefore) {
      this.el.appendChild(this.wrapLI(item));
    } else {
      this.el.insertBefore(this.wrapLI(item), insertBefore);
    }
  }
  wrapLI(item) {
    const li = document.createElement("li");
    item.render(li);
    if (item instanceof Button && item.menu) {
      item.menu.render(li);
    }
    item.on("remove", () => {
      li.remove();
    });
    return li;
  }
  /**
   * Show aligned to the given component.
   *
   * It will align the top left of the menu top the bottom left of the component. It will also be at least as wide as
   * the given component by setting the min-width style.
   *
   * @todo avoid going out of the viewport
   * @param cmp
   */
  showFor(alignEl) {
    this.alignTo = alignEl;
    this.show();
  }
  /**
   * Align the menu with it's "alignTo" element.
   */
  align() {
    if (!this.alignTo) {
      return;
    }
    const rect = this.alignTo.getBoundingClientRect();
    const x = Math.max(0, this.expandLeft ? rect.right - this.el.offsetWidth : rect.x);
    const y = Math.max(0, rect.bottom);
    this.x = x;
    this.y = y;
    if (this.alignToInheritWidth) {
      this.el.style.minWidth = rect.width + "px";
    }
    if (this.el.offsetWidth > window.innerWidth) {
      this.el.style.width = window.innerWidth + "px";
    }
    if (this.el.offsetHeight > window.innerHeight) {
      this.el.style.height = window.innerHeight + "px";
    }
    if (y + this.el.offsetHeight > window.innerHeight) {
      this.y = Math.max(0, rect.top - this.el.offsetHeight);
    }
    if (!this.expandLeft && x + this.el.offsetWidth > window.innerWidth) {
      this.expandLeft = true;
      this.x = Math.max(0, rect.right - this.el.offsetWidth);
    }
  }
  /**
   * Set X coordinate
   *
   * @param x
   */
  set x(x) {
    this.el.style.left = x + "px";
  }
  get x() {
    return parseInt(this.el.style.left);
  }
  /**
   * Set Y coordinate
   *
   * @param y
   */
  set y(y) {
    this.el.style.top = y + "px";
  }
  get y() {
    return parseInt(this.el.style.top);
  }
  internalSetHidden(hidden) {
    if (!hidden) {
      if (!this.parent) {
        root.items.add(this);
      }
      if (!this.rendered) {
        this.render();
      }
      this.align();
      if (_Menu.openedMenu == this) {
        return true;
      }
      if (_Menu.openedMenu) {
        _Menu.openedMenu.el.classList.remove("goui-fade-out");
        _Menu.openedMenu.close();
      }
      _Menu.openedMenu = this;
      window.addEventListener("mousedown", (ev) => {
        this.close();
      }, { once: true });
      this.el.addEventListener("mousedown", (ev) => {
        ev.stopPropagation();
      });
      this.el.classList.add("goui-fade-out");
    }
    super.internalSetHidden(hidden);
  }
  /**
   * Show menu at coordinates on the page.
   *
   * Useful for a context menu
   *
   * @param coords
   */
  showAt(coords) {
    this.x = coords.x;
    this.y = coords.y;
    this.show();
  }
  /**
   * Closes the menu.
   *
   * It will hide or remove it depending on the "removeOnClose" property.
   */
  close() {
    return this.removeOnClose ? this.remove() : this.hide();
  }
  /**
   * @inheritDoc
   */
  focus(o) {
    var _a;
    (_a = this.items.get(0)) == null ? void 0 : _a.focus(o);
  }
};
var menu = (config, ...items) => createComponent(new Menu(), config, items);

// goui/script/Router.ts
var Router = class extends Observable {
  constructor() {
    super();
    this.routes = [];
    this.suspendEvent = false;
    this.loadedPath = "";
    this.debug = false;
    this.params = [];
    window.addEventListener("hashchange", () => {
      this.start();
    }, false);
  }
  /**
   * Get the router path
   */
  getPath() {
    return window.location.hash.substr(1);
  }
  /**
   * Get the parameters evaluated from the router path
   */
  getParams() {
    return this.params;
  }
  /**
   * Set route path without executing matching routes.
   *
   * @param path
   */
  setPath(path) {
    if (path != window.location.hash) {
      this.suspendEvent = true;
      const oldPath = this.getPath();
      window.location.hash = path;
      this.fire("change", this.getPath(), oldPath);
    }
  }
  /**
   * Add a route
   *
   * The first matching route will be executed
   *
   * @example
   * ```
   * go.Router.add(/^([a-zA-Z0-9]*)\/([\d]*)$/, (entity:string, id:string) => {
   *
   * });
   * ```
   *
   * @param re eg. /^notes/(.*)$/
   * @param handler Is called with the arguments matched in the route regexp. May return Promise so the router start()
   *  promise will resolve when this promise is resolved.
   */
  add(re, handler) {
    if (this.debug) {
      console.debug("Router add: ", re);
    }
    if (typeof re == "function") {
      handler = re;
      this.defaultRoute = handler;
      return this;
    }
    const route = { re, handler };
    this.routes.push(route);
    return this;
  }
  /**
   * Start the router and run the matching route handlers
   */
  start() {
    const path = this.getPath();
    const oldPath = this.loadedPath;
    if (this.suspendEvent) {
      setTimeout(() => {
        this.suspendEvent = false;
      });
      return Promise.resolve();
    }
    this.loadedPath = path;
    for (let i = 0; i < this.routes.length; i++) {
      const args = path.match(this.routes[i].re);
      if (args) {
        if (this.debug) {
          console.debug("Router match: ", this.routes[i].re);
        }
        args.shift();
        return this.handleRoute(this.routes[i].handler, args, oldPath);
      }
    }
    return this.defaultRoute ? this.handleRoute(this.defaultRoute, [], oldPath) : Promise.resolve();
  }
  handleRoute(handler, match, oldPath) {
    for (let n = 0, l = match.length; n < l; n++) {
      match[n] = match[n] ? decodeURIComponent(match[n]) : match[n];
    }
    this.params = match;
    const result = handler.apply({}, match);
    window.scrollTo(0, 0);
    this.fire("change", this.getPath(), oldPath);
    return result instanceof Promise ? result : Promise.resolve();
  }
  /**
   * Reload current page.
   */
  reload() {
    this.start();
  }
  /**
   * Go to the give router path
   *
   * @param path
   * @return Promise<Router>
   */
  goto(path) {
    const p2 = new Promise((resolve, reject) => {
      this.on("change", (path1, oldPath) => {
        resolve(this);
      }, { once: true });
    });
    window.location.hash = path || "";
    return p2;
  }
};
var router = new Router();

// goui/script/component/Button.ts
var Button = class _Button extends Component {
  constructor() {
    super("button");
    this.baseCls = "goui-button";
    /**
     * Turn on if you want this button to be clickable fast.
     * We disable this by default because some users tend to double click on all buttons and we don't
     * want to double submit.
     */
    this.allowFastClick = false;
    this.type = "button";
  }
  /**
   * Find the first menu in the tree of submenu's
   */
  findTopMenu() {
    if (!(this.parent instanceof Menu)) {
      return void 0;
    }
    if (!(this.parent.parent instanceof _Button)) {
      return this.parent;
    } else {
      const next = this.parent.parent.findTopMenu();
      if (next) {
        return next;
      } else {
        return this.parent;
      }
    }
  }
  /**
   * Button type. "button" or "submit", defaults to "button".
   */
  set type(type) {
    this.el.type = type;
  }
  get type() {
    return this.el.type;
  }
  internalRender() {
    const el = super.internalRender();
    if (this.route != void 0) {
      if (this.handler) {
        throw "You can't set both handler and route on a button";
      }
      this.handler = () => {
        router.goto(this.route);
      };
    }
    if (this.menu) {
      this.menu.hide();
      if (!(this.parent instanceof Menu)) {
        this.el.addEventListener("mouseenter", this.onMenuMouseEnter.bind(this));
        this.el.addEventListener("click", this.onMenuButtonClick.bind(this));
      } else {
        this.menu.renderTo = void 0;
      }
    }
    el.addEventListener("click", (e) => {
      if (this.handler && e.button == 0 && (this.allowFastClick || e.detail < 2)) {
        this.handler.call(this, this, e);
        const topMenu = this.findTopMenu();
        if (topMenu && topMenu.isDropdown) {
          topMenu.close();
        }
      }
      this.fire("click", this, e);
    });
    return el;
  }
  onMenuButtonClick(ev) {
    if (this._menu.hidden) {
      this.showMenu();
    } else {
      this._menu.hide();
    }
  }
  onMenuMouseEnter(ev) {
    if (Menu.openedMenu && Menu.openedMenu != this._menu && Menu.openedMenu.parent.parent === this._menu.parent.parent) {
      this.showMenu();
    }
  }
  /**
   * Add menu to this button
   */
  set menu(menu2) {
    if (menu2) {
      menu2.parent = this;
      menu2.removeOnClose = false;
      menu2.isDropdown = true;
      this.el.classList.add("has-menu");
    }
    this._menu = menu2;
  }
  get menu() {
    return this._menu;
  }
  showMenu() {
    if (!this._menu) {
      return;
    }
    if (!this._menu.hidden) {
      return;
    }
    if (this.fire("beforeshowmenu", this, this._menu) === false) {
      return;
    }
    if (!this._menu.alignTo) {
      this._menu.alignTo = this.el;
    }
    this._menu.show();
    this.fire("showmenu", this, this._menu);
  }
  internalRemove() {
    if (this.menu) {
      this.menu.remove();
    }
    super.internalRemove();
  }
  /**
   * Button text
   *
   * Set's the button text and adds a "text" css class
   *
   * This overrides the "html" property. Use html if you want something custom.
   */
  set text(text) {
    if (text) {
      this._text = text;
      this.el.classList.add("with-text");
    } else {
      this.el.classList.remove("with-text");
    }
    this.textEl.innerText = text + "";
  }
  get text() {
    return this._text + "";
  }
  /**
   * Set's the button icon and adds a "icon" css class
   */
  set icon(icon) {
    this._icon = icon;
    if (this._icon != void 0) {
      this.el.classList.add("with-icon");
    } else {
      this.el.classList.remove("with-icon");
    }
    this.iconEl.innerText = icon + "";
  }
  get icon() {
    return this._icon;
  }
  get iconEl() {
    if (!this._iconEl) {
      this._iconEl = document.createElement("i");
      this._iconEl.classList.add("icon");
      if (this._textEl) {
        this.el.insertBefore(this._iconEl, this._textEl);
      } else {
        this.el.appendChild(this._iconEl);
      }
    }
    return this._iconEl;
  }
  get textEl() {
    if (!this._textEl) {
      this._textEl = document.createElement("span");
      this._textEl.classList.add("text");
      this.el.appendChild(this._textEl);
    }
    return this._textEl;
  }
};
var btn = (config) => createComponent(new Button(), config);

// goui/script/component/picker/ColorPicker.ts
var ColorPicker = class _ColorPicker extends Component {
  constructor() {
    super();
    this._value = "";
    /**
     * Set color value as button text color if this menu belongs to a button
     */
    this.updateButton = true;
    this.baseCls = "goui-dropdown goui-menu-color";
    this.colors = [
      // '000000', //Black
      // 'FFFFFF', //White
      // '009BC9', //Group-Office blue
      // //'243A80', //Intermesh blue
      // '78A22F', //default secondary
      // 'FF9100',  //Default accent
      "B71C1C",
      "C62828",
      "D32F2F",
      "E53935",
      "F44336",
      // Red
      "880E4F",
      "AD1457",
      "C2185B",
      "D81B60",
      "E91E63",
      // Pink
      "4A148C",
      "6A1B9A",
      "7B1FA2",
      "8E24AA",
      "9C27B0",
      // Purple
      "1A237E",
      "283593",
      "303F9F",
      "3949AB",
      "3F51B5",
      // Indigo
      "0D47A1",
      "1565C0",
      "1976D2",
      "1E88E5",
      "2196F3",
      // Blue
      "006064",
      "00838F",
      "0097A7",
      "00ACC1",
      "00BCD4",
      // Cyan
      "004D40",
      "00695C",
      "00796B",
      "00897B",
      "009688",
      // Teal
      "1B5E20",
      "2E7D32",
      "388E3C",
      "43A047",
      "4CAF50",
      // Green
      "827717",
      "9E9D24",
      "AFB42B",
      "C0CA33",
      "CDDC39",
      // Lime
      //'F57F17', 'F9A825', 'FBC02D', 'FDD835', 'FFEB3B', // Yellow
      "FF6F00",
      "FF8F00",
      "FFA000",
      "FFB300",
      "FFC107",
      // Amber
      "E65100",
      "EF6C00",
      "F57C00",
      "FB8C00",
      "FF9800",
      // Orange
      "212121",
      "424242",
      "616161",
      "757575",
      "BDBDBD"
      // Grey
    ];
    this.items.add(
      comp(
        {},
        ...this.colors.map((color) => btn({
          itemId: color,
          cls: this.value == "#" + color ? "with-icon pressed" : "with-icon",
          listeners: {
            beforerender: (btn2) => {
              const colorDiv = document.createElement("div");
              colorDiv.style.backgroundColor = "#" + color;
              btn2.el.appendChild(colorDiv);
            }
          },
          handler: (btn2) => {
            this.value = btn2.itemId;
            this.fire("select", this, btn2.itemId);
          }
        }))
      )
    );
  }
  //picker interfae
  getText() {
    return this.value.substring(1);
  }
  setValue(val) {
    this.value = val;
  }
  static rgb2hex(str) {
    const rgb = str.match(/^rgb\((\d+),\s*(\d+),\s*(\d+)\)$/i);
    if (!rgb) {
      return "";
    }
    return "#" + ((1 << 24) + (parseInt(rgb[1]) << 16) + (parseInt(rgb[2]) << 8) + parseInt(rgb[3])).toString(16).slice(1);
  }
  /**
   * Color hex value eg. #000000
   */
  set value(color) {
    const hex = _ColorPicker.rgb2hex(color);
    if (hex) {
      color = hex.toUpperCase();
    } else {
      color = color.toUpperCase();
    }
    this._value = color;
    if (!this.rendered) {
      return;
    }
    if (color == "") {
      color = "auto";
    }
    this.items.forEach((btn2) => {
      btn2.el.classList.toggle("pressed", btn2.itemId == color);
    });
  }
  get value() {
    return this._value;
  }
};
var colorpicker = (config) => createComponent(new ColorPicker(), config);

// goui/script/component/picker/DatePicker.ts
var DatePicker = class extends Component {
  // private footer: HTMLElement;
  constructor() {
    super();
    this.enableRangeSelect = false;
    this.baseCls = "datepicker";
    this.showWeekNbs = true;
    this.value = new DateTime();
    this.now = new DateTime();
    this.el.cls("+preload");
    this.el.append(
      this.menu = E(
        "header",
        E("button", E("i", "chevron_left").cls("icon")).cls(["goui-button", "nav", "with-icon"], true).on("click", (_) => {
          this.moveMonth(-1);
        }),
        this.monthEl = E("button").cls("goui-button").on("click", (_) => {
          this.months.cls("!active");
          this.years.cls("-active");
        }),
        this.yearEl = E("button").cls("goui-button").on("click", (_) => {
          this.years.cls("!active");
          this.months.cls("-active");
          this.years.scrollTop = this.years.scrollHeight / 2 - this.years.clientHeight / 2;
        }),
        E("button", E("i", "chevron_right").cls("icon")).cls(["goui-button", "nav", "with-icon"], true).on("click", (_) => {
          this.moveMonth(1);
        })
      ),
      this.grid = E("div").cls("+cards"),
      E(
        "div",
        this.years = E("ul").on("click", ({ target }) => {
          const year = +target.textContent;
          if (!isNaN(year)) {
            this.value.setYear(year);
            this.setValue(this.value);
            this.years.cls("-active");
          }
        }),
        this.months = E("ul").on("click", ({ target }) => {
          if (target.dataset.nb) {
            this.value.setMonth(+target.dataset.nb);
            this.setValue(this.value);
            this.months.cls("-active");
          }
        })
      ).cls(["cards", "top"], true)
    );
    for (let i = -100; i < 100; i++) {
      this.years.append(E("li", this.value.getYear() + i + "").cls("now", i === 0));
    }
    this.months.replaceChildren(...DateTime.monthNames.map(
      (name, i) => E("li", name).cls("now", i === this.now.getMonth() - 1).attr("data-nb", i + 1)
    ));
  }
  moveMonth(amount) {
    this.grid.cls("reverse", amount > 0);
    this.grid.offsetHeight;
    this.value.addMonths(amount);
    this.setValue(this.value);
  }
  render(parentEl, insertBefore) {
    if (!this.withoutFooter) {
      this.el.append(
        E(
          "footer",
          E("button", t("Clear")).cls(["goui-button", "primary"], true).on("click", (_) => {
            this.fire("select", this, void 0);
          }),
          E("div").attr("style", "flex:1"),
          E("button", t("Today")).cls(["goui-button", "primary"], true).on("click", (_) => {
            this.fire("select", this, new DateTime());
          })
        )
      );
    }
    return super.render(parentEl, insertBefore);
  }
  /**
   * Refresh the view
   */
  refresh() {
    this.el.cls("+preload");
    if (this.value) {
      this.setValue(this.value);
    }
  }
  constrainValue() {
    if (this.maxDate) {
      if (this.value.format("Ymd") > this.maxDate.format("Ymd")) {
        this.value = this.maxDate.clone();
      }
    }
    if (this.minDate) {
      if (this.value.format("Ymd") < this.minDate.format("Ymd")) {
        this.value = this.minDate.clone();
      }
    }
  }
  setYearClases() {
    var _a;
    (_a = this.years.querySelector(".selected")) == null ? void 0 : _a.cls("-selected");
    for (let i = 0, l = this.years.children.length; i < l; i++) {
      const child = this.years.children[i];
      const curYear = parseInt(child.textContent);
      if (curYear === this.value.getYear()) {
        child.cls("+selected");
      }
      if (this.minDate && this.minDate.getYear() > curYear) {
        child.cls("+disabled");
      }
      if (this.maxDate && this.maxDate.getYear() < curYear) {
        child.cls("+disabled");
      }
    }
  }
  setMonthClasses() {
    for (let m = 0; m < 12; m++) {
      const child = this.months.children[m];
      const curMonth = this.value.getYear() + (m + 1).toString().padStart(2, "0");
      let sign;
      sign = m === this.value.getMonth() - 1 ? "+" : "-";
      child.cls(sign + "selected");
      sign = this.minDate && this.minDate.format("Ym") > curMonth || this.maxDate && this.maxDate.format("Ym") < curMonth ? "+" : "-";
      child.cls(sign + "disabled");
    }
  }
  internalRender() {
    var _a;
    this.constrainValue();
    const el = super.internalRender();
    const year = this.value.format("Y"), month = this.value.format("M");
    this.renderedMonth = year + month;
    this.yearEl.innerHTML = year + '<i class="icon">arrow_drop_down</i>';
    this.monthEl.innerHTML = month + '<i class="icon">arrow_drop_down</i>';
    this.setYearClases();
    this.setMonthClasses();
    const cal = E("div").cls("active");
    const dl2 = E("dl"), itr = this.value.clone().setMonthDay(1).setWeekDay(0), weekNbs = E("ol");
    dl2.append(...Object.values(DateTime.dayNames).map((s) => E("dt", s.substring(0, 2))));
    const minStr = this.minDate ? this.minDate.format("Ymd") : "00000101", maxStr = this.maxDate ? this.maxDate.format("Ymd") : "99991231";
    for (let i = 0; i < 42; i++) {
      const itrStr = itr.format("Ymd");
      if (this.showWeekNbs && i % 7 == 0) {
        weekNbs.append(E("li", itr.format("W")));
      }
      dl2.append(
        E("dd", itr.format("j")).attr("data-date", itr.format("Y-m-d")).cls("disabled", minStr > itrStr || maxStr < itrStr).cls("off", itr.format("Ym") !== this.value.format("Ym")).cls("today", itr.format(`Ymd`) === this.now.format("Ymd"))
      );
      itr.addDays(1);
    }
    if (this.showWeekNbs) {
      cal.append(weekNbs);
    }
    this.setupDraggability(dl2);
    cal.append(dl2);
    const old = (_a = this.grid.firstElementChild) == null ? void 0 : _a.cls("-active");
    this.grid.prepend(cal);
    setTimeout(function() {
      el.cls("-preload");
      old == null ? void 0 : old.remove();
    }, 250);
    return el;
  }
  setValue(start, end) {
    this.value = start;
    if (this.rendered && start.format("YM") !== this.renderedMonth) {
      this.el.cls("+preload");
      this.internalRender();
    }
    const startEl = this.grid.querySelector('dd[data-date="' + start.format("Y-m-d") + '"]'), endEl = !end ? startEl : this.grid.querySelector('dd[data-date="' + end.format("Y-m-d") + '"]');
    if (startEl && endEl) {
      this.markSelected(startEl, endEl);
    }
  }
  markSelected(start, end) {
    this.el.querySelectorAll(".selected").forEach((e) => e.cls(["selected", "tail", "head"], false));
    end.cls("+tail");
    start.cls("+head");
    let itr = start;
    while (itr && itr != end.nextElementSibling) {
      itr.cls("+selected");
      itr = itr.nextElementSibling;
    }
  }
  setupDraggability(dl2) {
    let anchor, start, end;
    const onMouseMove = ({ target }) => {
      if (!this.enableRangeSelect) {
        return false;
      }
      if (target.isA("DD") && target != end) {
        [start, end] = anchor.compareDocumentPosition(target) & 2 ? [target, anchor] : [anchor, target];
        this.markSelected(start, end);
      }
    }, onMouseUp = (_e) => {
      dl2.un("mousemove", onMouseMove);
      window.removeEventListener("mouseup", onMouseUp);
      if (!this.enableRangeSelect || start == end) {
        this.fire("select", this, new DateTime(start.attr("data-date")));
      } else {
        this.fire("select-range", this, new DateTime(start.attr("data-date")), new DateTime(end.attr("data-date")));
      }
    };
    dl2.on("mousedown", ({ target }) => {
      if (target.isA("dd")) {
        anchor = start = end = target;
        dl2.querySelectorAll("dd.selected").forEach((e) => e.cls(["selected", "tail", "head"], false));
        target.cls(["selected", "tail", "head"], true);
        dl2.on("mousemove", onMouseMove);
        window.addEventListener("mouseup", onMouseUp);
      }
    });
  }
};
var datepicker = (config) => createComponent(new DatePicker(), config);

// goui/script/component/form/SelectField.ts
var SelectField = class extends InputField {
  constructor() {
    super(...arguments);
    this.baseCls = "goui-form-field select no-floating-label";
    /**
     * The field of the select options that is used as value
     */
    this.valueField = "value";
    this.fireChangeOnBlur = false;
    /**
     * Renderer function. Defaults to returning a 'name' property.
     *
     * @param record
     */
    this.textRenderer = (record) => record.name;
  }
  createInput() {
    return document.createElement("select").on("change", (_) => this.fireChange());
  }
  // turned off fireChangeOnBlur but override onFocusIn() to get the oldValue
  onFocusIn(e) {
    this.captureValueForChange();
  }
  /**
   * Redraw the options. Can be useful when this.textRenderer() produces another result
   */
  drawOptions() {
    this.options = this.options;
  }
  internalRender() {
    if (this._store)
      this.options = this._store.items;
    return super.internalRender();
  }
  /**
   * Provide select input with options
   *
   * It should have at least have a field that corresponds with {@link Select.valueField}
   *
   * By default, it should have a "value" and "name" property. This can be changed with the {@link Select.valueField} and
   * {@link Select.textRenderer}.
   *
   * @param opts
   */
  set options(opts) {
    const v = this._value;
    this._options = opts;
    this.input.empty();
    opts.forEach((o) => {
      var _a;
      this.input.append(new Option(this.textRenderer(o), (_a = o[this.valueField]) != null ? _a : void 0));
    });
    this.internalSetValue(v);
  }
  get options() {
    var _a;
    return (_a = this._options) != null ? _a : [];
  }
  /**
   * A store to provide the {@link Select.options}.
   * @param store
   */
  set store(store3) {
    this._store = store3;
    store3.on("datachanged", () => this.options = store3.items);
  }
  get store() {
    return this._store;
  }
  set value(v) {
    super.value = v;
  }
  get value() {
    if (!this.rendered) {
      return this._value;
    }
    const opts = this.store ? this.store.items : this.options;
    let index = this.input.selectedIndex;
    let v;
    if (opts[index]) {
      v = opts[index][this.valueField];
    } else {
      v = void 0;
    }
    return v;
  }
};
var select = (config) => createComponent(new SelectField(), config);

// goui/script/component/form/TimeField.ts
var TimeField = class extends InputField {
  constructor() {
    super();
    this.baseCls = "goui-form-field time no-floating-label";
    this.type = "time";
  }
  set step(v) {
    this.input.step = v.toString();
  }
  get step() {
    return parseInt(this.input.step);
  }
  /**
   * The minimum number allowed
   *
   * The value of the time input is always in 24-hour format that includes leading zeros: hh:mm
   *
   * @param min
   */
  set min(min) {
    this.input.attr("min", min);
  }
  get min() {
    return this.input.attr("min");
  }
  /**
   * The maximum number allowed
   *
   * The value of the time input is always in 24-hour format that includes leading zeros: hh:mm
   *
   * @param max
   */
  set max(max) {
    this.input.attr("max", max);
  }
  outputFormat() {
    return "H:i";
  }
  /**
   * Get the date as DateTime object
   */
  getValueAsDateTime() {
    let v = this.value, date;
    if (!v || !(date = DateTime.createFromFormat(v, this.outputFormat()))) {
      return void 0;
    }
    return date;
  }
  set value(v) {
    super.value = v;
  }
  get value() {
    const v = super.value;
    return v ? v : void 0;
  }
};
var timefield = (config) => createComponent(new TimeField(), config);

// goui/script/component/form/DateField.ts
var DateField = class extends TimeField {
  constructor() {
    super();
    this.baseCls = "goui-form-field date no-floating-label";
    this.type = "date";
  }
  /**
   * Also render time input
   *
   * @param withTime
   */
  set withTime(withTime) {
    let v = this.value, newType = withTime ? "datetime-local" : "date";
    if (newType != this.input.type) {
      this.input.type = newType;
      if (!v) {
        return;
      }
      if (withTime) {
        v = this.appendTime(v);
      } else {
        const parts = v.split("T");
        v = parts[0];
        this.defaultTime = parts[1];
      }
      this.input.value = v;
    }
  }
  appendTime(v) {
    var _a;
    return v + "T" + ((_a = this.defaultTime) != null ? _a : new DateTime().format("H:i"));
  }
  get withTime() {
    return this.type == "datetime-local";
  }
  get value() {
    return super.value;
  }
  /**
   * The value of the date is in Y-m-d  or  Y-m-d H:i when withTime is true. {@link DateTime.format}
   * @param v
   */
  set value(v) {
    if (v) {
      const Tindex = v.indexOf("T");
      if (this.withTime) {
        if (Tindex == -1) {
          v = this.appendTime(v);
        }
      } else {
        if (Tindex != -1) {
          const parts = v.split("T");
          v = parts[0];
          this.defaultTime = parts[1];
        }
      }
    }
    super.value = v;
  }
  outputFormat() {
    return this.withTime ? "Y-m-dTH:i" : "Y-m-d";
  }
};
var datefield = (config) => createComponent(new DateField(), config);

// goui/script/component/form/ContainerField.ts
var ContainerField = class extends Field {
  constructor(tagName = "div") {
    super(tagName);
    this.baseCls = "";
    this.hideLabel = true;
    this.fireChangeOnBlur = false;
    /**
     * Not needed on container field as the fields within handle this.
     */
    this.validateOnBlur = false;
    this.cls = "flow";
  }
  /**
   * Find all form fields
   *
   * @param nameOrItemId If given only items with matching name or itemId are returned
   */
  findFields(nameOrItemId = void 0) {
    const fields = [];
    const fn = (item) => {
      if (item == this) {
        return;
      }
      if (item.isFormField) {
        if (!nameOrItemId || (item.name == nameOrItemId || item.itemId == nameOrItemId)) {
          fields.push(item);
        }
        return false;
      }
    };
    this.cascade(fn);
    return fields;
  }
  /**
   * Find form field by name or item ID
   *
   * It cascades down the component hierarchy.
   *
   * @param nameOrItemId
   */
  findField(nameOrItemId) {
    let field;
    const fn = (item) => {
      if (item.isFormField && item.name == nameOrItemId || item.itemId == nameOrItemId) {
        field = item;
        return false;
      }
    };
    this.cascade(fn);
    return field;
  }
  /**
   * Copies the current value to the reset value. Typically happens when this component was added to a parent and
   * when the form it belongs too loads.
   */
  trackReset() {
    this.findFields().forEach((field) => {
      field.trackReset();
    });
  }
  isModified() {
    const f = this.findFields();
    for (let i = 0, l = f.length; i < l; i++) {
      if (f[i].isModified()) {
        return true;
      }
    }
    return false;
  }
  set value(v) {
    for (let name in v) {
      let fields = this.findFields(name);
      fields.forEach((field) => field.setValue ? field.setValue(v[name]) : field.value = v[name]);
    }
    super.value = v;
  }
  get value() {
    const formProps = structuredClone(super.value) || {};
    this.findFields().forEach((field) => {
      const fieldName = field.getName ? field.getName() : field.name;
      const fieldVal = field.getValue ? field.getValue() : field.value;
      if (fieldName && !field.disabled) {
        formProps[fieldName] = fieldVal;
      }
    });
    return formProps;
  }
  validate() {
    super.validate();
    let invalid;
    this.findFields().forEach((i) => {
      if (!i.disabled && !i.isValid()) {
        invalid = i;
      }
    });
    if (invalid) {
      this.setInvalid("There's an invalid field");
    }
  }
  /**
   * Find the first invalid field
   */
  findFirstInvalid() {
    const items = this.findFields();
    for (let i = 0, l = items.length; i < l; i++) {
      if (!items[i].disabled && !items[i].isValid()) {
        return items[i];
      }
    }
    return void 0;
  }
  renderControl() {
  }
  clearInvalid() {
    super.clearInvalid();
    const items = this.findFields();
    items.forEach((field) => {
      field.clearInvalid();
    });
  }
  applyInvalidMsg() {
    if (this.invalidMsg) {
      this.el.classList.add("invalid");
    } else {
      this.el.classList.remove("invalid");
    }
  }
  /**
   * @inheritDoc
   */
  focus(o) {
    const fields = this.findFields();
    if (fields.length) {
      fields[0].focus(o);
      this.fire("focus", this, o);
    } else {
      super.focus(o);
    }
  }
};
var containerfield = (config, ...items) => createComponent(new ContainerField(), config, items);

// goui/script/Notifier.ts
var Notifier = class {
  /**
   * Show an error toast
   *
   * @param msg any string or object with a message property
   * @param timeout Timeout in seconds before it automatically disappears. It also dissappears on any mouseclick
   */
  static error(msg, timeout = 0) {
    console.error(msg);
    if (msg instanceof Object && msg.message) {
      msg = msg.message;
    } else if (typeof msg != "string") {
      msg = msg + "";
    }
    return new Message(msg, "error", timeout);
  }
  /**
   * Show success toast
   *
   * @param msg
   * @param timeout
   */
  static success(msg, timeout = 3e3) {
    return new Message(msg, "success", timeout);
  }
  /**
   * Show a notice toast
   *
   * @param msg
   * @param timeout
   */
  static notice(msg, timeout = 3e3) {
    return new Message(msg, "notice", timeout);
  }
  /**
   * Show a warning toast
   * @param msg
   * @param timeout
   */
  static warning(msg, timeout = 3e3) {
    return new Message(msg, "warning", timeout);
  }
};
var Message = class {
  constructor(msg, type, timeout = 3e3) {
    const alert = comp(
      {
        cls: "goui-alert " + type
      },
      comp({
        tagName: "span",
        text: msg
      })
    );
    root.items.add(alert);
    if (timeout) {
      this.timeout = window.setTimeout(() => {
        alert.remove();
      }, timeout);
    }
    setTimeout(() => {
      document.body.addEventListener("click", () => {
        alert.remove();
        if (this.timeout) {
          clearTimeout(this.timeout);
        }
      }, { once: true });
    });
  }
};

// goui/script/component/form/Form.ts
var Form = class extends ContainerField {
  constructor() {
    super("form");
    this.baseCls = "goui-form";
    this.hideLabel = true;
  }
  internalRender() {
    const el = super.internalRender();
    this.el.noValidate = true;
    el.addEventListener("submit", (event) => {
      event.preventDefault();
      this.submit();
    });
    el.addEventListener("reset", (e) => {
      e.preventDefault();
      this.reset();
    });
    el.addEventListener("keydown", (e) => {
      if ((e.ctrlKey || e.metaKey) && e.key == "Enter") {
        e.preventDefault();
        if (document.activeElement && "blur" in document.activeElement) {
          document.activeElement.blur();
        }
        this.submit();
      }
    });
    return el;
  }
  /**
   * @inheritDoc
   */
  reset() {
    this.findFields().forEach((field) => {
      field.reset();
    });
  }
  set value(v) {
    super.value = v;
    this.trackReset();
  }
  get value() {
    return super.value;
  }
  /**
   * Get the modified field values since the form was:
   *
   * - rendered OR
   * - value was set (usually through a load) OR
   * - submitted
   */
  get modified() {
    const v = {};
    this.findFields().forEach((field) => {
      if (field instanceof Field) {
        if (field.name && !field.disabled && field.isModified()) {
          v[field.name] = field.value;
        }
      } else {
        const fieldName = field.getName();
        if (fieldName && !field.disabled && field.isDirty()) {
          v[fieldName] = field.getValue();
        }
      }
    });
    return v;
  }
  /**
   * Validates the form and submits it using the handler function passed with the config.
   */
  submit() {
    return __async(this, null, function* () {
      const el = this.el;
      this.clearInvalid();
      if (this.isValid()) {
        el.cls(["+valid", "-invalid"]);
        let handlerResponse = void 0;
        if (this.handler) {
          try {
            handlerResponse = yield this.handler(this);
          } catch (e) {
            el.cls(["-valid", "+invalid"]);
            const msg = typeof e == "string" ? e : e.message;
            Notifier.error(msg);
            return;
          }
        }
        this.fire("submit", this, handlerResponse);
      } else {
        el.cls(["-valid", "+invalid"]);
        const invalid = this.findFirstInvalid();
        if (invalid) {
          invalid.focus();
        }
        this.setInvalid(t("You have errors in your form. The invalid fields are marked."));
      }
    });
  }
  setInvalid(msg) {
    super.setInvalid(msg);
    Notifier.error(msg);
  }
};
/**
 * When this is set to true, the field will use the values set as their original value, used for resetting and
 * determining if the field was modified.
 */
Form.TRACK_RESET_VALUES = false;
var form = (config, ...items) => createComponent(new Form(), config, items);

// goui/script/component/form/NumberField.ts
var NumberField = class extends InputField {
  constructor() {
    super();
    this.baseCls = "goui-form-field number";
    this.type = "number";
  }
  validate() {
    super.validate();
    const v = this.value;
    if (v && isNaN(v)) {
      this.setInvalid("Incorrect number format");
    }
    if (this.max !== void 0 && v && v > this.max) {
      this.setInvalid(t("Number is bigger than the maximum of {max}.").replace("{max}", this.max.toLocaleString()));
    }
    if (this.min !== void 0 && (!v || v < this.min)) {
      this.setInvalid(t("Number is bigger than the maximum of {max}.").replace("{max}", this.min.toLocaleString()));
    }
  }
  set value(v) {
    if (this.isEmptyNumber(v)) {
      super.value = void 0;
    } else if (isNaN(v)) {
      console.error("Invalid number given for field " + this.name, v);
      super.value = void 0;
    } else {
      super.value = +v.toFixed(this.decimals);
    }
  }
  isEmptyNumber(v) {
    return v === void 0 || v === null || v === "";
  }
  get value() {
    const v = super.value;
    return v === void 0 || this.isEmptyNumber(v) || isNaN(v) ? void 0 : +(+v).toFixed(this.decimals);
  }
  /**
   * The step attribute is a number that specifies the granularity that the value must adhere to or the keyword any.
   *
   * @link https://developer.mozilla.org/en-US/docs/Web/HTML/Attributes/step
   * @param step
   */
  set step(step) {
    if (step === void 0) {
      this.input.removeAttribute("step");
    } else {
      this.input.setAttribute("step", step.toString());
    }
  }
  get step() {
    var _a;
    return parseFloat((_a = this.input.getAttribute("step")) != null ? _a : "0");
  }
  /**
   * Set the number of decimals. It uses the step attribute to accomplish this.
   *
   * @param decimals
   */
  set decimals(decimals) {
    if (!decimals) {
      this.input.removeAttribute("step");
    } else {
      this.input.setAttribute("step", "0.".padEnd(decimals + 1, "0") + "1");
    }
  }
  get decimals() {
    const step = this.input.attr("step");
    if (!step) {
      return void 0;
    }
    return step.length - 2;
  }
  /**
   * The minimum number allowed
   *
   * @param min
   */
  set min(min) {
    if (min === void 0) {
      this.input.removeAttribute("min");
    } else {
      this.input.setAttribute("min", min.toString());
    }
  }
  get min() {
    const min = this.input.getAttribute("min");
    if (min === null) {
      return void 0;
    }
    return parseFloat(min);
  }
  /**
   * The maximum number allowed
   *
   * @param max
   */
  set max(max) {
    if (max === void 0) {
      this.input.removeAttribute("max");
    } else {
      this.input.setAttribute("max", max.toString());
    }
  }
  get max() {
    const max = this.input.getAttribute("max");
    if (max === null) {
      return void 0;
    }
    return parseFloat(max);
  }
};
var numberfield = (config) => createComponent(new NumberField(), config);

// goui/script/component/CardContainer.ts
var CardContainer = class extends Component {
  constructor() {
    super();
    this.baseCls = "cards";
    this.items.on("beforeadd", (card, item) => {
      item.hide();
      item.on("show", (comp2) => {
        const index = this.findItemIndex(comp2);
        this.activeItem = index;
      });
      item.el.classList.add("card-container-item");
    });
  }
  internalRender() {
    this.setCardVisibilities();
    const el = super.internalRender();
    return el;
  }
  renderItems() {
    if (this.items) {
      this.items.forEach((item) => {
        if (item.hidden) {
          item.on("show", (item2) => {
            this.renderItem(item2);
          }, { once: true });
        } else {
          this.renderItem(item);
        }
      });
    }
  }
  setCardVisibilities() {
    this.items.forEach((item, index) => {
      if (index == this.activeItem) {
        item.show();
      } else {
        item.hide();
      }
    });
  }
  /**
   * The active card index. Defaults to 0 if not given.
   */
  set activeItem(ref) {
    let index;
    if (ref instanceof Component) {
      index = this.findItemIndex(ref);
    } else {
      index = ref;
    }
    const old = this._activeItem;
    this._activeItem = index;
    if (old !== index) {
      this.fire("cardchange", this, index, old);
    }
    this.setCardVisibilities();
  }
  /**
   * The active card index. Defaults to 0 if not given.
   */
  get activeItem() {
    if (this._activeItem == void 0 && this.items.count()) {
      this._activeItem = 0;
    }
    return this._activeItem;
  }
  focus(o) {
    if (this.activeItem) {
      const activeItem = this.items.get(this.activeItem);
      if (activeItem) {
        activeItem.focus(o);
        return;
      }
    }
    super.focus(o);
  }
};
var cards = (config, ...items) => createComponent(new CardContainer(), config, items);

// goui/script/component/picker/RecurrencePicker.ts
var _RecurrencePicker = class _RecurrencePicker extends CardContainer {
  constructor(startDate) {
    super();
    this.baseCls = "recurrencepicker";
    this.weekOfMonth = 1;
    this.startDate = startDate;
    const weeklyOptions = checkboxgroup({
      itemId: "weeklyOptions",
      label: "Weekdays",
      options: [0, 1, 2, 3, 4, 5, 6].map((i) => {
        return { label: DateTime.dayNames[DateTime.dayMap[i]].substring(0, 2), name: DateTime.dayMap[i] };
      })
    });
    this.menu = comp({}, ...this.quickMenuItems());
    const intervalField = numberfield({
      decimals: 0,
      name: "interval",
      itemId: "interval",
      min: 1,
      width: 70,
      value: 1
    });
    const frequencyField = select({
      name: "frequency",
      itemId: "frequency",
      width: 140,
      textRenderer: (r) => intervalField.value == 1 ? r.text : r.plural,
      options: Object.keys(_RecurrencePicker.frequencies).map((k) => ({
        value: k,
        text: _RecurrencePicker.frequencies[k][0],
        plural: _RecurrencePicker.frequencies[k][1]
      })),
      listeners: {
        "change": (me, v) => {
          this.changeFrequency(v);
        }
      }
    });
    intervalField.on("setvalue", (me, newVal, oldVal) => {
      if (oldVal == 1 && newVal != 1 || oldVal != 1 && newVal == 1) {
        frequencyField.drawOptions();
      }
    });
    this.count = numberfield({
      itemId: "repeatCount",
      name: "count",
      hidden: true,
      max: 1e3,
      width: 80,
      value: 13,
      decimals: 0,
      hint: t("times")
      // should be suffix
    });
    this.until = datefield({
      itemId: "endDate",
      name: "until",
      min: this.startDate.format("Y-m-d"),
      width: 180,
      hidden: true,
      required: false
    });
    this.form = form(
      {},
      comp(
        { cls: "flow pad" },
        comp({ text: t("Every"), width: 50, style: { alignSelf: "center" } }),
        intervalField,
        frequencyField,
        select({
          hidden: true,
          disabled: true,
          name: "monthlyType",
          itemId: "monthlyOptions",
          label: t("at the"),
          width: 160,
          value: "byMonthDay",
          options: [
            { value: "byMonthDay", name: this.startDate.format("jS") },
            { value: "byDay", name: this.getSuffix() + " " + this.startDate.format("l") }
          ]
        }),
        weeklyOptions,
        textfield({
          hidden: true,
          name: "byDay",
          listeners: {
            "change": (fld, val) => {
              for (let j = 0; j < 7; j++) {
                const cb = weeklyOptions.items.get(j);
                cb.value = val.indexOf(cb.name) !== -1;
              }
            }
          }
        }),
        comp(
          { cls: "flow" },
          comp({ html: t("Ends"), width: 50, style: { alignSelf: "center" } }),
          select({
            width: 100,
            name: "endsRadio",
            value: "forever",
            textRenderer: (r) => r.text,
            options: [
              { text: t("Never"), value: "forever" },
              { text: t("After"), value: "count" },
              { text: t("At"), value: "until" }
            ],
            listeners: {
              "setvalue": (me, v) => {
                this.count.hidden = this.count.disabled = v != "count";
                this.until.hidden = this.until.disabled = v != "until";
              }
            }
          }),
          this.count,
          this.until
        )
      ),
      tbar(
        {},
        btn({
          text: t("Back"),
          handler: (b) => {
            this.activeItem = 0;
          }
        }),
        comp({ flex: 1 }),
        btn({
          text: t("Ok"),
          handler: (btn2) => {
            this.setValue(this.createCustomRule(this.form.value));
            this.parent.close();
            this.activeItem = 0;
          }
        })
      )
    );
    this.items.add(this.menu, this.form);
  }
  quickMenuItems() {
    return [
      btn({
        text: t("Not recurring"),
        handler: (_) => this.setValue(void 0)
      }),
      comp({ tagName: "hr" }),
      btn({
        text: t("Daily"),
        handler: (_) => this.setValue({ frequency: "daily" })
      }),
      btn({
        text: t("Weekly") + " " + t("at ") + this.startDate.format("l"),
        handler: (_) => this.setValue({ frequency: "weekly" })
      }),
      btn({
        text: t("Monthly") + " " + t("at day") + " " + this.startDate.format("j"),
        handler: (_) => this.setValue({ frequency: "monthly", byMonthDay: [+this.startDate.format("j")] })
      }),
      btn({
        text: t("Monthly") + " " + t("at the") + " " + this.getSuffix() + " " + this.startDate.format("l"),
        handler: (_) => this.setValue({
          frequency: "monthly",
          byDay: [{ day: DateTime.dayMap[this.startDate.getWeekDay()], nthOfPeriod: this.weekOfMonth }]
        })
      }),
      btn({
        text: t("Annually") + " " + t("at ") + this.startDate.format("j F"),
        handler: (_) => this.setValue({ frequency: "yearly" })
      }),
      btn({
        text: t("Each working day"),
        handler: (_) => this.setValue({
          frequency: "weekly",
          byDay: [{ day: "mo" }, { day: "tu" }, { day: "we" }, { day: "th" }, { day: "fr" }]
        })
      }),
      comp({ tagName: "hr" }),
      btn({
        text: t("Customize") + "...",
        handler: () => {
          var _a;
          this.changeFrequency(((_a = this.rule) == null ? void 0 : _a.frequency) || "yearly");
          this.height = this.height;
          this.activeItem = 1;
        }
      })
    ];
  }
  createCustomRule(values) {
    const rule = { frequency: values.frequency };
    if (values.interval != 1)
      rule.interval = values.interval;
    if (values.until && values.endsRadio === "until")
      rule.until = values.until;
    if (values.count && values.endsRadio === "count")
      rule.count = values.count;
    if (values.monthlyType) {
      switch (values.monthlyType) {
        case "byMonthDay":
          rule.byMonthDay = [parseInt(this.startDate.format("j"))];
          break;
        case "byDay":
          rule.byDay = [{
            day: DateTime.dayMap[this.startDate.getWeekDay()],
            nthOfPeriod: this.weekOfMonth
          }];
          break;
      }
    }
    if (rule.frequency === "weekly") {
      ["mo", "tu", "we", "th", "fr", "sa", "su"].forEach((day) => {
        var _a;
        if (values[day])
          ((_a = rule.byDay) != null ? _a : rule.byDay = []).push({ day });
      });
    }
    return rule;
  }
  setStartDate(date) {
    this.startDate = date.clone();
    for (var i = 0, m = date.getMonth(); m == date.getMonth(); date = date.addDays(-7)) {
      i++;
    }
    this.weekOfMonth = i;
    this.menu.items.clear().add(...this.quickMenuItems());
  }
  changeFrequency(f) {
    const record = _RecurrencePicker.frequencies[f];
    this.form.findChild("frequency").value = f;
    var repeat = this.count;
    if (repeat.disabled) {
      repeat.value = record[2];
    }
    var until = this.until;
    if (until.disabled) {
      var add = record[3].split("-");
      add[1] == "d" ? this.startDate.addDays(parseInt(add[0])) : this.startDate.addYears(parseInt(add[0]));
      until.value = this.startDate.format("Y-m-d");
    }
    ["weekly", "monthly"].map((p2) => {
      const el = this.form.findChild(p2 + "Options");
      el.hidden = p2 != f;
      el.disabled = p2 != f;
    });
  }
  getSuffix(week) {
    week = week || this.weekOfMonth;
    switch (week) {
      case 1:
        return t("first");
      case 2:
        return t("second");
      case 3:
        return t("third");
      case 4:
        return t("fourth");
      default:
        return t("last");
    }
  }
  setValue(rrule) {
    if (this.rule == rrule)
      return;
    this.rule = rrule;
    this.fire("select", this, rrule);
    const form2 = this.form;
    if (rrule && rrule.frequency) {
      form2.value = rrule;
      this.changeFrequency(rrule.frequency);
      if (rrule.until) {
        form2.findField("endsRadio").value = "until";
        this.until.value = rrule.until;
      } else if (rrule.count) {
        form2.findField("endsRadio").value = "count";
        this.count.value = rrule.count;
      }
      if (rrule.byDay) {
        if (rrule.frequency === "weekly") {
          rrule.byDay.forEach((nDay) => {
            form2.findField(nDay.day).value = true;
          });
        } else if (rrule.frequency === "monthly") {
          form2.findField("monthlyOptions").value = "byDay";
        }
      }
      if (rrule.byMonthDay) {
        form2.findField("monthlyOptions").value = "byMonthDay";
      }
    }
  }
};
_RecurrencePicker.frequencies = {
  "daily": [t("day"), t("days"), 30, "30-d", t("Daily")],
  "weekly": [t("week"), t("weeks"), 13, "91-d", t("Weekly")],
  "monthly": [t("month"), t("months"), 12, "1-y", t("Monthly")],
  "yearly": [t("year"), t("years"), 5, "5-y", t("Annually")]
};
var RecurrencePicker = _RecurrencePicker;

// goui/script/component/picker/ListPicker.ts
var ListPicker = class extends Component {
  constructor(list2) {
    super();
    this.list = list2;
    this.list.rowSelectionConfig = { multiSelect: false };
    this.items.add(list2);
    this.list.on("rowclick", (table2, rowIndex, row, ev) => {
      this.onSelect();
    });
    this.list.on("hide", () => {
      this.list.rowSelection.clear();
    });
    this.list.store.on("datachanged", () => {
      if (this.list.store.count() > 0) {
        this.list.rowSelection.selected = [0];
      }
    }, { buffer: 0 });
    this.list.el.addEventListener("mousedown", (ev) => {
      ev.stopPropagation();
    });
    this.list.el.addEventListener("keydown", (ev) => {
      switch (ev.key) {
        case "Enter":
          ev.preventDefault();
          this.onSelect();
          break;
      }
    });
  }
  focus(o) {
    this.list.focus(o);
  }
  onSelect() {
    const selected = this.list.rowSelection.selected;
    if (selected.length) {
      this.fire("select", this, this.list.store.get(selected[0]));
    }
  }
};
var listpicker = (config) => createComponent(new ListPicker(config.list), config);

// goui/script/component/menu/ColorMenu.ts
var ColorMenu = class _ColorMenu extends Menu {
  constructor() {
    super();
    this._value = "";
    /**
     * Set color value as button text color if this menu belongs to a button
     */
    this.updateButton = true;
    this.baseCls = "goui-dropdown goui-menu-color";
    this.colors = [
      "000000",
      //Black
      "FFFFFF",
      //White
      "009BC9",
      //Group-Office blue
      //'243A80', //Intermesh blue
      "78A22F",
      //default secondary
      "FF9100",
      //Default accent
      "B71C1C",
      "C62828",
      "D32F2F",
      "E53935",
      "F44336",
      // Red
      "880E4F",
      "AD1457",
      "C2185B",
      "D81B60",
      "E91E63",
      // Pink
      "4A148C",
      "6A1B9A",
      "7B1FA2",
      "8E24AA",
      "9C27B0",
      // Purple
      "311B92",
      "4527A0",
      "512DA8",
      "5E35B1",
      "673AB7",
      // Deep purple
      "1A237E",
      "283593",
      "303F9F",
      "3949AB",
      "3F51B5",
      // Indigo
      "0D47A1",
      "1565C0",
      "1976D2",
      "1E88E5",
      "2196F3",
      // Blue
      "01579B",
      "0277BD",
      "0288D1",
      "039BE5",
      "03A9F4",
      // Light blue
      "006064",
      "00838F",
      "0097A7",
      "00ACC1",
      "00BCD4",
      // Cyan
      "004D40",
      "00695C",
      "00796B",
      "00897B",
      "009688",
      // Teal
      "1B5E20",
      "2E7D32",
      "388E3C",
      "43A047",
      "4CAF50",
      // Green
      "33691E",
      "558B2F",
      "689F38",
      "7CB342",
      "8BC34A",
      // Light Green
      "827717",
      "9E9D24",
      "AFB42B",
      "C0CA33",
      "CDDC39",
      // Lime
      "F57F17",
      "F9A825",
      "FBC02D",
      "FDD835",
      "FFEB3B",
      // Yellow
      "FF6F00",
      "FF8F00",
      "FFA000",
      "FFB300",
      "FFC107",
      // Amber
      "E65100",
      "EF6C00",
      "F57C00",
      "FB8C00",
      "FF9800",
      // Orange
      "212121",
      "424242",
      "616161",
      "757575",
      "BDBDBD"
      // Grey
    ];
    this.items.add(btn({
      itemId: "auto",
      text: "Auto",
      cls: this.value == "" ? "pressed" : "",
      handler: () => {
        this.value = "";
        this.fire("select", this, "");
      }
    }));
    this.colors.forEach((color) => {
      this.items.add(btn({
        itemId: "#" + color,
        cls: this.value == color ? "with-icon pressed" : "with-icon",
        listeners: {
          beforerender: (btn2) => {
            const colorDiv = document.createElement("div");
            colorDiv.style.backgroundColor = "#" + color;
            btn2.el.appendChild(colorDiv);
          }
        },
        handler: (btn2) => {
          this.value = btn2.itemId;
          this.fire("select", this, btn2.itemId);
        }
      }));
    });
  }
  static rgb2hex(str) {
    const rgb = str.match(/^rgb\((\d+),\s*(\d+),\s*(\d+)\)$/i);
    if (!rgb) {
      return "";
    }
    return "#" + ((1 << 24) + (parseInt(rgb[1]) << 16) + (parseInt(rgb[2]) << 8) + parseInt(rgb[3])).toString(16).slice(1);
  }
  /**
   * Color hex value eg. #000000
   */
  set value(color) {
    const hex = _ColorMenu.rgb2hex(color);
    if (hex) {
      color = hex.toUpperCase();
    } else {
      color = color.toUpperCase();
    }
    this._value = color;
    if (this.updateButton) {
      this.parent.el.style.color = color;
    }
    if (!this.rendered) {
      return;
    }
    if (color == "") {
      color = "auto";
    }
    this.items.forEach((btn2) => {
      btn2.el.classList.toggle("pressed", btn2.itemId == color);
    });
  }
  get value() {
    return this._value;
  }
};
var colormenu = (config) => createComponent(new ColorMenu(), config);

// goui/script/component/form/AutocompleteField.ts
var AutocompleteField = class extends TextField {
  /**
   *
   * @param list The table to use for suggestions
   * @param buffer Buffer typing in the input in ms
   */
  constructor(list2, buffer = 300) {
    super();
    this.list = list2;
    this.buffer = buffer;
    this.autocomplete = "off";
    this.baseCls += " autocomplete";
    this.picker = listpicker({
      list: list2
    });
    this.picker.on("select", (tablePicker, record) => {
      tablePicker.list.findAncestorByType(Menu).hide();
      this.focus();
      this.value = this.pickerRecordToValue(this, record);
      this.fire("select", this, record);
    });
    this.menu = menu(
      {
        height: 300,
        cls: "scroll",
        listeners: {
          hide: (menu2) => {
            if (menu2.rendered) {
              const textfield2 = menu2.findAncestorByType(TextField);
              textfield2.focus();
            }
          }
        }
      },
      this.picker
    );
    this.menuButton = btn({
      icon: "expand_more",
      type: "button",
      handler: () => {
        this.fire("autocomplete", this, "");
      },
      menu: this.menu
    });
  }
  /**
   * Method that transforms a record from the TablePicker store to a value for this field.
   * This is not necessarily a text value. In conjunction with {@see valueToTextField()} this
   * could also be an ID of an object for example.
   *
   * @param field
   * @param record
   */
  pickerRecordToValue(field, record) {
    return record.id;
  }
  /**
   * This method transforms the value in to a text representation for the input field
   *
   * @param field
   * @param value
   */
  valueToTextField(field, value) {
    return __async(this, null, function* () {
      return "";
    });
  }
  internalSetValue(v) {
    if (v == void 0) {
      return super.internalSetValue(v);
    }
    this.valueToTextField(this, v + "").then((v2) => {
      if (this.input) {
        super.internalSetValue(v2);
      } else {
        this.on("render", () => {
          super.internalSetValue(v2);
        }, { once: true });
      }
    });
  }
  get value() {
    return this._value;
  }
  set value(v) {
    super.value = v;
  }
  internalRender() {
    this.buttons = this.buttons || [];
    this.buttons.push(this.menuButton);
    const el = super.internalRender();
    this.menu.alignTo = this.wrap;
    this.menu.alignToInheritWidth = true;
    this.input.addEventListener("input", FunctionUtil.buffer(this.buffer, this.onInput.bind(this)));
    this.input.addEventListener("keydown", (ev) => {
      switch (ev.key) {
        case "Enter":
          if (!this.menu.hidden) {
            ev.preventDefault();
            this.picker.onSelect();
          }
          break;
        case "ArrowDown":
          ev.preventDefault();
          this.fire("autocomplete", this, this.input.value);
          this.menuButton.showMenu();
          this.list.focus();
          break;
        case "Escape":
          if (!this.menu.hidden) {
            this.menu.hide();
            ev.preventDefault();
            ev.stopPropagation();
            this.focus();
          }
          break;
      }
    });
    return el;
  }
  onInput(ev) {
    this.menuButton.showMenu();
    this.fire("autocomplete", this, this.input.value);
  }
};
var autocomplete = (config) => createComponent(new AutocompleteField(config.list), config);

// goui/script/component/form/CheckboxField.ts
var CheckboxField = class extends InputField {
  /**
   *
   * @param type Render the checkbox as a checkbox, switch or toggle button
   */
  constructor(type = "box") {
    super();
    this.baseCls = "goui-form-field check";
    // we handle it with the native change event here
    this.fireChangeOnBlur = false;
    this.el.cls(type, true);
    this.type = "checkbox";
  }
  applyTitle() {
    if (this.title) {
      this.input.title = this.title;
    }
  }
  focus(o) {
    var _a;
    if (!this.input) {
      super.focus(o);
    }
    (_a = this.input) == null ? void 0 : _a.focus(o);
  }
  createLabel() {
    const lbl = E("span").cls("box-label");
    lbl.innerHTML = this.label;
    if (this._cls) {
      lbl.cls("+" + this._cls);
    }
    this.control.append(lbl);
    this._labelEl = lbl;
  }
  createControl() {
    this._input = this.createInput();
    this._input.on("change", () => this.fireChange());
    this._input.on("click", () => this.validate());
    const control = E(
      "div",
      this._input
    );
    return control;
  }
  set color(v) {
    this.input.style.backgroundColor = v || "";
  }
  get color() {
    return this.input.style.backgroundColor;
  }
  set value(v) {
    if (this.input) {
      this.input.checked = v;
    }
    super.value = v;
  }
  get value() {
    if (!this.input) {
      return !!super.value;
    } else {
      return this.input.checked;
    }
  }
};
var checkbox = (config) => {
  const type = (config == null ? void 0 : config.type) || "box";
  config == null ? true : delete config.type;
  return createComponent(new CheckboxField(type), config);
};

// goui/script/component/form/ColorField.ts
var ColorField = class extends Field {
  constructor() {
    super();
    this.picker = this.createPicker();
    this.buttons = [
      this.pickerButton = btn({
        icon: "expand_more",
        menu: menu(
          {
            alignTo: this.el,
            alignToInheritWidth: true
          },
          this.picker
        )
      })
    ];
  }
  createPicker() {
    const picker = new ColorPicker();
    picker.on("select", (colorPicker, val) => {
      this.pickerButton.menu.hide();
      this.clearInvalid();
      this.focus();
      this.value = val;
    });
    return picker;
  }
  createControl() {
    const ctrl = E("div").cls("+color-dot");
    this.el.cls("+no-floating-label");
    return ctrl;
  }
  setInvalid(msg) {
    super.setInvalid(msg);
    if (this.rendered) {
      this.applyInvalidMsg();
    }
  }
  clearInvalid() {
    super.clearInvalid();
    this.applyInvalidMsg();
  }
  set value(v) {
    this.control.style.backgroundColor = "#" + v;
    super.value = v;
  }
  get value() {
    return super.value;
  }
};
var colorfield = (config) => createComponent(new ColorField(), config);

// goui/script/component/form/DisplayField.ts
var defaultDisplayFieldRenderer = (v, field) => v != null ? v : "";
var DisplayField = class extends Field {
  /**
   *
   * @param renderer Renderer function for the value of the field
   */
  constructor(renderer = defaultDisplayFieldRenderer) {
    super();
    this.renderer = renderer;
    this.baseCls = "goui-display-field";
    /**
     * Escape value HTML
     *
     * {@link Format.escapeHTML}
     */
    this.escapeValue = true;
    /**
     * Hide this field when the value is empty
     */
    this.hideWhenEmpty = true;
  }
  createControl() {
    return document.createElement("div");
  }
  internalSetValue(v) {
    if (this.control) {
      const setFn = (str2) => {
        if (this.escapeValue) {
          str2 = Format.escapeHTML(str2);
        }
        this.control.innerHTML = str2;
        if (this.hideWhenEmpty)
          this.hidden = str2 == "";
      }, str = this.renderer(v, this);
      str instanceof Promise ? str.then(setFn) : setFn(str);
    }
  }
};
var displayfield = (config, ...items) => {
  var _a;
  return createComponent(new DisplayField((_a = config == null ? void 0 : config.renderer) != null ? _a : defaultDisplayFieldRenderer), config, items);
};
var displaydatefield = (config, ...items) => {
  var _a;
  if (!config.icon)
    config.icon = "today";
  if (!config.renderer)
    config.renderer = (v) => v ? new DateTime(v).format(Format.dateFormat) : "";
  return createComponent(new DisplayField((_a = config == null ? void 0 : config.renderer) != null ? _a : defaultDisplayFieldRenderer), config, items);
};

// goui/script/component/form/Fieldset.ts
var Fieldset = class extends Component {
  constructor() {
    super("fieldset");
    this.baseCls = "goui-fieldset";
    this.cls = "flow";
  }
  internalRender() {
    const el = super.internalRender();
    if (this.legend) {
      const l = document.createElement("h3");
      l.innerHTML = this.legend;
      l.classList.add("legend");
      el.insertBefore(l, el.firstChild);
    }
    return el;
  }
};
var fieldset = (config, ...items) => createComponent(new Fieldset(), config, items);

// goui/script/component/form/HtmlField.ts
var _HtmlField = class _HtmlField extends Field {
  constructor() {
    super("div");
    this.baseCls = "goui-form-field goui-html-field";
    /**
     * Toolbar items to enable.
     *
     * If you can't use inline css then use:
     *
     * ```
     * [
     * 		"bold", "italic", "underline",
     * 		"-",
     * 		"foreColor", "removeFormat",
     * 		"-",
     * 		"insertOrderedList",
     * 		"insertUnorderedList",
     * 		"-",
     * 		"indent",
     * 		"outdent",
     * 		"-",
     * 		"image"
     * 	]
     * ```
     */
    this.toolbarItems = [
      "bold",
      "italic",
      "underline",
      "strikeThrough",
      "-",
      "foreColor",
      "backColor",
      "removeFormat",
      "-",
      "justifyLeft",
      "justifyCenter",
      "justifyRight",
      "-",
      "insertOrderedList",
      "insertUnorderedList",
      "-",
      "indent",
      "outdent",
      "-",
      "image",
      "createLink"
    ];
    this.commands = {
      bold: { icon: "format_bold", title: "Bold" },
      italic: { icon: "format_italic", title: "Italic" },
      underline: { icon: "format_underlined", title: "Underline" },
      strikeThrough: { icon: "format_strikethrough", title: "Strikethrough" },
      foreColor: {
        title: "Text color",
        icon: "format_color_text",
        menu: colormenu({
          updateButton: false,
          listeners: {
            select: (menu2, color) => {
              this.execCmd("foreColor", color || "#000000");
            },
            beforeshow: (menu2) => {
              menu2.value = this.tbar.findItem("foreColor").el.style.color || "";
            }
          }
        }),
        applyFn: () => {
        },
        updateFn: (btn2) => {
          const s = document.getSelection();
          if (!s || !s.anchorNode) {
            btn2.el.style.color = "";
            return;
          }
          let el = s.anchorNode;
          while (!(el instanceof HTMLElement) && el != null) {
            el = el.parentElement;
          }
          if (!el) {
            btn2.el.style.color = "";
            return;
          }
          const closest = el.closest("[color]");
          if (!closest) {
            btn2.el.style.color = "";
            return;
          }
          btn2.el.style.color = closest.getAttribute("color") + "";
        }
      },
      backColor: {
        icon: "format_color_fill",
        title: "Background color",
        menu: colormenu({
          updateButton: false,
          listeners: {
            select: (menu2, color) => {
              this.execCmd("backColor", color || "#ffffff");
            },
            beforeshow: (menu2) => {
              menu2.value = this.tbar.findItem("backColor").el.style.color || "";
            }
          }
        }),
        applyFn: () => {
        },
        updateFn: (btn2) => {
          const s = document.getSelection();
          let bgcolor = s ? this.closestStyle(s.anchorNode, "backgroundColor") : void 0;
          btn2.el.style.backgroundColor = bgcolor || "";
        }
      },
      justifyLeft: { icon: "format_align_left", title: "Align left" },
      justifyCenter: { icon: "format_align_center", title: "Align center" },
      justifyRight: { icon: "format_align_right", title: "Align right" },
      insertOrderedList: { icon: "format_list_numbered", title: "Numbered list" },
      insertUnorderedList: { icon: "format_list_bulleted", title: "Bullet list" },
      indent: { icon: "format_indent_increase", title: "Indent" },
      outdent: { icon: "format_indent_decrease", title: "Outdent" },
      createLink: {
        icon: "link",
        title: "Create link",
        applyFn: () => {
          const url = prompt(t("Enter URL"), "https://");
          if (url)
            this.execCmd("createLink", url);
        }
      },
      image: {
        icon: "image",
        title: "Image",
        applyFn: () => {
          const inp = document.createElement("input");
          inp.setAttribute("type", "file");
          inp.accept = "image/*";
          inp.multiple = true;
          inp.onchange = () => {
            if (!inp.files) {
              this.editor.focus();
              return;
            }
            Array.from(inp.files).forEach((file) => {
              this.handleImage(file);
            });
          };
          window.addEventListener("focus", () => {
            this.editor.focus();
          }, { once: true });
          inp.click();
        }
      },
      removeFormat: { icon: "format_clear", title: "Remove formatting" }
      // emoji: {icon: 'insert_emoticon'},
      // html: {icon: 'code'}
    };
    this.lineIndex = 0;
    this.lineSequence = "";
    this.value = "";
    document.execCommand("useCSS", false, "true");
    document.execCommand("styleWithCSS", false, "false");
  }
  // noinspection JSUnusedGlobalSymbols
  getEditor() {
    if (!this.editor) {
      throw new Error("Render first");
    }
    return this.editor;
  }
  closestStyle(el, styleProp) {
    while (el = el.parentElement) {
      if (el == this.editor)
        return void 0;
      if (el.style[styleProp]) {
        return el.style[styleProp];
      }
    }
    return void 0;
  }
  updateToolbar() {
    const t2 = this.tbar;
    for (const cmd of this.toolbarItems) {
      if (cmd == "-")
        continue;
      const config = this.commands[cmd];
      if (config.updateFn) {
        config.updateFn.call(this, t2.findItem(cmd));
      } else {
        t2.findItem(cmd).el.classList.toggle("pressed", document.queryCommandState(cmd));
      }
    }
    this.fire("updatetoolbar", this);
  }
  execCmd(cmd, value) {
    document.execCommand(cmd, false, value);
    const t2 = this.tbar, config = this.commands[cmd];
    if (config) {
      if (config.updateFn) {
        config.updateFn.call(this, t2.findItem(cmd));
      } else {
        t2.findItem(cmd).el.classList.toggle("pressed", document.queryCommandState(cmd));
      }
    }
    this.editor.focus();
  }
  /**
   * Inserts an HTML string at the insertion point (deletes selection). Requires a valid HTML string as a value argument.
   *
   * @param html
   */
  insertHtml(html2) {
    document.execCommand("insertHTML", false, html2);
  }
  // noinspection JSUnusedGlobalSymbols
  internalRemove() {
    if (this.tbar) {
      this.tbar.remove();
    }
    super.internalRemove();
  }
  getToolbar() {
    if (this.tbar) {
      return this.tbar;
    }
    this.tbar = tbar({
      cls: "frame html-field-toolbar"
    });
    for (const cmd of this.toolbarItems) {
      if (cmd == "-") {
        this.tbar.items.add(comp({ tagName: "hr" }));
      } else {
        const config = this.commands[cmd];
        this.tbar.items.add(btn({
          itemId: cmd,
          icon: config.icon,
          menu: config.menu,
          title: config.title,
          tabIndex: 0,
          //needed for Safari. Otherwise the blur event doesn't have this button as relatedTarget.
          // See in this.editor.addEventListener("blur", below
          handler: (btn2) => {
            if (!config.applyFn) {
              this.execCmd(btn2.itemId);
            } else {
              config.applyFn.call(this, btn2);
            }
          }
        }));
      }
    }
    root.items.add(this.tbar);
    return this.tbar;
  }
  showToolbar() {
    const rect = this.el.getBoundingClientRect();
    this.getToolbar().show();
    const h = this.getToolbar().height;
    const style = this.getToolbar().el.style;
    const maxX = window.innerWidth - this.getToolbar().width;
    style.left = Math.min(maxX, rect.x + window.scrollX) + "px";
    style.top = window.scrollY + rect.top - h - 8 + "px";
  }
  hideToolbar() {
    this.getToolbar().hide();
  }
  createControl() {
    const el = this.el;
    el.addEventListener("click", () => {
      this.editor.focus();
    });
    const v = this.value;
    this.editor = document.createElement("div");
    this.editor.contentEditable = "true";
    this.editor.classList.add("editor");
    this.editor.classList.add("text");
    if (this.placeholder) {
      this.editor.dataset.placeholder = this.placeholder;
    }
    if (v) {
      this.editor.innerHTML = v;
    }
    el.appendChild(this.editor);
    const selectionChangeFn = FunctionUtil.buffer(300, () => this.updateToolbar());
    this.editor.addEventListener("focus", () => {
      document.addEventListener("selectionchange", selectionChangeFn);
      this.showToolbar();
    });
    this.editor.addEventListener("blur", (e) => {
      document.removeEventListener("selectionchange", selectionChangeFn);
      if (!(e.relatedTarget instanceof HTMLElement) || !this.getToolbar().el.contains(e.relatedTarget)) {
        this.hideToolbar();
      } else {
        this.editor.focus();
      }
    });
    this.editor.addEventListener("keydown", (ev) => {
      this.onKeyDown(ev);
    });
    this.editor.addEventListener("drop", (ev) => {
      this.onDrop(ev);
    });
    this.editor.addEventListener("dragover", (ev) => {
      ev.preventDefault();
    });
    this.editor.addEventListener("paste", (ev) => {
      this.onPaste(ev);
    });
    return this.editor;
  }
  // private isChildOfEditor(el:Node): boolean {
  // 	let p = el.parentNode;
  // 	if(p == this.editor) {
  // 		return true;
  // 	}
  //
  // 	if(!p || p == document.body) {
  // 		return false;
  // 	}
  //
  // 	return this.isChildOfEditor(p);
  //
  // }
  set value(v) {
    if (this.editor) {
      this.editor.innerHTML = v;
    }
    super.value = v;
  }
  get value() {
    if (!this.editor) {
      return super.value;
    } else {
      return this.editor.innerHTML;
    }
  }
  focus(o) {
    this.editor.focus(o);
  }
  static removeCharsFromCursorPos(count) {
    const sel = window.getSelection();
    const range = sel.getRangeAt(0);
    const clone = range.cloneRange();
    clone.setStart(range.startContainer, range.startOffset);
    clone.setEnd(range.startContainer, range.startOffset + count);
    clone.deleteContents();
  }
  onKeyDown(ev) {
    this.clearInvalid();
    if (ev.key == "Enter" || ev.key == "Backspace") {
      this.lineIndex = 0;
      this.lineSequence = "";
    } else if (this.lineIndex < 3 && ev.key.length == 1) {
      this.lineIndex++;
      this.lineSequence += ev.key;
    }
    if (browser.isWebkit() && ev.shiftKey && ev.key == "Enter" && (document.queryCommandState("insertorderedlist") || document.queryCommandState("insertunorderedlist"))) {
      ev.stopPropagation();
      this.execCmd("InsertHtml", browser.isFirefox() ? "<br />" : "<br /><br />");
      this.focus();
    } else if (ev.key == "Tab") {
      ev.preventDefault();
      if (document.queryCommandState("insertorderedlist") || document.queryCommandState("insertunorderedlist")) {
        this.execCmd(ev.shiftKey ? "outdent" : "indent");
      } else {
        this.execCmd("InsertText", "	");
      }
      this.focus();
    } else if (ev.key == " ") {
      if (this.lineSequence == "1. ") {
        this.execCmd("insertOrderedList");
        _HtmlField.removeCharsFromCursorPos(2);
        ev.preventDefault();
      } else if (this.lineSequence == "- ") {
        this.execCmd("insertUnorderedList");
        _HtmlField.removeCharsFromCursorPos(1);
        ev.preventDefault();
      }
    } else if (browser.isMac()) {
      if (ev.ctrlKey || ev.metaKey) {
        let cmd;
        switch (ev.key) {
          case "b":
            cmd = "bold";
            break;
          case "i":
            cmd = "italic";
            break;
          case "u":
            cmd = "underline";
            break;
        }
        if (cmd) {
          this.execCmd(cmd);
          ev.preventDefault();
        }
      }
    }
  }
  onDrop(e) {
    if (!e.dataTransfer || !e.dataTransfer.files) {
      return;
    }
    e.preventDefault();
    this.editor.focus();
    Array.from(e.dataTransfer.files).forEach((file) => {
      if (file.type.match(/^image\//)) {
        this.handleImage(file);
      } else {
        this.fire("attach", this, file);
      }
    });
  }
  /**
   * Generate unique ID
   */
  static imgUID() {
    return "img-" + ++_HtmlField._uid;
  }
  handleImage(file) {
    let imgEl;
    if (file.type.match(/^image\//)) {
      const objectURL = URL.createObjectURL(file), uid = _HtmlField.imgUID();
      const img2 = `<img id="${uid}" src="${objectURL}" alt="${file.name}" />`;
      this.insertHtml(img2);
      imgEl = document.getElementById(uid);
      imgEl.removeAttribute("id");
      imgEl.addEventListener("load", () => {
        const width = imgEl.offsetWidth, height = imgEl.offsetHeight;
        imgEl.setAttribute("style", `max-width: 100%;height:auto;aspect-ratio: ${width} / ${height};`);
      });
      this.fire("insertimage", this, file, imgEl);
    }
  }
  onPaste(e) {
    if (!e.clipboardData || !e.clipboardData.files) {
      return;
    }
    const files = Array.from(e.clipboardData.files);
    files.forEach((file) => {
      if (file.type.match(/^image\//)) {
        this.handleImage(file);
      } else {
        this.fire("attach", this, file);
      }
      e.preventDefault();
    });
  }
};
_HtmlField._uid = 0;
var HtmlField = _HtmlField;
var htmlfield = (config) => createComponent(new HtmlField(), config);

// goui/script/component/form/MapField.ts
var MapField = class extends Field {
  constructor(buildField) {
    super("div");
    this.buildField = buildField;
    this.baseCls = "";
    this._nextKey = 0;
    this.cls = "vbox gap";
  }
  renderControl() {
  }
  set value(v) {
    super.value = v;
    this.items.clear();
    if (v) {
      for (const key in v) {
        this.internalAdd(v[key], key);
      }
    }
  }
  get value() {
    const v = {};
    this.items.forEach((field) => {
      if (field instanceof Field) {
        const rowValue = field.value;
        let key = field.dataSet.key;
        if (this.keyFieldName && rowValue[this.keyFieldName]) {
          key = rowValue[this.keyFieldName];
          delete rowValue[this.keyFieldName];
        }
        v[key] = rowValue;
      }
    });
    return v;
  }
  isEmpty() {
    return this.items.count() === 0;
  }
  /**
   * Add value to the map.
   *
   * Also fires change event
   *
   * @param data
   * @param key
   */
  add(data, key) {
    if (!this.valueOnFocus) {
      this.captureValueForChange();
    }
    this.internalAdd(data, key);
    this.fireChange();
  }
  internalAdd(data, key) {
    if (typeof key === "number") {
      this._nextKey = Math.max(this._nextKey, key);
    }
    const field = this.buildField(data);
    field.dataSet.key = key || this.nextKey();
    field.value = data;
    this.items.add(field);
  }
  nextKey() {
    return ++this._nextKey;
  }
  reset() {
    super.reset();
    this.items.clear();
  }
};
var mapfield = (config, ...items) => createComponent(new MapField(config.buildField), config, items);

// goui/script/component/form/RadioField.ts
var RadioField = class extends Field {
  constructor(type = "box") {
    super("div");
    this.inputs = {};
    this.baseCls = "goui-form-field radiogroup";
    this.type = type;
    this.el.cls(type, true);
  }
  createLabel() {
    return;
  }
  createControl() {
    if (!this.name && !this.itemId) {
      this.itemId = "radio-" + Component.uniqueID();
    }
    this._labelEl = E("h3").cls("legend");
    this._labelEl.innerText = this.label;
    this.el.insertBefore(this._labelEl, this.wrap);
    const radio2 = E("div").cls("radio");
    if (this.invalidMsg) {
      this.applyInvalidMsg();
    }
    return radio2;
  }
  set options(options) {
    if (this._options) {
      this.control.empty();
      this.inputs = {};
    }
    options.forEach((o) => {
      const btn2 = E("input").on("change", () => {
        this.fireChange();
      });
      btn2.type = "radio";
      btn2.name = this.name || this.itemId;
      btn2.readOnly = this.readOnly;
      if (o.value) {
        btn2.value = o.value;
        if (this._value == o.value) {
          btn2.checked = true;
        }
        this.inputs[o.value] = btn2;
      }
      const lbl = E("span").cls("box-label");
      if (o.icon) {
        lbl.append(E("i", o.icon).cls("icon"));
      }
      lbl.append(o.text);
      this.control.append(E(
        "label",
        btn2,
        lbl
      ).cls("control"));
    });
  }
  get options() {
    var _a;
    return (_a = this._options) != null ? _a : [];
  }
  set value(v) {
    super.value = v;
    if (v && v in this.inputs)
      this.inputs[v].checked = true;
  }
  get value() {
    if (this.rendered) {
      for (let v in this.inputs) {
        if (this.inputs[v].checked) {
          return v;
        }
      }
    }
    return super.value;
  }
};
var radio = (config) => createComponent(new RadioField((config == null ? void 0 : config.type) || "box"), config);

// goui/script/component/form/RangeField.ts
var RangeField = class extends NumberField {
  constructor() {
    super();
    this.baseCls = "goui-form-field range no-floating-label";
    this.type = "range";
    this.value = 50;
    this.min = 0;
    this.max = 100;
  }
};
var rangefield = (config) => createComponent(new RangeField(), config);

// goui/script/component/form/RecurrenceField.ts
var RecurrenceField = class _RecurrenceField extends Field {
  constructor() {
    super();
    this.baseCls = "goui-form-field recurrence";
    this.picker = new RecurrencePicker(new DateTime());
    this.picker.on("select", (_, val) => {
      this.pickerButton.menu.hide();
      this.clearInvalid();
      this.focus();
      this.value = val;
      this.control.value = _RecurrenceField.toText(val, this.picker.startDate);
    });
    this.buttons = [
      this.pickerButton = btn({
        icon: "expand_more",
        menu: menu(
          {
            alignTo: this.el,
            alignToInheritWidth: true
          },
          this.picker
        )
      })
    ];
  }
  internalSetValue(v) {
    if (this.control) {
      this.control.value = _RecurrenceField.toText(v, this.picker.startDate);
    }
    this.picker.setValue(v);
  }
  createControl() {
    this.control = E("input").attr("type", "text").attr("readOnly", true).cls("text");
    return this.control;
  }
  setStartDate(date) {
    this.picker.setStartDate(date);
    if (this.control)
      this.control.value = _RecurrenceField.toText(this.value, this.picker.startDate);
  }
  static toText(rule, start) {
    const rr = rule;
    if (!rr || !rr.frequency) {
      return t("Not recurring");
    }
    const record = RecurrencePicker.frequencies[rr.frequency];
    if (!record) {
      return "Unsupported frequency: " + rr.frequency;
    }
    let str = record[4];
    if (rr.interval) {
      str = t("Every") + " " + rr.interval + " " + record[rr.interval > 1 ? 1 : 0];
    }
    if (rr.byDay) {
      let days = [], workdays = rr.byDay.length === 5;
      for (var i = 0; i < rr.byDay.length; i++) {
        if (rr.byDay[i].day == "sa" || rr.byDay[i].day == "su") {
          workdays = false;
        }
        var nthDay = "";
        if (rr.byDay[i].nthOfPeriod) {
          nthDay = t("the") + " " + _RecurrenceField.getSuffix(rr.byDay[i].nthOfPeriod) + " ";
        }
        days.push(nthDay + DateTime.dayNames[rr.byDay[i].day]);
      }
      if (workdays) {
        days = [t("Workdays")];
      }
      str += " " + t("at ") + days.join(", ");
    } else if (rr.frequency == "weekly") {
      str += " " + t("at ") + start.format("l");
    }
    if (rr.byMonthDay) {
      str += " " + t("at day") + " " + rr.byMonthDay.join(", ");
    }
    if (rr.count) {
      str += ", " + rr.count + " " + t("times");
    }
    if (rr.until) {
      str += ", " + t("until") + " " + new DateTime(rr.until).format("d-m-Y");
    }
    return str;
  }
  static getSuffix(week) {
    switch (week) {
      case 1:
        return t("first");
      case 2:
        return t("second");
      case 3:
        return t("third");
      case 4:
        return t("fourth");
      default:
        return t("last");
    }
  }
};
var recurrencefield = (config) => createComponent(new RecurrenceField(), config);

// goui/script/component/form/TextareaField.ts
var TextAreaField = class extends InputField {
  constructor() {
    super(...arguments);
    this.baseCls = "goui-form-field textarea";
    this._autoHeight = false;
  }
  createInput() {
    return document.createElement("textarea");
  }
  /**
   * Let the textarea grow to it's content
   *
   * @param v
   */
  set autoHeight(v) {
    if (this._autoHeight)
      return;
    this._autoHeight = true;
    const input = this._input;
    input.rows = 1;
    input.style.overflowY = "hidden";
    input.on("input", (ev) => {
      this.resize(input);
    });
    this.on("setvalue", () => {
      this.resize(input);
    });
  }
  get autoHeight() {
    return this._autoHeight;
  }
  resize(input) {
    input.style.height = "0";
    input.style.height = input.scrollHeight + "px";
  }
};
var textarea = (config) => createComponent(new TextAreaField(), config);

// goui/script/component/form/CheckboxGroup.ts
var CheckboxGroup = class extends Component {
  constructor() {
    super("fieldset");
    this.label = "";
    this.baseCls = "checkbox-group";
  }
  set options(options) {
    this.items.replace(...options.map((o) => {
      o.type = "button";
      return checkbox(o);
    }));
  }
  createLabel() {
    return;
  }
  renderItems() {
    const label = E("h3").cls("legend");
    label.innerText = this.label;
    this.el.appendChild(label);
    return super.renderItems();
  }
  get itemContainerEl() {
    if (!this._itemContainerEl) {
      this._itemContainerEl = E("div").cls("goui group");
      this.el.appendChild(this._itemContainerEl);
    }
    return this._itemContainerEl;
  }
};
var checkboxgroup = (config) => createComponent(new CheckboxGroup(), config);

// goui/script/component/form/ChipsField.ts
var ChipsField = class extends Field {
  constructor() {
    super(...arguments);
    this.baseCls = "goui-form-field chips";
    /**
     * Function that transforms the user text input to a chip.
     *
     * If it returns an empty value no chip will be rendered
     *
     * @param text
     */
    this.textInputToValue = (text) => __async(this, null, function* () {
      return text;
    });
    /**
     * Renders a value to the chip component
     * @param value
     */
    this.chipRenderer = (chip, value) => __async(this, null, function* () {
      chip.items.get(0).text = value;
    });
    this.selectedIndex = -1;
  }
  get editor() {
    return this._editor;
  }
  createControl() {
    this.chipsContainer = document.createElement("div");
    this.chipsContainer.classList.add("control-wrap");
    this._editor = comp({
      id: Component.uniqueID(),
      attr: {
        contentEditable: "true"
      },
      cls: "editor"
    });
    this.el.setAttribute("for", this._editor.id);
    this._editor.el.addEventListener("keydown", (ev) => {
      this.onEditorKeyDown(ev);
    });
    this.el.addEventListener("click", (ev) => {
      if (this.selectedIndex == -1) {
        this._editor.focus();
      }
    });
    this._editor.el.addEventListener("focus", (ev) => {
      this.clearSelection();
    });
    this.items.add(this._editor);
    return this.chipsContainer;
  }
  get value() {
    return super.value;
  }
  set value(v) {
    super.value = v;
  }
  onEditorKeyDown(ev) {
    this.clearInvalid();
    switch (ev.key) {
      case "Enter":
        ev.preventDefault();
        const chip = this.createChip();
        this.textInputToValue(this._editor.text).then((value) => {
          if (!value) {
            return;
          }
          return this.chipRenderer(chip, value).then(() => {
            this.items.insert(-1, chip);
            this.value.push(value);
          });
        });
        this._editor.text = "";
        break;
      case "ArrowRight":
        if (this.selectedIndex > -1) {
          this.select(this.selectedIndex + 1);
        }
        break;
      case "ArrowLeft":
        if (this.selectedIndex == -1) {
          if (this._editor.html == "") {
            ev.preventDefault();
            this.select(this.items.count() - 2);
          }
        } else {
          this.select(this.selectedIndex - 1);
        }
        break;
      case "Delete":
      case "Backspace":
        ev.stopPropagation();
        if (this._editor.html == "") {
          if (this.selectedIndex == -1) {
            ev.preventDefault();
            this.select(this.items.count() - 2);
          } else {
            this.items.get(this.selectedIndex).remove();
            this.value.splice(this.selectedIndex, 1);
            this.select(this.selectedIndex);
          }
        }
        break;
    }
  }
  createChip() {
    const chip = comp(
      {
        cls: "chip hbox"
      },
      comp({}),
      btn({
        title: t("Remove"),
        icon: "cancel",
        handler: (btn2) => {
          this.captureValueForChange();
          const index = this.items.indexOf(chip);
          chip.remove();
          this.value.splice(index, 1);
          this.fireChange();
        }
      })
    );
    chip.el.addEventListener("click", () => {
      this.select(this.items.indexOf(chip));
    });
    return chip;
  }
  get itemContainerEl() {
    return this.chipsContainer;
  }
  clearSelection() {
    if (this.selectedIndex > -1) {
      const item = this.items.get(this.selectedIndex);
      if (item) {
        item.el.classList.remove("selected");
      }
      this.selectedIndex = -1;
    }
  }
  select(index) {
    if (index < 0) {
      return;
    }
    this.clearSelection();
    if (index > this.items.count() - 2) {
      this._editor.focus();
      return;
    }
    this.selectedIndex = index;
    this.items.get(this.selectedIndex).el.classList.add("selected");
  }
  internalSetValue(v) {
    if (this.rendered) {
      for (let i = 0; i < this.items.count() - 1; i++) {
        this.items.removeAt(i);
      }
      while (this.items.count() > 1) {
        this.items.removeAt(0);
      }
      this.renderValue();
    }
  }
  internalRender() {
    const el = super.internalRender();
    this.renderValue();
    return el;
  }
  renderValue() {
    if (!this.value) {
      return;
    }
    this.value.forEach((v) => {
      const chip = this.createChip();
      this.chipRenderer(chip, v).then(() => {
        this.items.insert(-1, chip);
      });
    });
  }
  focus(o) {
    this.editor.focus(o);
  }
};
var chips = (config) => createComponent(new ChipsField(), config);

// goui/script/component/DraggableComponent.ts
var DraggableComponent = class extends Component {
  constructor(tagName = "div") {
    super(tagName);
    this._dragConstrainTo = window;
    /**
     * Enable dragging
     */
    this._draggable = true;
    this.onDragHandleClick = (e) => {
      e.stopPropagation();
    };
    this.onDragHandleMouseDown = (e) => {
      const target = e.target;
      if (target != this.el && (target.tagName == "BUTTON" || target.closest("BUTTON"))) {
        return;
      }
      if (e.button != 0) {
        return;
      }
      e.preventDefault();
      this.focus();
      const el = this.el, rect = el.getBoundingClientRect();
      if (this.setPosition === void 0) {
        const cmpStyle = getComputedStyle(el);
        this.setPosition = cmpStyle.position == "absolute" || cmpStyle.position == "fixed";
      }
      this.dragData = {
        startOffsetLeft: el.offsetLeft,
        startOffsetTop: el.offsetTop,
        grabOffsetLeft: e.clientX - rect.x,
        grabOffsetTop: e.clientY - rect.y,
        x: e.clientX,
        y: e.clientY,
        startX: e.clientX,
        startY: e.clientY,
        data: {}
      };
      if (this.fire("dragstart", this, this.dragData, e) !== false) {
        this.onDragStart(e);
      }
    };
    if (this.baseCls) {
      this.baseCls += " draggable";
    } else {
      this.baseCls = "draggable";
    }
    this.on("render", () => {
      if (this._draggable) {
        this.toggleDraggable(this._draggable);
      }
    }, { once: true });
  }
  /**
   * Enable or disable dragging
   */
  set draggable(draggable2) {
    this._draggable = draggable2;
    if (this.rendered) {
      this.toggleDraggable(draggable2);
    }
  }
  toggleDraggable(draggable2) {
    const dragHandle = this.getDragHandle();
    if (!dragHandle) {
      return;
    }
    if (draggable2) {
      dragHandle.classList.add("goui-draghandle");
      dragHandle.addEventListener("click", this.onDragHandleClick);
      dragHandle.addEventListener("mousedown", this.onDragHandleMouseDown);
    } else {
      dragHandle.classList.remove("goui-draghandle");
      dragHandle.removeEventListener("click", this.onDragHandleClick);
      dragHandle.removeEventListener("mousedown", this.onDragHandleMouseDown);
    }
  }
  get draggable() {
    return this._draggable;
  }
  /**
   * Returns the DOM element that can be grabbed to drag the component
   * @protected
   */
  getDragHandle() {
    return this.el;
  }
  /**
   * Constrain dragging to this element
   * @param el
   * @param pad Supply paddings
   */
  dragConstrainTo(el, pad3) {
    this._dragConstrainTo = el;
    this._dragConstrainPad = pad3;
  }
  calcConstrainBox(el, pad3) {
    let box = {
      left: 0,
      right: 0,
      bottom: 0,
      top: 0
    };
    if (el instanceof Window) {
      box.right = window.innerWidth;
      box.bottom = window.innerHeight;
    } else {
      const rect = el.getBoundingClientRect();
      box.left = rect.left;
      box.right = rect.right;
      box.bottom = rect.bottom;
      box.top = rect.top;
    }
    if (pad3) {
      if (pad3.left)
        box.left += pad3.left;
      if (pad3.right)
        box.right -= pad3.right;
      if (pad3.top)
        box.top += pad3.top;
      if (pad3.bottom)
        box.bottom -= pad3.bottom;
    }
    return box;
  }
  onDragStart(e) {
    e.preventDefault();
    this.constrainBox = this.calcConstrainBox(this._dragConstrainTo, this._dragConstrainPad);
    const onDrag = FunctionUtil.onRepaint((e2) => {
      this.onDrag(e2);
    });
    document.addEventListener("mousemove", onDrag);
    document.addEventListener("mouseup", (e2) => {
      document.removeEventListener("mousemove", onDrag);
      this.fire("drop", this, this.dragData, e2);
    }, { once: true });
  }
  onDrag(e) {
    const d = this.dragData;
    d.x = e.clientX;
    d.y = e.clientY;
    this.constrainCoords();
    if (this.setPosition) {
      this.el.style.top = d.startOffsetTop + d.y - d.startY + "px";
      this.el.style.left = d.startOffsetLeft + d.x - d.startX + "px";
    }
    this.fire("drag", this, this.dragData, e);
  }
  constrainCoords() {
    if (!this.constrainBox) {
      return;
    }
    const maxTop = this.constrainBox.bottom - this.el.offsetHeight + this.dragData.grabOffsetTop;
    const maxLeft = this.constrainBox.right - this.el.offsetWidth + this.dragData.grabOffsetLeft;
    this.dragData.y = Math.max(this.constrainBox.top + this.dragData.grabOffsetTop, Math.min(this.dragData.y, maxTop));
    this.dragData.x = Math.max(this.constrainBox.left + this.dragData.grabOffsetLeft, Math.min(this.dragData.x, maxLeft));
    return;
  }
  /**
   * Constrain the component to the given element
   *
   * @param el
   * @param pad
   */
  constrainTo(el, pad3) {
    const constraints = this.calcConstrainBox(el, pad3);
    let maxTop = constraints.bottom - this.el.offsetHeight, maxLeft = constraints.right - this.el.offsetWidth, minTop = 0, minLeft = 0;
    if (this.el.offsetTop > maxTop) {
      console.warn("Contraining to top " + maxTop);
      this.el.style.top = maxTop + "px";
    } else if (this.el.offsetTop < minTop) {
      console.warn("Contraining to top " + minTop);
      this.el.style.top = minTop + "px";
    }
    if (this.el.offsetLeft > maxLeft) {
      console.warn("Contraining to left " + maxLeft);
      this.el.style.left = maxLeft + "px";
    } else if (this.el.offsetLeft < minLeft) {
      console.warn("Contraining to left " + minLeft);
      this.el.style.left = minLeft + "px";
    }
  }
};
var draggable = (config, ...items) => createComponent(new DraggableComponent(config == null ? void 0 : config.tagName), config, items);

// goui/script/component/Window.ts
var Window2 = class _Window extends DraggableComponent {
  constructor() {
    super();
    this.baseCls = "goui-window";
    /**
     * Maximize the window
     */
    this.maximized = false;
    /**
     * Enable tool to maximize window
     */
    this.maximizable = false;
    /**
     * Enable tool to close window
     */
    this.closable = true;
    /**
     * Enable tool to collapse window
     */
    this.collapsible = false;
    /**
     * Make the window modal so the user can only interact with this window.
     */
    this.modal = false;
    this.resizable = true;
    this.width = 400;
    this.hidden = true;
  }
  // /**
  //  * Focus first item if possible.
  //  * @param o
  //  */
  // public focus(o?: FocusOptions) {
  //
  // 	const first = this.items.first();
  // 	if(first) {
  // 		return first.focus(o);
  // 	} else {
  // 		return super.focus(o);
  // 	}
  // }
  initMaximizeTool() {
    const maximizeBtn = btn({
      icon: "fullscreen",
      title: t("Maximize"),
      handler: () => {
        this.isMaximized() ? this.unmaximize() : this.maximize();
      }
    });
    this.on("maximize", () => {
      maximizeBtn.icon = "fullscreen_exit";
      maximizeBtn.title = t("Restore");
    });
    this.on("unmaximize", () => {
      maximizeBtn.icon = "fullscreen";
      maximizeBtn.title = t("Maximize");
    });
    return maximizeBtn;
  }
  set title(title) {
    if (this.titleCmp) {
      this.titleCmp.html = title;
    }
    this._title = title;
  }
  get title() {
    return this._title;
  }
  getDragHandle() {
    return this.getHeader().el;
  }
  getHeader() {
    var _a;
    if (!this.header) {
      this.header = tbar(
        {
          cls: "header"
        },
        this.titleCmp = comp({
          tagName: "h3",
          html: (_a = this.title) != null ? _a : ""
        }),
        "->"
      );
      if (this.maximizable) {
        this.header.items.add(this.initMaximizeTool());
      }
      if (this.collapsible) {
        this.header.items.add(btn({
          cls: "collapse-btn",
          icon: "",
          // set empty so 'collapsed class can set it class can set it
          handler: () => {
            this.collapsed = !this.collapsed;
          }
        }));
      }
      if (this.closable) {
        this.header.items.add(btn({
          icon: "close",
          handler: () => {
            this.close();
          }
        }));
      }
    }
    this.header.parent = this;
    return this.header;
  }
  set collapsed(collapsed) {
    this.el.classList.toggle("collapsed", collapsed);
  }
  get collapsed() {
    return this.el.classList.contains("collapsed");
  }
  internalRender() {
    const header = this.getHeader();
    header.render();
    const el = super.internalRender();
    this.on("drop", () => {
      this.saveState();
    });
    if (this.maximized) {
      this.maximize();
    }
    el.setAttribute("tabindex", "-1");
    this.el.addEventListener("keydown", (e) => {
      if (e.key == "Escape") {
        this.close();
      }
    });
    if (this.resizable) {
      this.observerResize();
    }
    if (this.modal) {
      el.classList.add("modal");
    }
    return el;
  }
  observerResize() {
    const saveState = FunctionUtil.buffer(200, () => {
      void this.saveState();
    });
    let init = false;
    this.resizeObserver = new ResizeObserver(() => {
      if (init) {
        saveState();
      } else {
        init = true;
      }
    });
    this.resizeObserver.observe(this.el);
  }
  buildState() {
    if (this.isMaximized()) {
      const s = this.getState();
      s.maximized = true;
      return s;
    } else {
      return {
        width: this.width,
        height: this.height,
        left: this.el.offsetLeft,
        top: this.el.offsetTop
      };
    }
  }
  restoreState(s) {
    if (s.width)
      this.width = s.width;
    if (s.height)
      this.height = s.height;
    if (s.top != void 0) {
      this.el.style.top = s.top + "px";
      if (s.left != void 0) {
        this.el.style.left = s.left + "px";
      }
    } else {
      this.center();
    }
    if (s.maximized) {
      this.maximized = true;
    }
  }
  /**
   * Open the window by rendering it into the DOM body element
   * Use show()
   * @deprecated
   */
  open() {
    return this.show();
  }
  internalSetHidden(hidden) {
    if (!hidden) {
      if (Menu.openedMenu) {
        Menu.openedMenu.close();
      }
      this.focussedBeforeOpen = document.activeElement || void 0;
      if (!this.rendered) {
        root.items.add(this);
        if (this.modal) {
          this.modalOverlay = comp({
            cls: "goui-window-modal-overlay goui-goui-fade-in goui-goui-fade-out",
            hidden: true
          });
          this.modalOverlay.el.style.zIndex = parseInt(getComputedStyle(this.el).zIndex).toString();
          root.items.insert(-1, this.modalOverlay);
          this.disableBodyScroll();
          this.modalOverlay.el.addEventListener("click", () => {
            this.focus();
          });
          this.modalOverlay.show();
        }
        super.internalSetHidden(hidden);
        if (!this.hasState()) {
          this.shrinkToFit();
          this.center();
        } else {
          this.constrainTo(window);
        }
      } else {
        super.internalSetHidden(hidden);
      }
      this.focus();
    } else {
      super.internalSetHidden(hidden);
    }
  }
  shrinkToFit() {
    if (this.el.offsetHeight > window.innerHeight) {
      this.el.style.height = window.innerHeight * 0.9 + "px";
    }
    if (this.el.offsetWidth > window.innerWidth) {
      this.el.style.width = window.innerWidth * 0.9 + "px";
    }
  }
  disableBodyScroll() {
    if (window.getComputedStyle(document.body).overflow != "hidden") {
      document.body.style.top = `-${window.scrollY}px`;
      document.body.style.position = "fixed";
    }
  }
  enableBodyScroll() {
    const scrollY = document.body.style.top, scrollTo = parseInt(scrollY || "0") * -1;
    document.body.style.position = "";
    document.body.style.top = "";
    document.documentElement.style.scrollBehavior = "auto";
    window.scrollTo({
      top: scrollTo,
      behavior: "auto"
    });
    document.documentElement.style.scrollBehavior = "";
  }
  /**
   * @inheritDoc
   */
  remove() {
    if (this.focussedBeforeOpen instanceof HTMLElement) {
      this.focussedBeforeOpen.focus();
    }
    if (this.resizeObserver) {
      this.resizeObserver.disconnect();
    }
    if (this.modalOverlay) {
      this.enableBodyScroll();
      this.modalOverlay.remove();
    }
    this.header.remove();
    return super.remove();
  }
  /**
   * Close the window by removing it
   */
  close() {
    if (this.fire("beforeclose", this) === false) {
      return;
    }
    this.remove();
    this.fire("close", this);
  }
  /**
   * Center the window in the screen
   */
  center() {
    this.el.style.top = (window.innerHeight - this.height) / 2 + "px";
    this.el.style.left = (window.innerWidth - this.width) / 2 + "px";
    return this;
  }
  /**
   * Returns true if the window is maximized
   */
  isMaximized() {
    return this.el.classList.contains("maximized");
  }
  /**
   * Grow window to the maximum of the viewport
   */
  maximize() {
    this.collapsed = false;
    this.el.classList.add("maximized");
    this.fire("maximize", this);
    return this;
  }
  /**
   * Make the window smaller than the viewport
   */
  unmaximize() {
    this.el.classList.remove("maximized");
    this.fire("unmaximize", this);
    return this;
  }
  /**
   * Display an error message
   *
   * @param msg - The error message to be displayed.
   * @return Promise<void> - A promise that resolves when the alert window is closed
   */
  static error(msg) {
    console.error(msg);
    if (msg.message) {
      msg = msg.message;
    }
    return _Window.alert(msg, t("Error") + " - " + new DateTime().format("Y-m-d H:i:s"));
  }
  /**
   * Show modal alert window
   *
   * @param text - The alert message or an object with a 'message' property
   * @param [title="Alert"] - The title of the alert window
   * @return Promise<void> - A promise that resolves when the alert window is closed
   */
  static alert(text, title = t("Alert")) {
    if (text && text.message) {
      console.error(text);
      text = text.message;
    }
    return new Promise((resolve) => {
      win(
        {
          width: 600,
          modal: true,
          title,
          listeners: {
            close: () => {
              resolve();
            }
          }
        },
        comp({
          flex: 1,
          cls: "scroll pad",
          html: text
        })
      ).show();
    });
  }
  /**
   * Prompt the user for a text input value.
   *
   * @param text - The message to display to the user.
   * @param inputLabel - The label for the input field.
   * @param [defaultValue=""] - The default value for the input field.
   * @param [title="Please enter"] - The title for the prompt window.
   * @returns {Promise<string | undefined>} - A promise that resolves with the input value or undefined if the user cancelled.
   */
  static prompt(text, inputLabel, defaultValue = "", title = t("Please enter")) {
    return new Promise((resolve) => {
      let cancelled = true;
      const w = win(
        {
          modal: true,
          title,
          width: 600,
          listeners: {
            focus: () => {
              w.items.get(0).focus();
            },
            close: () => {
              if (cancelled) {
                resolve(void 0);
              }
            }
          }
        },
        form(
          {
            handler: (form2) => {
              resolve(form2.value.input);
              cancelled = false;
              w.close();
            }
          },
          fieldset(
            {},
            comp({
              tagName: "p",
              html: text
            }),
            textfield({
              label: inputLabel,
              name: "input",
              required: true,
              value: defaultValue
            })
          ),
          tbar(
            {},
            comp({
              flex: 1
            }),
            btn({
              type: "submit",
              text: "Ok"
            })
          )
        )
      );
      w.show();
    });
  }
  /**
   * Asks the user for confirmation.
   * @param {string} text - The text to display in the confirmation dialog.
   * @param {string} [title=t("Please confirm")] - The title of the confirmation dialog.
   * @return {Promise<boolean>} - A promise that resolves to `true` if the user confirms, or `false` if the user cancels.
   */
  static confirm(text, title = t("Please confirm")) {
    return new Promise((resolve) => {
      const w = win(
        {
          modal: true,
          title,
          closable: false,
          width: 600,
          listeners: {
            focus: (w2) => {
              w2.findChild("no").focus();
            }
          }
        },
        comp({
          cls: "pad",
          html: text
        }),
        tbar(
          {},
          "->",
          btn({
            itemId: "no",
            text: t("No"),
            handler: () => {
              resolve(false);
              w.close();
            }
          }),
          btn({
            itemId: "yes",
            text: t("Yes"),
            cls: "filled primary",
            handler: () => {
              resolve(true);
              w.close();
            }
          })
        )
      );
      w.show();
    });
  }
};
var win = (config, ...items) => createComponent(new Window2(), config, items);

// goui/script/component/form/DataSourceForm.ts
var DataSourceForm = class extends Form {
  constructor(dataSource) {
    super();
    this.dataSource = dataSource;
    this.handler = (form1) => __async(this, null, function* () {
      var _a;
      try {
        let data, v = this.currentId ? this.modified : this.value;
        this.fire("beforesave", this, v);
        if (!this.isNew) {
          data = yield this.dataSource.update(this.currentId, v);
        } else {
          data = yield this.dataSource.create(v);
        }
        if (data) {
          this.fire("save", this, data, this.isNew);
        }
      } catch (e) {
        if (e.type == "invalidProperties") {
          this.handleServerValidation(e);
          return;
        }
        console.error(t("Error"), e);
        if (this.fire("saveerror", this, e) !== false) {
          void Window2.error((_a = e.message) != null ? _a : t("Unknown error"));
        }
      } finally {
        this.unmask();
      }
    });
  }
  handleServerValidation(error) {
    for (const propertyName in error.validationErrors) {
      const field = this.findField(propertyName);
      if (!field) {
        continue;
      }
      field.setInvalid(error.validationErrors[propertyName].description);
    }
    const invalid = this.findFirstInvalid();
    if (invalid) {
      invalid.focus();
    }
    this.setInvalid(t("You have errors in your form. The invalid fields are marked."));
  }
  create(data) {
    this.reset();
    this.fire("load", this, data);
    if (data) {
      this.value = data;
    }
  }
  reset() {
    super.reset();
    delete this.currentId;
  }
  get isNew() {
    return !this.currentId;
  }
  /**
   * Load an entity into the form
   *
   * @param id
   */
  load(id) {
    return __async(this, null, function* () {
      this.mask();
      try {
        this.currentId = id;
        let entity = yield this.dataSource.single(id);
        if (!entity) {
          throw "Failed to load entity with id " + id;
        }
        this.fire("load", this, entity);
        this.value = entity;
      } catch (e) {
        console.error(t("Error"), e);
        if (this.fire("loaderror", this, e) !== false) {
          void Window2.error(e.message);
        }
      } finally {
        this.unmask();
      }
    });
  }
};
var datasourceform = (config, ...items) => createComponent(new DataSourceForm(config.dataSource), config, items);

// goui/script/component/form/AutocompleteChips.ts
var AutocompleteChips = class extends ChipsField {
  /**
   * Constructor
   *
   * @param list The drop down list or {@see Table}.
   * @param buffer
   */
  constructor(list2, buffer = 300) {
    super();
    this.list = list2;
    this.buffer = buffer;
    //disable create
    this.textInputToValue = (text) => __async(this, null, function* () {
      return false;
    });
    this.menu = menu(
      {
        cls: "goui-dropdown scroll",
        removeOnClose: false,
        height: 300,
        listeners: {
          hide: this.onMenuHide.bind(this)
        }
      },
      list2
    );
    this.menuButton = btn({
      icon: "expand_more",
      type: "button",
      handler: (button, ev) => {
        this.fire("autocomplete", this, "");
      },
      menu: this.menu
    });
    this.initList();
  }
  initList() {
    this.list.el.addEventListener("keydown", (ev) => {
      switch (ev.key) {
        case "Enter":
          ev.preventDefault();
          this.menu.hide();
          break;
      }
    });
    if (!this.list.rowSelection) {
      this.list.rowSelectionConfig = { multiSelect: false };
    }
    if (!this.list.rowSelection.multiSelect) {
      this.list.on("rowclick", () => {
        this.menu.hide();
      });
      this.list.store.on("datachanged", () => {
        this.list.rowSelection.selected = [0];
      }, { buffer: 0 });
    } else {
      this.list.store.on("datachanged", () => {
        this.list.rowSelection.clear();
        this.list.store.data.forEach((record, index) => {
          if (this.isPickerRecordInValue(record)) {
            this.list.rowSelection.add(index);
          }
        });
      }, { buffer: 0 });
    }
  }
  isPickerRecordInValue(record) {
    if (!this.value || !this.value.length) {
      return false;
    }
    if (!this.valuesToCompare) {
      this.valuesToCompare = this.value.map((i) => JSON.stringify(i));
      setTimeout(() => {
        this.valuesToCompare = void 0;
      });
    }
    const v = JSON.stringify(this.pickerRecordToValue(this, record));
    return this.valuesToCompare.indexOf(v) > -1;
  }
  /**
   * Method that transforms a record from the TablePicker store to a value for this field.
   * This is not necessarily a text value. In conjunction with {@see valueToTextField()} this
   * could also be an ID of an object for example.
   *
   * @param field
   * @param record
   */
  pickerRecordToValue(field, record) {
    return record.id;
  }
  internalRender() {
    this.buttons = this.buttons || [];
    this.buttons.push(this.menuButton);
    const el = super.internalRender();
    this.menu.alignTo = this.wrap;
    this.menu.alignToInheritWidth = true;
    this.editor.el.addEventListener("input", FunctionUtil.buffer(this.buffer, this.onInput.bind(this)));
    this.editor.el.addEventListener("keydown", (ev) => {
      switch (ev.key) {
        case "Enter":
          if (!this.menu.hidden) {
            ev.preventDefault();
            this.menu.hide();
          }
          break;
        case "ArrowDown":
          ev.preventDefault();
          this.menuButton.showMenu();
          this.list.focus();
          break;
        case "Escape":
          if (!this.menu.hidden) {
            this.menu.hide();
            ev.preventDefault();
            ev.stopPropagation();
            this.focus();
          }
          break;
      }
    });
    return el;
  }
  onInput(ev) {
    this.menuButton.showMenu();
    this.fire("autocomplete", this, this.editor.el.innerText);
  }
  onMenuHide() {
    if (!this.menu.rendered) {
      return;
    }
    const newValues = this.list.rowSelection.selected.map((index) => {
      const record = this.list.store.get(index);
      return this.pickerRecordToValue(this, record);
    });
    this.list.rowSelection.clear();
    this.editor.el.innerText = "";
    this.focus();
    if (this.list.rowSelection.multiSelect) {
      this.value = newValues;
    } else {
      if (!this.value) {
        this.value = [];
      }
      this.value = this.value.concat(newValues);
    }
  }
};
var autocompletechips = (config) => createComponent(new AutocompleteChips(config.list), config);

// goui/script/data/Store.ts
var Store = class extends Collection {
  constructor() {
    super(...arguments);
    // private static stores: Record<string, Store> = {};
    //
    // /**
    //  * If the ID is given the store can be fetched with
    //  */
    // public id?: string;
    this._loading = false;
    this._loaded = false;
    /**
     * Sort the data on field and direction
     */
    this.sort = [];
  }
  /**
   * True when the store is loading
   */
  get loading() {
    return this._loading;
  }
  /**
   * True when data has been loaded at least once
   */
  get loaded() {
    return this._loaded;
  }
  /**
   * Load a set of records
   *
   * @param records
   * @param append
   */
  loadData(records, append = true) {
    append ? this.add(...records) : this.replace(...records);
    this._loaded = true;
    this.fire("load", this, records, append);
  }
  set data(records) {
    this.loadData(records);
  }
  get data() {
    return this.getArray();
  }
  /**
   * Reload the data from the source
   */
  reload() {
    return __async(this, null, function* () {
      this._loaded = false;
      return this.load();
    });
  }
  clear() {
    this._loaded = false;
    return super.clear();
  }
  /**
   * Returns the loaded records. If append is true it only returns the new records.
   * Override this function for new store types.
   *
   * @param append
   * @protected
   */
  internalLoad(append) {
    this.loadData(ArrayUtil.multiSort(this.items, this.sort), append);
    return Promise.resolve(this.items);
  }
  /**
   * Loads the data from source into the store
   *
   * @param append
   */
  load(append = false) {
    this._loading = true;
    this.fire("beforeload", this, append);
    return this.internalLoad(append).catch((reason) => {
      console.error(reason);
      this.fire("loadexception", this, reason);
      throw reason;
    }).finally(() => {
      this._loading = false;
    });
  }
  /**
   * Load the next set of records when paging.
   * Doesn't do anything in the array store but can be implemented in async stores.
   */
  loadNext(append = false) {
    return Promise.resolve([]);
  }
  /**
   * Load the next set of records when paging.
   * Doesn't do anything in the array store but can be implemented in async stores.
   */
  loadPrevious() {
    return Promise.resolve([]);
  }
  hasNext() {
    return false;
  }
  hasPrevious() {
    return false;
  }
};
var store = (config) => createComponent(new Store(), config);

// goui/script/data/DataSourceStore.ts
var DataSourceStore = class extends Store {
  // private bufferedLoad?: (...args:any[]) => Promise<StoreRecord[]>;
  constructor(dataSource) {
    super();
    this.dataSource = dataSource;
    this.queryParams = {};
    this.hasMore = false;
    // public properties?: string[] = [];
    /**
     * Reload when the datasource changes
     */
    this.monitorChanges = true;
    /**
     * Builds record from entity
     * @param entity
     */
    this.buildRecord = (entity) => __async(this, null, function* () {
      return entity;
    });
    this.filters = {};
    this.dataSource.on("change", (dataSource2, changes) => {
      void this.onChange(dataSource2, changes);
    });
  }
  /**
   * Reloads the store when the datasource changes
   *
   * @protected
   */
  onChange(DataSource, changes) {
    return __async(this, null, function* () {
      if (this.loaded && this.monitorChanges && !this.loading) {
        void this.reload();
      }
    });
  }
  internalLoad(append = false) {
    return __async(this, null, function* () {
      const queryParams = structuredClone(this.queryParams);
      queryParams.sort = this.sort;
      if (queryParams.limit) {
        queryParams.limit++;
      }
      const queryResponse = yield this.dataSource.query(queryParams);
      if (this.queryParams.limit) {
        this.hasMore = queryResponse.ids.length > this.queryParams.limit;
        if (this.hasMore) {
          queryResponse.ids.pop();
        }
      }
      const getResponse = yield this.dataSource.get(queryResponse.ids);
      const entities = yield this.fetchRelations(getResponse.list), records = yield Promise.all(entities.map(this.buildRecord));
      this.loadData(records, append);
      return records;
    });
  }
  fetchRelations(records) {
    return __async(this, null, function* () {
      if (!this.relations) {
        return records;
      }
      const promises = [];
      for (const relationName in this.relations) {
        const rel = this.relations[relationName];
        let id;
        for (let i = 0, l = records.length; i < l; i++) {
          id = ObjectUtil.path(records[i], rel.path);
          if (id) {
            promises.push(rel.dataSource.single(id).then((e) => {
              if (e) {
                records[i][relationName] = e;
              }
            }));
          }
        }
      }
      return Promise.all(promises).then(() => records);
    });
  }
  reload() {
    this.queryParams.position = 0;
    return super.reload();
  }
  loadNext(append = false) {
    if (!this.queryParams.limit) {
      throw new Error("Limit must be set for pagination");
    }
    this.queryParams.position = this.queryParams.position || 0;
    this.queryParams.position += this.queryParams.limit;
    return this.load(append);
  }
  loadPrevious() {
    if (!this.queryParams.limit) {
      throw new Error("Limit and position must be set!");
    }
    this.queryParams.position = this.queryParams.position || 0;
    this.queryParams.position -= this.queryParams.limit;
    return this.load(false);
  }
  hasNext() {
    return this.hasMore;
  }
  hasPrevious() {
    return this.queryParams.position > 0;
  }
  /**
   * Load more data when this element is scrolled down
   *
   * @param el
   */
  addScrollLoader(el) {
    const onScroll = () => {
      const pixelsLeft = el.scrollHeight - el.scrollTop - el.offsetHeight;
      if (pixelsLeft < 100) {
        if (!this.loading && this.hasNext()) {
          void this.loadNext(true);
        }
      }
    };
    el.addEventListener("scroll", onScroll, { passive: true });
    this.on("load", (store3, records, append) => {
      setTimeout(() => {
        onScroll();
      });
    });
    return onScroll;
  }
  patchFilter(ref, filter) {
    var _a;
    const f = (_a = this.getFilter(ref)) != null ? _a : {};
    return this.setFilter(ref, Object.assign(f, filter));
  }
  setFilter(ref, filter) {
    if (filter === void 0) {
      delete this.filters[ref];
    } else {
      this.filters[ref] = filter;
    }
    const conditions = [];
    for (const k in this.filters) {
      conditions.push(this.filters[k]);
    }
    switch (conditions.length) {
      case 0:
        delete this.queryParams.filter;
        break;
      case 1:
        this.queryParams.filter = conditions[0];
        break;
      default:
        this.queryParams.filter = {
          operator: "AND",
          conditions
        };
        break;
    }
    return this;
  }
  clearFilter(...names) {
    names.forEach((n) => this.setFilter(n, void 0));
  }
  getFilter(name) {
    return this.filters[name];
  }
};
var datasourcestore = (config) => {
  let f;
  if (config.filters) {
    f = config.filters;
    delete config.filters;
  }
  const store3 = createComponent(new DataSourceStore(config.dataSource), config);
  if (f) {
    for (const filterName in f) {
      store3.setFilter(filterName, f[filterName]);
    }
  }
  return store3;
};

// goui/script/data/AbstractDataSource.ts
var CommitErrorType = /* @__PURE__ */ ((CommitErrorType2) => {
  CommitErrorType2[CommitErrorType2["forbidden"] = 0] = "forbidden";
  CommitErrorType2[CommitErrorType2["overQuota"] = 1] = "overQuota";
  CommitErrorType2[CommitErrorType2["tooLarge"] = 2] = "tooLarge";
  CommitErrorType2[CommitErrorType2["rateLimit"] = 3] = "rateLimit";
  CommitErrorType2[CommitErrorType2["notFound"] = 4] = "notFound";
  CommitErrorType2[CommitErrorType2["invalidPatch"] = 5] = "invalidPatch";
  CommitErrorType2[CommitErrorType2["willDestroy"] = 6] = "willDestroy";
  CommitErrorType2[CommitErrorType2["invalidProperties"] = 7] = "invalidProperties";
  CommitErrorType2[CommitErrorType2["singleton"] = 8] = "singleton";
  CommitErrorType2[CommitErrorType2["requestTooLarge"] = 9] = "requestTooLarge";
  CommitErrorType2[CommitErrorType2["stateMismatch"] = 10] = "stateMismatch";
  return CommitErrorType2;
})(CommitErrorType || {});
var AbstractDataSource = class extends Observable {
  constructor(id) {
    super();
    this.id = id;
    /**
     * Store data in the browser storage so it will persist across sessions
     */
    this.persist = true;
    /**
     * Extra parameters to send to the Foo/set
     */
    this.commitBaseParams = {};
    /**
     * Extra /set parameters that will reset after commit
     */
    this.setParams = {};
    this.data = {};
    this.creates = {};
    this.updates = {};
    this.destroys = {};
    this.getIds = {};
    this._createId = 0;
    this.delayedCommit = FunctionUtil.buffer(0, () => {
      void this.commit();
    });
    this.delayedGet = FunctionUtil.buffer(0, () => {
      void this.doGet();
    });
  }
  /**
   * Get the local server state ID of the store
   * @protected
   */
  getState() {
    return __async(this, null, function* () {
      if (!this._state && this.persist) {
        this._state = yield this.browserStore.getItem("__state__");
      }
      return this._state;
    });
  }
  /**
   * Set's the local server state ID
   *
   * Setting it to undefined will reset the store.
   *
   * @param state
   * @protected
   */
  setState(state) {
    return __async(this, null, function* () {
      this._state = state;
      if (!this.persist) {
        return;
      }
      return this.browserStore.setItem("__state__", state);
    });
  }
  clearCache() {
    this.data = {};
    return this.browserStore.clear();
  }
  /**
   * Get the browser storage object to save state to the browser
   * @private
   */
  get browserStore() {
    if (!this._browserStore) {
      this._browserStore = new BrowserStore("ds-" + this.id);
    }
    return this._browserStore;
  }
  /**
   * Get entities from the store
   *
   * It will return a list of entities ordered by the requested ID's
   *
   * @param ids
   */
  get(ids) {
    return __async(this, null, function* () {
      const promises = [], order = {};
      if (ids == void 0) {
        ids = (yield this.query()).ids;
      }
      ids.forEach((id, index) => {
        order[id] = index++;
        promises.push(this.single(id));
      });
      let entities = yield Promise.all(promises);
      const response = {
        list: [],
        notFound: [],
        state: yield this.getState()
      };
      entities.forEach((e, index) => {
        if (e === void 0) {
          response.notFound.push(ids[index]);
        } else {
          response.list.push(e);
        }
      });
      response.list = response.list.sort(function(a2, b) {
        return order[a2.id] - order[b.id];
      });
      return response;
    });
  }
  add(data) {
    return __async(this, null, function* () {
      this.data[data.id] = data;
      if (!this.persist) {
        return Promise.resolve(data);
      }
      yield this.browserStore.setItem(data.id, data);
      return data;
    });
  }
  remove(id) {
    return __async(this, null, function* () {
      console.debug("Removing " + this.id + ": " + id);
      delete this.data[id];
      if (!this.persist) {
        return Promise.resolve(id);
      }
      yield this.browserStore.removeItem(id);
      return id;
    });
  }
  /**
   * Get a single entity.
   *
   * Multiple calls will be buffered and returned together on the next event loop. This way multiple calls can
   * be joined together in a single HTTP request to the server.
   *
   * @param id
   */
  single(id) {
    return __async(this, null, function* () {
      const p2 = new Promise((resolve, reject) => {
        if (!this.getIds[id]) {
          this.getIds[id] = {
            resolves: [resolve],
            rejects: [reject]
          };
        } else {
          this.getIds[id].resolves.push(resolve);
          this.getIds[id].rejects.push(reject);
        }
      });
      this.delayedGet();
      return p2;
    });
  }
  returnGet(id) {
    let r;
    if (!this.getIds[id]) {
      return;
    }
    while (r = this.getIds[id].resolves.shift()) {
      r.call(this, structuredClone(this.data[id]));
    }
    delete this.getIds[id];
  }
  /**
   * Does the actual getting of entities. First checks if present in this onbject, otherwise it will be requested
   * from the remote source.
   *
   * @protected
   */
  doGet() {
    return __async(this, null, function* () {
      const unknownIds = [];
      for (let id in this.getIds) {
        if (this.data[id]) {
          this.returnGet(id);
        } else if (this.persist) {
          const data = yield this.browserStore.getItem(id);
          if (data) {
            this.data[id] = data;
            this.returnGet(id);
          } else {
            unknownIds.push(id);
          }
        } else {
          unknownIds.push(id);
        }
      }
      if (!unknownIds.length) {
        return;
      }
      this.internalGet(unknownIds).then((response) => this.checkState(response.state, response)).then((response) => {
        var _a;
        response.list.forEach((e) => {
          this.add(e);
          this.returnGet(e.id);
        });
        (_a = response.notFound) == null ? void 0 : _a.forEach((id) => {
          let r;
          while (r = this.getIds[id].resolves.shift()) {
            r.call(this, void 0);
          }
          delete this.getIds[id];
        });
      }).catch((e) => {
        unknownIds.forEach((id) => {
          if (this.getIds[id]) {
            let r;
            while (r = this.getIds[id].rejects.shift()) {
              r.call(this, e);
            }
            delete this.getIds[id];
          }
        });
      });
    });
  }
  /**
   * Create entity
   *
   * Multiple calls will be joined together in a single call on the next event loop
   *
   * @param data
   * @param createId The create ID to use when committing this entity to the server
   */
  create(data, createId) {
    if (createId === void 0) {
      createId = this.createID();
    }
    const p2 = new Promise((resolve, reject) => {
      this.creates[createId] = {
        data,
        resolve,
        reject
      };
    }).finally(() => {
      delete this.creates[createId];
    });
    this.delayedCommit();
    return p2;
  }
  /**
   * Reset the data source.
   *
   * Clears all data and will resync
   */
  reset() {
    return __async(this, null, function* () {
      return this.setState(void 0);
    });
  }
  /**
   * Update an entity
   *
   * Multiple calls will be joined together in a single call on the next event loop
   *
   * @param id
   * @param data
   */
  update(id, data) {
    const p2 = new Promise((resolve, reject) => {
      this.updates[id] = {
        data,
        resolve,
        reject
      };
    }).finally(() => {
      delete this.updates[id];
    });
    this.delayedCommit();
    return p2;
  }
  createID() {
    return "_new_" + ++this._createId;
  }
  /**
   * Destroy an entity
   *
   * Multiple calls will be joined together in a single call on the next event loop
   *
   * @param id
   */
  destroy(id) {
    const p2 = new Promise((resolve, reject) => {
      this.destroys[id] = {
        resolve,
        reject
      };
    }).finally(() => {
      delete this.destroys[id];
    });
    this.delayedCommit();
    return p2;
  }
  /**
   * Ask for confirmation and delete entities by ID
   *
   * @example
   * ```
   * const tbl = this.projectTable!,
   * 	ids = tbl.rowSelection!.selected.map(index => tbl.store.get(index)!.id);
   *
   * const result = await jmapds("Project3")
   * 	.confirmDestroy(ids);
   *
   * if(result != false) {
   * 	btn.parent!.hide();
   * }
   * ```
   * @param ids The ID's to delete
   */
  confirmDestroy(ids) {
    return __async(this, null, function* () {
      const count = ids.length;
      if (!count) {
        return false;
      }
      let msg;
      if (count == 1) {
        msg = t("Are you sure you want to delete the selected item?");
      } else {
        msg = t("Are you sure you want to delete {count} items?").replace("{count}", count);
      }
      const confirmed = yield Window2.confirm(msg);
      if (!confirmed) {
        return false;
      }
      root.mask(300);
      return Promise.all(ids.map((id) => {
        return this.destroy(id);
      })).finally(() => {
        root.unmask();
      }).catch((e) => {
        console.error(e);
        Window2.error(e.message);
      });
    });
  }
  /**
   * Fetch updates from remote
   */
  updateFromServer() {
    return __async(this, null, function* () {
      let hasMoreChanges = true, hasAChange = false;
      const allChanges = {
        created: [],
        updated: [],
        destroyed: [],
        oldState: "",
        newState: ""
      }, promises = [];
      try {
        while (hasMoreChanges) {
          const state = yield this.getState();
          if (state === void 0) {
            return;
          }
          if (!allChanges.oldState) {
            allChanges.oldState = state;
          }
          const changes = yield this.internalRemoteChanges(state);
          if (changes.created) {
            for (let id of changes.created) {
              promises.push(this.remove(id));
              allChanges.created.push(id + "");
              hasAChange = true;
            }
          }
          if (changes.updated) {
            for (let id of changes.updated) {
              promises.push(this.remove(id));
              allChanges.updated.push(id + "");
              hasAChange = true;
            }
          }
          if (changes.destroyed) {
            for (let id of changes.destroyed) {
              promises.push(this.remove(id));
              allChanges.destroyed.push(id + "");
              hasAChange = true;
            }
          }
          yield Promise.all(promises);
          yield this.setState(changes.newState);
          allChanges.newState = changes.newState;
          hasMoreChanges = !!changes.hasMoreChanges;
        }
      } catch (e) {
        console.error(this.id + " Error while updating from server. Resetting data source.");
        console.error(e);
        yield this.reset();
      }
      if (hasAChange) {
        this.fire("change", this, allChanges);
      }
    });
  }
  /**
   * Commit pending changes to remote
   */
  commit() {
    return __async(this, null, function* () {
      const params = Object.assign({
        create: {},
        update: {},
        destroy: [],
        ifInState: yield this.getState()
      }, this.commitBaseParams, this.setParams);
      this.setParams = {};
      for (let id in this.creates) {
        params.create[id] = this.creates[id].data;
      }
      for (let id in this.updates) {
        params.update[id] = this.updates[id].data;
      }
      for (let id in this.destroys) {
        params.destroy.push(id);
      }
      this.internalCommit(params).then((response) => __async(this, null, function* () {
        if (response.created) {
          for (let clientId in response.created) {
            let data = Object.assign(params.create ? params.create[clientId] || {} : {}, response.created[clientId] || {});
            this.add(data).then(() => this.creates[clientId].resolve(data));
          }
        }
        if (response.notCreated) {
          for (let clientId in response.notCreated) {
            this.creates[clientId].reject(response.notCreated[clientId]);
          }
        }
        if (response.updated) {
          for (let serverId in response.updated) {
            if (!this.data[serverId]) {
              continue;
            }
            let data = params.update && params.update[serverId] ? Object.assign(this.data[serverId], params.update[serverId]) : this.data[serverId];
            data = Object.assign(data, response.updated[serverId] || {});
            this.add(data).then((data2) => this.updates[serverId].resolve(data2));
          }
        }
        if (response.notUpdated) {
          for (let serverId in response.notUpdated) {
            this.updates[serverId].reject(response.notUpdated[serverId]);
          }
        }
        if (response.destroyed) {
          for (let i = 0, l = response.destroyed.length; i < l; i++) {
            this.remove(response.destroyed[i]).then((id) => this.destroys[id].resolve(id));
          }
        }
        if (response.notDestroyed) {
          for (let serverId in response.notDestroyed) {
            this.destroys[serverId].reject(response.notDestroyed[serverId]);
          }
        }
        yield this.setState(response.newState);
        this.fire("change", this, {
          created: response.created ? Object.keys(response.created) : [],
          updated: response.updated ? Object.keys(response.updated) : [],
          destroyed: response.destroyed || [],
          oldState: response.oldState,
          newState: response.newState
        });
      })).catch((e) => {
        for (let clientId in this.creates) {
          this.creates[clientId].reject(e);
        }
        for (let clientId in this.updates) {
          this.updates[clientId].reject(e);
        }
        for (let clientId in this.destroys) {
          this.destroys[clientId].reject(e);
        }
        throw e;
      }).finally(() => {
        this.creates = {};
        this.updates = {};
        this.destroys = {};
      });
    });
  }
  /**
   * Query the server for a list of entity ID's
   *
   * It takes filters and sort parameters.
   *
   * @link https://jmap.io/spec-core.html#query
   */
  query() {
    return __async(this, arguments, function* (params = {}) {
      let r = yield this.internalQuery(params);
      return this.checkState(r.queryState, r);
    });
  }
  /**
   * Check's if we are up-to-date with the server and fetches updates if needed.
   *
   * If no state is returned by the data source this function will ignore states and the data source should then
   * always refresh data.
   *
   * @param serverState
   * @param retVal
   * @private
   */
  checkState(serverState, retVal) {
    return __async(this, null, function* () {
      let state = yield this.getState();
      if (!state && serverState) {
        yield this.setState(serverState);
        state = serverState;
      }
      if (serverState != state) {
        return this.updateFromServer().then(() => retVal);
      } else {
        return Promise.resolve(retVal);
      }
    });
  }
};

// goui/script/data/RestDataSource.ts
var RestDataSource = class extends AbstractDataSource {
  /**
   * Constructor
   *
   * @param uri The base URI of the REST API. for example https://groupoffice.com/api
   * @param id The Data source ID. Will be appended to the base uri above. for Example
   *  if the ID is "users" the uri will be: "https://groupoffice.com/api/users"
   */
  constructor(uri, id) {
    id = id || "restDataSource";
    super(id);
    this.uri = uri;
    if (uri.substring(-1, 1) != "/") {
      this.uri = uri + "/";
    }
  }
  read(id, path = "", options = {}) {
    if (typeof id === "number") {
      path += "/" + id;
    } else {
      if (path != "" && path.substring(-1, 1) != "/") {
      }
    }
    return this.request(path, options);
  }
  doRequest(path = "", options = {}) {
    return this.request(path, options);
  }
  request(path = "", options = {}) {
    const baseOpts = {
      mod: "cors",
      method: "GET"
    };
    Object.assign(baseOpts, options);
    if (path != "" && path.substring(0, 1) != "") {
      path = "/" + path;
    }
    return fetch(this.uri + path, baseOpts).then((response) => {
      return response.json();
    });
  }
  internalCommit(params) {
    const response = {
      updated: {},
      created: {},
      destroyed: []
    }, promises = [];
    for (let id in params.update) {
      promises.push(
        this.request(id, {
          method: "PUT",
          body: JSON.stringify(params.update[id])
        }).then((data) => {
          if (!response.updated) {
            response.updated = {};
          }
          response.updated[id] = data;
        }).catch((reason) => {
          if (!response.notUpdated) {
            response.notUpdated = {};
          }
          response.notUpdated[id] = reason;
        })
      );
    }
    for (let id in params.create) {
      promises.push(
        this.request("", {
          method: "POST",
          body: JSON.stringify(params.create[id])
        }).then((data) => {
          if (!response.created) {
            response.created = {};
          }
          response.created[id] = data;
        }).catch((reason) => {
          if (!response.notCreated) {
            response.notCreated = {};
          }
          response.notCreated[id] = reason;
        })
      );
    }
    for (let id of params.destroy) {
      promises.push(
        this.request(id + "", {
          method: "DELETE"
        }).then(() => {
          if (!response.destroyed) {
            response.destroyed = [];
          }
          response.destroyed.push(id);
        }).catch((reason) => {
          if (!response.notDestroyed) {
            response.notDestroyed = {};
          }
          response.notDestroyed[id] = reason;
        })
      );
    }
    return Promise.all(promises).then(() => {
      return response;
    });
  }
  internalGet(ids) {
    return __async(this, null, function* () {
      const promises = [];
      ids.forEach((id) => {
        promises.push(
          this.request(id + "").then((data) => {
            return data.data;
          })
        );
      });
      const users = yield Promise.all(promises);
      const r = {
        list: []
      };
      users.forEach((u) => {
        r.list.push(u);
      });
      return r;
    });
  }
  internalQuery(params) {
    return __async(this, null, function* () {
      const response = yield this.request();
      response.data.forEach((r) => {
        if (!this.data[r.id]) {
          this.add(r);
        }
      });
      const ids = response.data.map((r) => r.id);
      return {
        ids,
        total: response.total
      };
    });
  }
  /**
   * No sync implemented for REST. States are ignored and data must always be refreshed.
   * @param state
   * @protected
   */
  internalRemoteChanges(state) {
    return __async(this, null, function* () {
      return {};
    });
  }
};

// goui/script/component/table/MultiSelectToolbar.ts
var MultiSelectToolbar = class extends Toolbar {
  constructor(table2) {
    super();
    this.table = table2;
    this.table.rowSelection.on("selectionchange", (tableRowSelect) => {
      const l = tableRowSelect.selected.length;
      this.hidden = l < 2;
      if (!this.hidden) {
        this.label.text = l + " " + t("selected");
      }
    });
    this.label = comp({ tagName: "h3" });
    this.backBtn = btn({
      title: t("Back"),
      icon: "chevron_left",
      handler: () => {
        this.table.rowSelection.clear();
      }
    });
    this.items.add(this.backBtn, this.label);
    this.hidden = true;
    this.cls = "multiselect";
  }
};
var mstbar = (config, ...items) => {
  const c = new MultiSelectToolbar(config.table);
  if (config) {
    Object.assign(c, config);
  }
  c.items.add(...tbarItems(items));
  return c;
};

// goui/script/component/table/RowSelect.ts
var RowSelect = class extends Observable {
  constructor(list2) {
    super();
    this.list = list2;
    this._selected = [];
    /**
     * Last selected index used for multi selection with shift
     * @private
     */
    this.lastIndex = -1;
    this.multiSelect = true;
    this.hasKeyUpListener = false;
    this.list.on("beforerender", (me) => {
      const tableEl = me.el;
      tableEl.cls("+rowselect");
      tableEl.addEventListener("keydown", (e) => {
        this.onKeyDown(e);
      });
      me.on("rowmousedown", (table2, index, row, e) => {
        this.onRowMouseDown(table2, index, e);
      });
      tableEl.addEventListener("focus", (e) => {
        if (!this.selected.length && this.list.store.get(0)) {
          this.selected = [0];
        }
      });
    });
    const fireSelectionChange = () => {
      this.fire("selectionchange", this);
    };
    this.fireSelectionChange = FunctionUtil.buffer(0, fireSelectionChange);
  }
  clear() {
    this.selected = [];
  }
  selectAll() {
    const selected = [];
    for (let i = 0, c = this.list.store.count(); i < c; i++) {
      selected.push(i);
    }
    this.selected = selected;
  }
  /**
   * Get selected indexes
   *
   * Note that this is a copy and can't be edited directly. You have to set the selected property with a changed array.
   */
  get selected() {
    return [...this._selected];
  }
  set selected(newSelection) {
    this.setSelected(newSelection);
  }
  onRowMouseDown(_list, index, e) {
    let selection = this.selected;
    if (e.shiftKey && this.multiSelect) {
      const start = Math.min(index, this.lastIndex);
      const end = Math.max(index, this.lastIndex);
      for (let i = start; i <= end; i++) {
        if (selection.indexOf(i) === -1)
          selection.push(i);
      }
    } else if ((e.ctrlKey || e.metaKey) && this.multiSelect) {
      const currentIndex = selection.indexOf(index);
      if (currentIndex > -1) {
        selection.splice(currentIndex, 1);
      } else {
        selection.push(index);
      }
    } else {
      selection = [index];
    }
    this.selected = selection;
  }
  add(index) {
    if (this._selected.indexOf(index) > -1) {
      return;
    }
    this._selected.push(index);
    this.fire("rowselect", this, index);
    this.fireSelectionChange();
  }
  /**
   * Remove an index from the selection
   *
   * @param index
   * @param silent Don't fire events
   */
  remove(index, silent = false) {
    const selectedIndex = this._selected.indexOf(index);
    if (selectedIndex == -1) {
      return;
    }
    this._selected.splice(selectedIndex, 1);
    if (!silent) {
      this.fire("rowdeselect", this, index);
      this.fireSelectionChange();
    }
  }
  /**
   * Set selected indexes
   *
   * @param newSelection
   * @param silent Suspends 'selectionchange' event
   */
  setSelected(newSelection, silent = false) {
    const old = this._selected;
    this._selected = newSelection;
    const deselect = ArrayUtil.diff(old, newSelection);
    const select2 = ArrayUtil.diff(newSelection, old);
    deselect.forEach((i) => {
      this.fire("rowdeselect", this, i);
    });
    select2.forEach((i, index) => {
      this.fire("rowselect", this, i);
    });
    if (newSelection.length) {
      this.lastIndex = newSelection[0];
    }
    const change = select2.length > 0 || deselect.length > 0;
    if (!silent && change) {
      this.fireSelectionChange();
    }
    return change;
  }
  onKeyDown(e) {
    if (e.key == "Escape") {
      this.clear();
      return;
    }
    if (e.key == "Shift") {
      this.shiftStartIndex = this.lastIndex;
    }
    if (e.key != "ArrowDown" && e.key != "ArrowUp") {
      return;
    }
    e.preventDefault();
    let index = 0, change = false;
    if (e.key == "ArrowDown") {
      if (this.lastIndex == this.list.store.count() - 1) {
        return;
      }
      index = this.lastIndex + 1;
    } else if (e.key == "ArrowUp") {
      if (this.lastIndex == 0) {
        return;
      }
      index = this.lastIndex - 1;
    }
    if (e.shiftKey && this.multiSelect) {
      const selected = this.selected;
      if (e.key == "ArrowDown" && index > this.shiftStartIndex || e.key == "ArrowUp" && index < this.shiftStartIndex) {
        if (selected.indexOf(index) == -1) {
          selected.push(index);
        }
      } else {
        const removeIndex = selected.indexOf(this.lastIndex);
        if (removeIndex > -1) {
          selected.splice(removeIndex, 1);
        }
      }
      change = this.setSelected(selected, true);
    } else {
      change = this.setSelected([index], true);
      this.list.focusRow(index);
    }
    if (change && !this.hasKeyUpListener) {
      this.hasKeyUpListener = true;
      this.list.el.addEventListener("keyup", () => {
        this.fire("selectionchange", this);
        this.hasKeyUpListener = false;
      }, { once: true });
    }
    this.lastIndex = index;
  }
};
var rowselect = (config) => createComponent(new RowSelect(config.list), config);

// goui/script/DragData.ts
var dragData = {};

// goui/script/component/List.ts
var dropPin = comp({
  cls: "drop-pin",
  hidden: true
});
root.items.add(dropPin);
var List = class extends Component {
  constructor(store3, renderer, tagName = "ul") {
    super(tagName);
    this.store = store3;
    this.renderer = renderer;
    /**
     * Shown when the list is empty.
     */
    this.emptyStateHtml = `<div class="goui-empty-state"><i class="icon">article</i><p>${t("Nothing to show")}</p></div>`;
    this.emptyStateTag = "li";
    /**
     * Allow items to be dragged
     */
    this.draggable = false;
    /**
     * Allow to drop between items
     */
    this.dropBetween = false;
    /**
     * Allow to drop on items
     */
    this.dropOn = false;
    this.itemTag = "li";
    this.tabIndex = 0;
    store3.on("beforeload", () => {
      this.mask();
    });
    store3.on("load", () => {
      this.unmask();
      if (this.emptyEl) {
        this.emptyEl.hidden = this.store.count() > 0;
      }
    });
    store3.on("loadexception", (store4, reason) => {
      Window2.error(reason);
      this.unmask();
    });
  }
  // protected fragment?: DocumentFragment;
  /**
   * Row selection object
   * @param rowSelectionConfig
   */
  set rowSelectionConfig(rowSelectionConfig) {
    if (typeof rowSelectionConfig != "boolean") {
      rowSelectionConfig.list = this;
      this.rowSelect = rowselect(rowSelectionConfig);
    } else {
      this.rowSelect = rowselect({ list: this });
    }
  }
  set sortable(sortable) {
    this.draggable = true;
    this.dropBetween = true;
    this.dropOn = false;
    const ref = this.on("drop", (list2, e, dropRow, dropIndex, position, dragData2) => {
      const store3 = dragData2.cmp.store;
      store3.removeAt(dragData2.storeIndex);
      if (dragData2.storeIndex < dropIndex) {
        dropIndex--;
      }
      switch (position) {
        case "before":
          store3.insert(dropIndex, dragData2.record);
          break;
        case "after":
          store3.insert(dropIndex + 1, dragData2.record);
          break;
      }
    });
  }
  get rowSelection() {
    return this.rowSelect;
  }
  get el() {
    return super.el;
  }
  internalRender() {
    const el = super.internalRender();
    this.initNavigateEvent();
    el.on("mousedown", (e) => {
      this.onMouseEvent(e, "rowmousedown");
    }).on("dblclick", (e) => {
      this.onMouseEvent(e, "rowdblclick");
    }).on("click", (e) => {
      this.onMouseEvent(e, "rowclick");
    }).on("contextmenu", (e) => {
      this.onMouseEvent(e, "rowcontextmenu");
    }).on("keydown", (e) => {
      this.onKeyDown(e);
    });
    this.renderEmptyState();
    this.renderBody();
    this.initStore();
    return el;
  }
  onKeyDown(e) {
    if (e.key == "Delete" || e.metaKey && e.key == "Backspace") {
      this.fire("delete", this);
    }
  }
  initStore() {
    this.store.on("remove", this.onRecordRemove.bind(this));
    this.store.on("add", this.onRecordAdd.bind(this));
  }
  onRecordRemove(collection, item, index) {
    var _a;
    const rows = this.getRowElements();
    (_a = rows[index]) == null ? void 0 : _a.remove();
    if (this.rowSelection) {
      this.rowSelection.remove(index, true);
    }
  }
  //todo inserting doesn't work with groups yet. It can only append to the last
  onRecordAdd(collection, item, index) {
    const container = this.renderGroup(item);
    const row = this.renderRow(item, index);
    if (index == collection.count() - 1) {
      container.append(row);
    } else {
      const before = container.children[index];
      container.insertBefore(row, before);
    }
    this.onRowAppend(row, item, index);
  }
  getRowElements() {
    return Array.from(this.el.querySelectorAll(":scope > .data"));
  }
  initNavigateEvent() {
    this.on("rowmousedown", (list2, storeIndex, row, ev) => {
      if (!ev.shiftKey && !ev.ctrlKey) {
        this.fire("navigate", this, storeIndex);
      }
    });
    if (this.rowSelection) {
      this.el.addEventListener("keydown", (ev) => {
        if (!ev.shiftKey && !ev.ctrlKey && (ev.key == "ArrowDown" || ev.key == "ArrowUp")) {
          const selected = this.rowSelect.selected;
          if (selected.length) {
            const storeIndex = selected[0];
            this.fire("navigate", this, storeIndex);
          }
        }
      });
    }
  }
  renderEmptyState() {
    this.emptyEl = E(this.emptyStateTag).css({ "captionSide": "bottom", height: "100%" });
    this.emptyEl.hidden = this.store.count() > 0;
    this.emptyEl.innerHTML = this.emptyStateHtml;
    this.el.appendChild(this.emptyEl);
  }
  renderBody() {
    this.renderRows(this.store.items);
    if (this.rowSelect) {
      this.rowSelect.on("rowselect", (rowSelect, storeIndex) => {
        const tr = this.getRowElements()[storeIndex];
        if (!tr) {
          return;
        }
        tr.classList.add("selected");
      });
      this.rowSelect.on("rowdeselect", (rowSelect, storeIndex) => {
        const tr = this.getRowElements()[storeIndex];
        if (!tr) {
          console.error("No row found for selected index: " + storeIndex + ". Maybe it's not rendered yet?");
          return;
        }
        tr.classList.remove("selected");
      });
    }
  }
  focusRow(index) {
    const tr = this.getRowElements()[index];
    if (tr) {
      tr.focus();
    }
  }
  renderRows(records) {
    for (let i = 0, l = records.length; i < l; i++) {
      const container = this.renderGroup(records[i]), row = this.renderRow(records[i], i);
      if (this.rowSelection && this.rowSelection.selected.indexOf(i) > -1) {
        row.cls("+selected");
      }
      container.append(row);
      this.onRowAppend(row, records[i], i);
    }
    this.fire("renderrows", this, records);
  }
  onRowAppend(row, record, index) {
  }
  renderGroup(record) {
    return this.el;
  }
  renderRow(record, storeIndex) {
    const row = E(this.itemTag).cls("+data").attr("tabindex", "0");
    if (this.draggable) {
      row.draggable = true;
      row.ondragstart = this.onNodeDragStart.bind(this);
    }
    this.bindDropEvents(row);
    const r = this.renderer(record, row, this, storeIndex);
    if (typeof r === "string") {
      row.innerHTML = r;
    } else if (Array.isArray(r)) {
      r.forEach((c) => {
        c.parent = this;
        c.render(row);
      });
    }
    if (this.rowSelection && this.rowSelection.selected.indexOf(storeIndex) > -1) {
      row.classList.add("selected");
    }
    return row;
  }
  // private onScroll() {
  // 	const el = this.el;
  // 	const pixelsLeft = el.scrollHeight - el.scrollTop - el.offsetHeight;
  //
  // 	if (pixelsLeft < 100) {
  // 		if (!this.store.loading && this.store.hasNext()) {
  // 			this.store.loadNext(true).finally(() => {
  // 				this.fire("scrolleddown", this);
  // 			});
  // 		}
  // 	}
  // }
  onMouseEvent(e, type) {
    const row = this.findRowByEvent(e), index = row ? this.getRowElements().indexOf(row) : -1;
    if (index !== -1) {
      this.fire(type, this, index, row, e);
    }
  }
  findRowByEvent(e) {
    return e.target.closest(".data");
  }
  bindDropEvents(row) {
    row.ondrop = this.onNodeDrop.bind(this);
    row.ondragend = this.onNodeDragEnd.bind(this);
    row.ondragover = this.onNodeDragOver.bind(this);
    row.ondragenter = this.onNodeDragEnter.bind(this);
    row.ondragleave = this.onNodeDragLeave.bind(this);
  }
  onNodeDragStart(e) {
    e.stopPropagation();
    const row = e.target;
    e.dataTransfer.setData("text/plain", "goui");
    dragData.row = row;
    dragData.cmp = this;
    dragData.storeIndex = this.getRowElements().indexOf(row);
    dragData.record = this.store.get(dragData.storeIndex);
    root.el.cls("+dragging");
  }
  onNodeDragEnd(e) {
    root.el.cls("-dragging");
  }
  onNodeDragEnter(e) {
    const dropRow = this.findDropRow(e);
    if (this.dropAllowed(e, dropRow)) {
      e.stopPropagation();
      e.preventDefault();
      dropRow.cls("+drag-over");
      this.onNodeDragEnterAllowed(e, dropRow);
    }
  }
  onNodeDragEnterAllowed(e, dropRow) {
  }
  onNodeDragLeave(e) {
    const dropRow = this.findDropRow(e);
    if (this.dropAllowed(e, dropRow)) {
      e.stopPropagation();
      e.preventDefault();
      this.clearOverClasses(dropRow);
      this.onNodeDragLeaveAllowed(e, dropRow);
    }
  }
  onNodeDragLeaveAllowed(e, dropRow) {
  }
  findDropRow(e) {
    return e.target.closest("LI");
  }
  clearOverClasses(dropRow) {
    dropRow.cls("-drag-over");
    dropRow.classList.remove("before");
    dropRow.classList.remove("after");
    dropRow.classList.remove("on");
    dropPin.hidden = true;
  }
  onNodeDragOver(e) {
    if (!this.dropOn && !this.dropBetween) {
      return;
    }
    const dropRow = this.findDropRow(e);
    if (this.dropAllowed(e, dropRow)) {
      e.stopPropagation();
      e.preventDefault();
      const pos = this.getDropPosition(e);
      dropRow.classList.toggle("before", "before" == pos);
      dropRow.classList.toggle("after", "after" == pos);
      dropRow.classList.toggle("on", "on" == pos);
      dropPin.hidden = pos == "on" || pos == void 0;
      const dropRowRect = dropRow.getBoundingClientRect();
      dropPin.el.style.top = (pos == "before" ? dropRowRect.y : dropRowRect.y + dropRowRect.height - 1) + "px";
      dropPin.el.style.left = dropRowRect.x + "px";
      dropPin.el.style.width = dropRowRect.width + "px";
    }
  }
  dropAllowed(e, dropRow) {
    return this.fire("dropallowed", this, e, dropRow, dragData);
  }
  getDropPosition(e) {
    if (!this.dropBetween) {
      return this.dropOn ? "on" : void 0;
    }
    const betweenZone = 6;
    if (e.offsetY < betweenZone) {
      return "before";
    } else if (e.offsetY > e.target.offsetHeight - betweenZone) {
      return "after";
    } else {
      return this.dropOn ? "on" : void 0;
    }
  }
  onNodeDrop(e) {
    const dropPos = this.getDropPosition(e);
    if (!dropPos) {
      return;
    }
    e.preventDefault();
    e.stopPropagation();
    const dropRow = this.findDropRow(e), dropIndex = this.getRowElements().indexOf(dropRow);
    this.clearOverClasses(dropRow);
    this.fire("drop", this, e, dropRow, dropIndex, dropPos, dragData);
  }
};
var list = (config) => createComponent(new List(config.store, config.renderer), config);

// goui/script/component/table/Table.ts
var Table = class extends List {
  /**
   *
   * @param store Store to provide data
   * @param columns The table columns
   */
  constructor(store3, columns) {
    super(store3, (record, row, me, storeIndex) => {
      for (let c of this.columns) {
        c.parent = this;
        if (c.hidden) {
          continue;
        }
        const td = document.createElement("td");
        if (c.align) {
          td.style.textAlign = c.align;
        }
        if (c.cls) {
          td.classList.add(...c.cls.split(" "));
        }
        let value = c.property ? ObjectUtil.path(record, c.property) : void 0;
        if (c.renderer) {
          const r = c.renderer(value, record, td, this, storeIndex);
          if (typeof r === "string") {
            td.innerHTML = r;
          } else if (r instanceof Component) {
            r.parent = this;
            r.render(td);
          } else {
            r.then((s) => {
              if (s instanceof Component) {
                s.parent = this;
                s.render(td);
              } else {
                td.innerHTML = s;
              }
            });
          }
        } else {
          td.innerText = value ? value : "";
        }
        row.append(td);
      }
    }, "table");
    this.columns = columns;
    /**
     * Make the table fits its container in width by setting min-width: 100%
     * Defaults to true
     */
    this.fitParent = false;
    /**
     * Show headers
     */
    this.headers = true;
    this.emptyStateTag = "caption";
    /**
     * Group renderer function
     */
    this.groupByRenderer = (groupBy) => groupBy != null ? groupBy : t("None");
    this.minCellWidth = 30;
    this.baseCls = "goui-table";
    this.itemTag = "tr";
    this.colsAreFixed = false;
  }
  getRowElements() {
    return Array.from(this.el.getElementsByClassName("data"));
  }
  internalRemove() {
    if (this.columnMenu) {
      this.columnMenu.remove();
    }
    return super.internalRemove();
  }
  restoreState(state) {
    if (state.sort) {
      this.store.sort = state.sort;
    }
    if (state.columns) {
      for (let id in state.columns) {
        let col = this.findColumnById(id);
        if (col) {
          Object.assign(col, state.columns[id]);
        }
      }
    }
  }
  /**
   * Find column by "property" property.
   *
   * It's the property path of the data linked to the column
   *
   * @param id
   */
  findColumnById(id) {
    return this.columns.find((col) => {
      return col.id == id;
    });
  }
  buildState() {
    const cols = {};
    this.columns.forEach((c) => {
      cols[c.id] = {
        width: c.width,
        hidden: c.hidden
      };
    });
    return {
      sort: this.store.sort,
      columns: cols
    };
  }
  renderBody() {
    if (this.fitParent) {
      this.el.style.minWidth = "100%";
    }
    if (this.headers) {
      this.renderHeaders();
    } else {
      this.renderColGroup();
    }
    super.renderBody();
  }
  rerender() {
    const el = this.el;
    this.groupEl = void 0;
    el.innerHTML = "";
    this.renderBody();
  }
  showColumnMenu(ev) {
    ev.preventDefault();
    if (!this.columnMenu) {
      this.columnMenu = menu({
        isDropdown: true,
        removeOnClose: false
      });
      this.columns.forEach((c) => {
        if (c.header && c.hidable) {
          this.columnMenu.items.add(checkbox({
            label: c.header,
            name: c.id,
            value: !c.hidden,
            listeners: {
              change: (field) => {
                c.hidden = !field.value;
                this.saveState();
                this.rerender();
              }
            }
          }));
        }
      });
    }
    this.columnMenu.showAt(ev);
  }
  createColumnSplitter(h, header, colIndex) {
    if (h.resizable) {
      const splitter2 = draggable({
        tagName: "hr",
        setPosition: false,
        parent: this
      });
      splitter2.on("dragstart", (cmp, dragData2) => {
        if (!this.colsAreFixed) {
          this.fixColumnWidths();
        }
        dragData2.data.startWidth = header.offsetWidth;
        splitter2.dragConstrainTo(this.headersRow, {
          left: this.calcTableWidth(colIndex) + this.minCellWidth,
          right: -1e4
        });
      });
      splitter2.on("drag", (cmp, dragData2) => {
        const w = dragData2.data.startWidth + dragData2.x - dragData2.startX;
        header.style.width = w + "px";
        h.width = w;
        this.el.style.width = this.calcTableWidth() + "px";
      });
      splitter2.on("drop", () => {
        this.saveState();
      });
      return splitter2;
    } else {
      return comp({ tagName: "hr" });
    }
  }
  renderColGroup() {
    const colGroup = document.createElement("colgroup");
    let index = -1;
    for (let h of this.columns) {
      index++;
      if (h.hidden) {
        continue;
      }
      const col = document.createElement("col");
      if (h.width) {
        col.style.width = h.width + "px";
      }
      if (h.align) {
        col.style.textAlign = h.align;
      }
      if (h.cls) {
        col.classList.add(...h.cls.split(" "));
      }
      colGroup.appendChild(col);
    }
    this.el.appendChild(colGroup);
    return colGroup;
  }
  renderHeaders() {
    const thead = document.createElement("thead");
    this.headersRow = document.createElement("tr");
    this.headersRow.addEventListener("contextmenu", (ev) => {
      this.showColumnMenu(ev);
    });
    let index = -1;
    for (let h of this.columns) {
      index++;
      if (h.hidden) {
        continue;
      }
      const header = document.createElement("th");
      if (h.width) {
        header.style.width = h.width + "px";
      }
      if (h.align) {
        header.style.textAlign = h.align;
      }
      if (h.cls) {
        header.classList.add(...h.cls.split(" "));
      }
      if (h.headerRenderer) {
        const r = h.headerRenderer(h, header, this);
        if (typeof r === "string") {
          header.innerHTML = r;
        } else if (r instanceof Component) {
          r.render(header);
        }
      } else {
        header.innerHTML = h.header || "";
      }
      h.headerEl = header;
      if (h.resizable) {
        const splitter2 = this.createColumnSplitter(h, header, index);
        splitter2.render(header);
      } else {
        comp({ tagName: "hr" }).render(header);
      }
      if (h.sortable && h.property) {
        header.addEventListener("click", () => {
          this.onSort(h.property, header);
        });
        const sort = this.store.sort;
        if (sort.length) {
          sort.forEach((comparator) => {
            if (h.property == comparator.property) {
              header.classList.add("sorted");
              header.classList.add(comparator.isAscending || comparator.isAscending === void 0 ? "asc" : "desc");
            }
          });
        }
      }
      this.headersRow.appendChild(header);
    }
    thead.appendChild(this.headersRow);
    this.el.appendChild(thead);
    return this.headersRow;
  }
  onSort(dataIndex, header) {
    this.fire("sort", this, dataIndex);
    const s = this.store.sort;
    let sortIndex = s.length - 1 || 0;
    if (s[sortIndex]) {
      if (s[sortIndex].property == dataIndex) {
        s[sortIndex].isAscending = !s[sortIndex].isAscending;
      } else {
        s[sortIndex].property = dataIndex;
      }
    } else {
      s[sortIndex] = {
        isAscending: true,
        property: dataIndex
      };
    }
    this.headersRow.childNodes.forEach((node) => {
      let th = node;
      if (th == header) {
        th.classList.add("sorted");
        th.classList.remove(s[sortIndex].isAscending ? "desc" : "asc");
        th.classList.add(s[sortIndex].isAscending ? "asc" : "desc");
      } else {
        th.classList.remove("sorted");
        th.classList.remove("asc");
        th.classList.remove("desc");
      }
    });
    this.store.reload().catch((reason) => {
      Notifier.error(reason);
    });
    this.saveState();
  }
  /**
   * When resizing columns we must calculate absolute pixel widths
   *
   * @private
   */
  fixColumnWidths() {
    this.columns.forEach((col) => {
      if (!col.hidden) {
        col.width = col.headerEl.offsetWidth;
        col.headerEl.style.width = col.width + "px";
      }
    });
    this.el.style.minWidth = "";
    this.el.style.width = this.calcTableWidth() + "px";
  }
  /**
   * Returns the sum of column widths
   *
   * @param untilColumnIndex Calc width until this column
   */
  calcTableWidth(untilColumnIndex = -1) {
    return this.columns.reduce((previousValue, nextValue, currentIndex) => {
      if (nextValue.hidden || untilColumnIndex > -1 && currentIndex >= untilColumnIndex) {
        return previousValue;
      }
      return previousValue + nextValue.width;
    }, 0);
  }
  renderGroup(record) {
    if (!this.groupBy) {
      if (!this.groupEl) {
        this.groupEl = document.createElement("tbody");
        this.el.append(this.groupEl);
      }
      return this.groupEl;
    }
    const groupBy = ObjectUtil.path(record, this.groupBy);
    if (!this.groupEl || groupBy != this.lastGroup) {
      const tr = document.createElement("tr");
      tr.classList.add("group");
      const th = document.createElement("th");
      th.colSpan = this.columns.length;
      const r = this.groupByRenderer(groupBy, record, th, this);
      if (typeof r === "string") {
        th.innerHTML = r;
      } else if (r instanceof Component) {
        r.render(th);
      } else if (r instanceof Promise) {
        r.then((s) => {
          if (s instanceof Component) {
            s.render(th);
          } else {
            th.innerHTML = s;
          }
        });
      } else {
        console.warn("Renderer returned invalid value: ", r);
      }
      tr.append(th);
      this.groupEl = document.createElement("tbody");
      this.groupEl.append(tr);
      this.el.append(this.groupEl);
      this.lastGroup = groupBy;
    }
    return this.groupEl;
  }
  onRecordRemove(collection, item, index) {
    var _a;
    let groupEl;
    if (this.groupBy) {
      const rows = this.getRowElements();
      groupEl = (_a = rows[index]) == null ? void 0 : _a.parentElement;
    }
    super.onRecordRemove(collection, item, index);
    if (groupEl && groupEl.children.length == 1) {
      if (groupEl == this.groupEl) {
        this.groupEl = void 0;
      }
      groupEl.remove();
    }
  }
  findDropRow(e) {
    return e.target.closest("TR");
  }
};
var table = (config) => createComponent(new Table(config.store, config.columns), config);

// goui/script/component/table/TableColumns.ts
var TableColumn = class extends Observable {
  /**
   *
   * The column ID. Also used for 'property'
   */
  constructor(id) {
    super();
    this.id = id;
    /**
     * Make the column resizable by the user
     */
    this.resizable = false;
    /**
     * Make it sortable by the user
     */
    this.sortable = false;
    /**
     * Text alignment
     */
    this.align = "left";
    /**
     * Hide the column. It can be enabled by the user via the context menu.
     */
    this.hidden = false;
    /**
     * Enable this column in the enabled columns menu
     */
    this.hidable = true;
    this.property = id;
  }
};
var column = (config) => createComponent(new TableColumn(config.id), config);
var DateTimeColumn = class extends TableColumn {
  constructor() {
    super(...arguments);
    this.renderer = (date) => {
      return Format.dateTime(date);
    };
    //argh!? https://stackoverflow.com/questions/43121661/typescript-type-inference-issue-with-string-literal
    this.align = "right";
    this.width = 190;
  }
};
var datetimecolumn = (config) => createComponent(new DateTimeColumn(config.id), config);
var DateColumn = class extends TableColumn {
  constructor() {
    super(...arguments);
    this.renderer = (date) => {
      return Format.date(date);
    };
    //argh!? https://stackoverflow.com/questions/43121661/typescript-type-inference-issue-with-string-literal
    this.align = "right";
    this.width = 128;
  }
};
var datecolumn = (config) => createComponent(new DateColumn(config.id), config);
var CheckboxColumn = class extends TableColumn {
  constructor(id) {
    super(id);
    this.renderer = (val) => {
      return checkbox({
        value: val
      });
    };
    this.cls = "checkbox-select-column";
  }
};
var checkboxcolumn = (config) => createComponent(new CheckboxColumn(config.id), config);
var CheckboxSelectColumn = class extends TableColumn {
  constructor(id = "checkboxselect") {
    super(id);
    this.headerRenderer = (col, headerEl, table2) => {
      return checkbox({
        listeners: {
          change: (field, newValue, oldValue) => {
            if (newValue) {
              table2.rowSelection.selectAll();
            } else {
              table2.rowSelection.clear();
            }
          }
        }
      });
    };
    this.renderer = (val, record, td, table2, rowIndex) => {
      if (val && table2.rowSelection) {
        table2.rowSelection.add(rowIndex);
      }
      return checkbox({
        value: val,
        listeners: {
          render: (field) => {
            field.el.addEventListener("mousedown", (ev) => {
              ev.stopPropagation();
            });
            table2.rowSelection.on("selectionchange", () => {
              field.value = table2.rowSelection.selected.indexOf(rowIndex) > -1;
            });
          },
          change: (field, newValue, oldValue) => {
            const index = table2.store.indexOf(record);
            if (newValue) {
              table2.rowSelection.add(index);
            } else {
              table2.rowSelection.remove(index);
            }
          }
        }
      });
    };
    this.hidable = false;
    this.cls = "checkbox-select-column";
  }
};
var checkboxselectcolumn = (config) => createComponent(new CheckboxSelectColumn(config && config.id ? config.id : "checkboxselect"), config);
var menucolumn = (...items) => column({
  width: 48,
  id: "btn",
  // headerRenderer: (col: TableColumn, headerEl: HTMLTableCellElement, table: Table) => {
  // 	headerEl.style.position = "sticky";
  // 	headerEl.style.right = "0";
  // 	return "";
  // },
  renderer: (columnValue, record, td, table2, rowIndex) => {
    items.forEach((i) => {
      i.dataSet.table = table2;
      i.dataSet.rowIndex = rowIndex;
    });
    return btn({
      icon: "more_vert",
      menu: menu({}, ...items)
    });
  }
});

// goui/script/component/form/ComboBox.ts
var ComboBoxDefaultRenderer = (field, r) => r[field.displayProperty];
var ComboBox = class extends AutocompleteField {
  constructor(dataSource, displayProperty = "name", valueProperty = "id", renderer = ComboBoxDefaultRenderer, storeConfig = {
    queryParams: {
      limit: 50
    }
  }) {
    storeConfig.dataSource = dataSource;
    const dropDownTable = table({
      headers: false,
      fitParent: true,
      store: datasourcestore(storeConfig),
      columns: [
        column({
          id: displayProperty,
          resizable: true,
          width: 312,
          sortable: true,
          renderer: (columnValue, record, td, table1) => {
            return Format.escapeHTML(renderer(this, record));
          }
        })
      ]
    });
    super(dropDownTable);
    this.dataSource = dataSource;
    this.displayProperty = displayProperty;
    this.valueProperty = valueProperty;
    this.renderer = renderer;
    /**
     * When autocompleting from the datasource this filter name will be used.
     */
    this.filterName = "text";
    this.picker.list.store.addScrollLoader(this.menu.el);
    this.on("autocomplete", (_field, input) => __async(this, null, function* () {
      var _a;
      this.list.store.queryParams.position = 0;
      if (!this.list.store.queryParams.filter) {
        this.list.store.queryParams.filter = {};
      }
      if (this.filter) {
        Object.assign(this.list.store.queryParams.filter, this.filter);
      }
      this.list.store.queryParams.filter[(_a = this.filterName) != null ? _a : this.displayProperty] = input;
      yield this.list.store.load();
    }));
  }
  pickerRecordToValue(_field, record) {
    return record[this.valueProperty];
  }
  valueToTextField(_field, value) {
    return __async(this, null, function* () {
      if (value === "") {
        return "";
      }
      const record = yield this.dataSource.single(value);
      return this.renderer(_field, record);
    });
  }
};
var combobox = (config) => {
  var _a, _b, _c, _d;
  return createComponent(new ComboBox(config.dataSource, (_a = config.displayProperty) != null ? _a : "name", (_b = config.valueProperty) != null ? _b : "id", (_c = config.renderer) != null ? _c : ComboBoxDefaultRenderer, (_d = config.storeConfig) != null ? _d : {
    queryParams: {
      limit: 50
    }
  }), config);
};

// goui/script/component/form/DurationField.ts
var DurationField = class extends Field {
  /**
   *
   * @param outputFormat Format it will as value
   *
   * {@link DurationField.value}
   *
   * It can be any string format supported by {@link DateInterval.format}
   */
  constructor(outputFormat = "h:I") {
    super();
    this.outputFormat = outputFormat;
    this.baseCls = "goui-form-field duration no-floating-label";
    /**
     * Minimum allowed duration to be entered
     */
    this.min = void 0;
    /**
     * Maximum allowed duration to be entered
     */
    this.max = void 0;
  }
  validate() {
    super.validate();
    const v = this.getValueAsDateInterval();
    if (this.max !== void 0 && v.compare(this.max) == 1) {
      this.setInvalid(t("The maximum duration is {duration}").replace("{duration}", this.max.format("h:I")));
    }
    if (this.min !== void 0 && v.compare(this.min) == -1) {
      this.setInvalid(t("The minimum duration is {duration}").replace("{duration}", this.min.format("h:I")));
    }
  }
  /**
   * Get the value DateInterval object
   */
  getValueAsDateInterval() {
    const di = new DateInterval();
    di.hours = parseInt(this.hoursInput.value);
    di.minutes = parseInt(this.minutesInput.value);
    return di;
  }
  createControl() {
    const ctrl = E("div").cls("goui hbox");
    const onBlur = function() {
      if (!this.value) {
        return;
      }
      this.value = this.value.padStart(2, "0");
      return true;
    };
    const onFocus = function() {
      this.setSelectionRange(0, this.value.length);
    };
    const onKeyDown = (ev) => {
      switch (ev.key) {
        case "Tab":
        case "Enter":
        case "Backspace":
        case "Delete":
          return true;
        case ":":
          this.minutesInput.focus();
          ev.preventDefault();
          break;
        default:
          if (!/^[0-9]$/i.test(ev.key)) {
            ev.preventDefault();
          }
      }
    };
    this.hoursInput = document.createElement("input");
    this.hoursInput.classList.add("text");
    this.hoursInput.classList.add("hour");
    this.hoursInput.inputMode = "numeric";
    this.hoursInput.type = "text";
    this.hoursInput.pattern = "[0-9]+";
    this.hoursInput.onblur = onBlur;
    this.hoursInput.onfocus = onFocus;
    this.hoursInput.placeholder = "--";
    this.hoursInput.autocomplete = "off";
    this.hoursInput.onkeydown = onKeyDown;
    this.minutesInput = document.createElement("input");
    this.minutesInput.classList.add("text");
    this.minutesInput.classList.add("minute");
    this.minutesInput.inputMode = "numeric";
    this.minutesInput.type = "text";
    this.minutesInput.pattern = "[0-9]+";
    this.minutesInput.maxLength = 2;
    const hoursInput = this.hoursInput;
    this.minutesInput.onblur = function() {
      onBlur.call(this);
      if (!this.value && hoursInput.value) {
        this.value = "00";
      }
    };
    this.minutesInput.onfocus = onFocus;
    this.hoursInput.onkeydown = onKeyDown;
    this.minutesInput.oninput = (_ev) => {
      if (parseInt(this.minutesInput.value) > 59) {
        this.minutesInput.value = "59";
      }
    };
    this.minutesInput.placeholder = "--";
    this.minutesInput.autocomplete = "off";
    ctrl.append(this.hoursInput, ":", this.minutesInput);
    return ctrl;
  }
  internalSetValue(v) {
    if (v && this.hoursInput && this.minutesInput) {
      const dateInterval = DateInterval.createFromFormat(v, this.outputFormat);
      if (dateInterval) {
        this.hoursInput.value = dateInterval.format("h");
        this.minutesInput.value = dateInterval.format("I");
      } else {
        this.minutesInput.value = "00";
      }
    }
  }
  set value(v) {
    super.value = v;
  }
  get value() {
    if (!this.hoursInput.value) {
      return void 0;
    }
    const dateInterval = new DateInterval();
    dateInterval.hours = parseInt(this.hoursInput.value);
    dateInterval.minutes = parseInt(this.minutesInput.value);
    return dateInterval.format(this.outputFormat);
  }
};
var durationfield = (config) => {
  var _a;
  return createComponent(new DurationField((_a = config == null ? void 0 : config.outputFormat) != null ? _a : "h:I"), config);
};

// goui/script/component/form/ListField.ts
var ListField = class _ListField extends Field {
  constructor(list2, valueProperty = "id") {
    super();
    this.list = list2;
    this.valueProperty = valueProperty;
    this.baseCls = "goui-form-field listfield";
    this.picker = listpicker({
      list: this.list
    });
    this.picker.on("select", (tablePicker, record) => {
      tablePicker.list.findAncestorByType(Menu).hide();
      this.focus();
      this.value = this.pickerRecordToValue(this, record);
      this.fire("select", this, record);
    });
    this.menu = menu(
      {
        height: 300,
        cls: "scroll",
        listeners: {
          hide: (menu2) => {
            if (menu2.rendered) {
              const f = menu2.findAncestorByType(_ListField);
              f.focus();
            }
          }
        }
      },
      this.picker
    );
    this.menuButton = btn({
      icon: "expand_more",
      type: "button",
      handler: () => {
        void this.list.store.load();
        this.picker.show();
      },
      menu: this.menu
    });
  }
  internalRender() {
    this.buttons = this.buttons || [];
    this.buttons.push(this.menuButton);
    const el = super.internalRender();
    this.menu.alignTo = this.wrap;
    this.menu.alignToInheritWidth = true;
    return el;
  }
  pickerRecordToValue(_field, record) {
    return record[this.valueProperty];
  }
  createControl() {
    this.container = document.createElement("div");
    return this.container;
  }
  internalSetValue(v) {
    if (v == void 0) {
      return super.internalSetValue(v);
    }
    this.renderValue(this, v + "").then((v2) => {
      if (this.container)
        this.container.innerHTML = v2;
    });
  }
  /**
   * This method transforms the value in to a HTML representation
   *
   * @param field
   * @param value
   */
  renderValue(field, value) {
    return __async(this, null, function* () {
      return "";
    });
  }
};
var listfield = (config) => createComponent(new ListField(config.list), config);

// goui/script/component/form/HiddenField.ts
var HiddenField = class extends Field {
  constructor() {
    super();
    this.baseCls = "";
    this.hidden = true;
  }
};
var hiddenfield = (config) => createComponent(new HiddenField(), config);

// goui/script/component/Avatar.ts
var _Avatar = class _Avatar extends Component {
  constructor() {
    super(...arguments);
    this.baseCls = "goui-avatar";
  }
  set displayName(displayName) {
    this.text = this.initials(displayName);
    this.title = displayName;
    let j = 0;
    for (let i = 0, l = this.displayName.length; i < l; i++) {
      j += displayName.charCodeAt(i);
    }
    this.el.style.backgroundColor = "#" + _Avatar.colors[j % _Avatar.colors.length];
  }
  get displayName() {
    return this.title;
  }
  set backgroundImage(imgUrl) {
    this.el.style.backgroundImage = imgUrl ? "url(" + imgUrl + ")" : "none";
    if (imgUrl) {
      this.text = "";
    }
  }
  get backgroundImage() {
    return this.el.style.backgroundImage;
  }
  /**
   * Grabs the first char of the first and last word.
   *
   * @param {string} name
   * @returns {string}
   */
  initials(name) {
    const parts = name.split(" "), l = parts.length;
    if (l > 2) {
      parts.splice(1, l - 2);
    }
    return parts.map((name2) => name2.substr(0, 1).toUpperCase()).join("");
  }
};
_Avatar.colors = [
  "C62828",
  "AD1457",
  "6A1B9A",
  "4527A0",
  "283593",
  "1565C0",
  "0277BD",
  "00838F",
  "00695C",
  "2E7D32",
  "558B2F",
  "9E9D24",
  "F9A825",
  "FF8F00",
  "EF6C00",
  "424242"
];
var Avatar = _Avatar;
var avatar = (config) => createComponent(new Avatar(), config);

// goui/script/component/CardMenu.ts
var CardMenu = class extends Component {
  constructor() {
    super("menu");
    this.baseCls = "goui-cardmenu";
    this.on("beforerender", () => {
      if (!this.cardContainer) {
        this.cardContainer = this.parent.findChildByType(CardContainer);
      }
      this.cardContainer.on("cardchange", (cardContainer, index) => {
        this.updateActiveTab();
      });
      this.createMenu();
      this.cardContainer.items.on("remove", (collection, item, index) => {
        const cardMenuItem = this.findItem("card-" + index);
        if (cardMenuItem) {
          cardMenuItem.remove();
        }
      });
      this.cardContainer.items.on("add", (collection, item, index) => {
        this.createMenu();
      });
      this.cardContainer.on("beforerender", () => {
        this.updateActiveTab();
      });
    });
  }
  focus(o) {
    var _a;
    const first = (_a = this.cardContainer) == null ? void 0 : _a.items.first();
    if (first) {
      first.focus(o);
    } else {
      super.focus(o);
    }
  }
  updateActiveTab() {
    const activeItem = this.cardContainer.items.get(this.cardContainer.activeItem);
    this.items.forEach((item, menuIndex) => {
      if (activeItem && (item.itemId == activeItem.itemId || item.itemId == activeItem.id)) {
        item.el.classList.add("active");
      } else {
        item.el.classList.remove("active");
      }
    });
  }
  createMenu() {
    this.cardContainer.items.forEach((item, index) => {
      if (!item.itemId) {
        item.itemId = "card-" + index;
      }
      if (this.findItem(item.itemId)) {
        return;
      }
      this.items.insert(
        index,
        btn({
          type: "button",
          itemId: item.itemId,
          cls: index == this.cardContainer.activeItem ? "active" : "",
          text: item.title,
          handler: () => {
            this.cardContainer.activeItem = item;
          }
        })
      );
      item.title = "";
    });
  }
};
var cardmenu = (config, ...items) => createComponent(new CardMenu(), config, items);

// goui/script/component/DescriptionList.ts
var DescriptionList = class extends Component {
  constructor() {
    super("dl");
    this.baseCls = "goui-dl";
  }
  internalRender() {
    const el = super.internalRender();
    this.renderList();
    return el;
  }
  /**
   * Set the records to display
   *
   * @example
   * ```
   * const records: DLRecord = [
   * 			['Number', record.number],
   * 			['Description', record.description],
   * 			['Created At', Format.date(record.createdAt)]
   * 		];
   * dl.setRecords(records);
   * ```
   * @param records
   */
  set records(records) {
    this._records = records;
    this.renderList();
  }
  get records() {
    return this._records || [];
  }
  renderList() {
    this.items.clear();
    this.records.forEach((record) => {
      this.items.add(comp({
        tagName: "dt",
        text: record.shift()
      }));
      record.forEach((r) => {
        const dd = comp({
          tagName: "dd"
        });
        if (typeof r == "function")
          r(dd);
        else {
          dd.text = r + "";
        }
        this.items.add(dd);
      });
    });
  }
};
var dl = (config, ...items) => createComponent(new DescriptionList(), config, items);

// goui/script/component/Paginator.ts
var Paginator = class extends Toolbar {
  constructor(store3) {
    super();
    this.store = store3;
    this.baseCls = "goui-toolbar goui-paginator";
    this.items.add(
      this.prev = btn({
        icon: "chevron_left",
        text: "Previous",
        disabled: true,
        handler: () => __async(this, null, function* () {
          yield this.store.loadPrevious();
        })
      }),
      comp({
        flex: 1
      }),
      this.next = btn({
        icon: "chevron_right",
        text: "Next",
        disabled: true,
        handler: () => __async(this, null, function* () {
          yield this.store.loadNext();
        })
      })
    );
    this.store.on("load", () => {
      this.onStoreLoad();
    });
  }
  onStoreLoad() {
    this.prev.disabled = !this.store.hasPrevious();
    this.next.disabled = !this.store.hasNext();
  }
};
var paginator = (config) => createComponent(new Paginator(config.store), config);

// goui/script/component/SearchButton.ts
var SearchButton = class extends Button {
  constructor() {
    super();
    this.buffer = 300;
    this.handler = (button, ev) => {
      this.mainTbar = button.parent;
      this.getSearchTBar().show();
      this.searchField.focus();
    };
    this.icon = "search";
    this.title = t("Search");
    this.searchField = textfield({
      label: t("Search"),
      flex: 1,
      buttons: [
        btn({
          type: "button",
          icon: "clear",
          handler: () => {
            this.reset();
          }
        })
      ]
    });
    this.searchField.on("render", () => {
      this.searchField.input.addEventListener("input", FunctionUtil.buffer(this.buffer, this.onInput.bind(this)));
      this.searchField.el.addEventListener("keydown", (e) => {
        if (e.key == "Enter") {
          e.preventDefault();
          e.stopPropagation();
        }
        if (e.key == "Escape") {
          e.preventDefault();
          e.stopPropagation();
          this.close();
        }
      });
    });
  }
  reset() {
    this.searchField.reset();
    this.close();
    this.fire("reset", this);
    this.fire("input", this, "");
    this.el.classList.remove("accent");
    this.el.classList.remove("filled");
  }
  close() {
    this.searchTBar.hide();
    this.mainTbar.show();
    this.focus();
  }
  onInput() {
    this.el.classList.toggle("accent", !!this.searchField.value);
    this.el.classList.toggle("filled", !!this.searchField.value);
    this.fire("input", this, this.searchField.value);
  }
  getSearchTBar() {
    if (!this.searchTBar) {
      this.searchTBar = tbar(
        {
          cls: "search"
        },
        btn({
          icon: "chevron_left",
          title: t("Back"),
          handler: () => {
            this.close();
          }
        }),
        this.searchField
      );
      this.mainTbar.items.add(this.searchTBar);
    }
    return this.searchTBar;
  }
};
var searchbtn = (config) => createComponent(new SearchButton(), config);

// goui/script/component/Splitter.ts
var Splitter = class extends DraggableComponent {
  /**
   *
   * @param resizeComponentPredicate 	The component to resize in height or width.
   *    Can be a component ID, itemId property, Component instance or custom function
   *
   */
  constructor(resizeComponentPredicate) {
    super("hr");
    this.resizeComponentPredicate = resizeComponentPredicate;
    /**
     * The minimum size it will set. Note that you can also put a min-width or min-height on the element with css.
     */
    this.minSize = 50;
    this.on("added", (comp2, index, parent) => {
      parent.on("beforerender", () => {
        this.findResizeComponent(resizeComponentPredicate);
      });
    });
  }
  findResizeComponent(resizeComponentPredicate) {
    if (!(resizeComponentPredicate instanceof Component)) {
      this._resizeComponent = this.parent.findChild(this.resizeComponentPredicate);
      if (!this._resizeComponent) {
        console.warn(this.resizeComponentPredicate);
        throw "Splitter could not find component to resize!";
      }
    } else {
      this._resizeComponent = resizeComponentPredicate;
    }
    const state = this.getState();
    if (state) {
      this.applyStateToResizeComp(state);
    }
  }
  applyStateToResizeComp(state) {
    if (this._resizeComponent) {
      if (state) {
        if (state.width)
          this._resizeComponent.width = state.width;
        if (state.height)
          this._resizeComponent.height = state.height;
      }
    }
  }
  buildState() {
    return this.resizeWidth ? { width: this._resizeComponent.width } : { height: this._resizeComponent.height };
  }
  internalRender() {
    const el = super.internalRender();
    this.on("dragstart", (comp2, dragData2, e) => {
      if (this.resizeWidth === void 0) {
        this.resizeWidth = this.el.offsetHeight > this.el.offsetWidth;
      }
      if (this.invert === void 0) {
        const splitterIndex = this.parent.findItemIndex(this);
        const resizeCmpIndex = this.parent.findItemIndex(this._resizeComponent);
        this.invert = splitterIndex < resizeCmpIndex;
      }
      if (this.resizeWidth) {
        dragData2.data.startWidth = this._resizeComponent.el.offsetWidth;
      } else {
        dragData2.data.startHeight = this._resizeComponent.el.offsetHeight;
      }
    });
    this.on("drag", (dc, dragData2, ev) => {
      if (this.resizeWidth) {
        let offset = dragData2.x - dragData2.startX;
        if (this.invert) {
          offset *= -1;
        }
        let width = Math.max(this.minSize, dragData2.data.startWidth + offset);
        if (this.maxSize) {
          width = Math.min(this.maxSize, width);
        }
        this._resizeComponent.width = width * 10 / REM_UNIT_SIZE;
      } else {
        let offset = dragData2.y - dragData2.startY;
        if (this.invert) {
          offset *= -1;
        }
        let height = Math.max(this.minSize, dragData2.data.startHeight + offset);
        if (this.maxSize) {
          height = Math.min(this.maxSize, height);
        }
        this._resizeComponent.height = height * 10 / REM_UNIT_SIZE;
      }
    });
    this.on("drop", () => {
      this.saveState();
    });
    return el;
  }
};
var splitter = (config) => createComponent(new Splitter(config.resizeComponentPredicate), config);

// goui/script/component/Tree.ts
var TreeRowRenderer = (record, row, me, storeIndex) => {
  const node = E("div").cls("node"), caret = E("span").cls("caret"), icon = E("i").cls("icon"), label = E("a");
  icon.innerText = record.icon || "folder";
  label.innerText = record.text;
  if (record.href) {
    label.href = record.href;
  }
  if (record.children && record.children.length == 0) {
    row.cls("+no-children");
  }
  if (record.cls) {
    row.cls("+" + record.cls);
  }
  node.append(caret, icon);
  if (record.check != void 0) {
    const c = checkbox({
      value: record.check,
      listeners: {
        change: (field, newValue, oldValue) => {
          record.check = newValue;
          const top = me.findTopTree();
          top.fire("checkchange", top, void 0, record, storeIndex, newValue);
        }
      }
    });
    c.render(node);
  }
  node.append(label);
  row.append(node);
};
var Tree = class _Tree extends List {
  constructor(renderer = TreeRowRenderer) {
    super(store(), renderer);
    this.renderer = renderer;
    this.subTrees = {};
    this.expandedIds = {};
    this.parentStoreIndex = -1;
    this.baseCls = "goui goui-tree";
    this.on("rowclick", (list2, storeIndex, row, ev) => {
      if (list2 == this) {
        void this._expand(row);
      }
    });
    this.store.on("remove", (collection, item, index) => {
      delete this.subTrees[index];
    });
    if (!(this.parent instanceof _Tree)) {
      this.on("beforerender", () => {
        this.fire("beforeexpand", this, this, void 0, -1);
      });
      this.on("render", () => {
        this.fire("expand", this, this, void 0, -1);
      });
    }
    this.emptyStateHtml = "";
  }
  set data(records) {
    this.store.loadData(records);
  }
  /**
   * The full hierarchical store data of the tree
   */
  get data() {
    return this.store.data;
  }
  /**
   * Reload the tree if loaded via the "expand" event.
   */
  reload() {
    this.store.clear();
    this.fireExpand();
    for (let id in this.subTrees) {
      this.subTrees[id].reload();
    }
  }
  fireExpand() {
    if (!(this.parent instanceof _Tree)) {
      this.fire("beforeexpand", this, this, void 0, -1);
      this.fire("expand", this, this, void 0, -1);
    } else {
      const record = this.parent.store.get(this.parentStoreIndex), top = this.findTopTree();
      top.fire("beforeexpand", top, this, record, this.parentStoreIndex);
      top.fire("expand", top, this, record, this.parentStoreIndex);
    }
  }
  /**
   * Find the first menu in the tree of submenu's
   */
  findTopTree() {
    if (this.parent instanceof _Tree) {
      return this.parent.findTopTree();
    } else {
      return this;
    }
  }
  /**
   * Expand this tree or a child node when index is given
   *
  	 * @param index
   */
  expand(index) {
    if (index === void 0 || index == -1) {
      if (this.parent instanceof _Tree) {
        this.parent.expand(this.parentStoreIndex);
      }
      return;
    }
    this._expand(this.getRowElements()[index]);
  }
  /**
   * Collapse this tree or a child node when index is given
   *
   * @param index
   */
  collapse(index) {
    if (!index || index == -1) {
      if (this.parent instanceof _Tree) {
        this.parent.collapse(this.parentStoreIndex);
      }
      return;
    }
    this._collapse(this.getRowElements()[index]);
  }
  _expand(row) {
    const storeIndex = this.getRowElements().indexOf(row);
    const record = this.store.get(storeIndex);
    if (!record) {
      debugger;
      throw "Record not found for index: " + storeIndex;
    }
    const tree2 = this.subTrees[storeIndex] ? this.subTrees[storeIndex] : this.renderSubTree(row, record, storeIndex);
    const top = this.findTopTree();
    if (top.fire("beforeexpand", top, tree2, record, storeIndex) === false) {
      return tree2;
    }
    row.cls("+expanded");
    tree2.el.style.height = tree2.el.scrollHeight + "px";
    tree2.el.addEventListener("transitionend", () => {
      if (row.has(".expanded")) {
        tree2.el.style.height = "auto";
      }
    }, { once: true });
    top.fire("expand", top, tree2, record, storeIndex);
    return tree2;
  }
  renderSubTree(row, record, storeIndex) {
    if (!record.id) {
      if (this.parentStoreIndex > -1) {
        record.id = this.parent.store.get(this.parentStoreIndex).id + "-" + storeIndex;
      } else {
        record.id = storeIndex + "";
      }
    }
    this.findTopTree().expandedIds[record.id] = true;
    const sub = new _Tree(this.renderer);
    sub.dropOn = this.dropOn;
    sub.dropBetween = this.dropBetween;
    sub.draggable = this.draggable;
    sub.parent = this;
    sub.parentStoreIndex = storeIndex;
    if (record.children) {
      sub.data = record.children;
    }
    sub.store.on("load", (store1, records, append) => {
      record.children = store1.data;
    });
    ["rowclick", "rowdblclick", "rowcontextmenu", "rowmousedown"].forEach((e) => {
      this.relayEvent(sub, e);
    });
    this.subTrees[storeIndex] = sub;
    const wrap = document.createElement("div");
    wrap.classList.add("wrap");
    row.append(wrap);
    sub.render(wrap);
    return sub;
  }
  _collapse(row) {
    row.cls("-expanded");
    const storeIndex = this.getRowElements().indexOf(row);
    this.subTrees[storeIndex].el.style.height = this.subTrees[storeIndex].el.offsetHeight + "px";
    requestAnimationFrame(() => {
      this.subTrees[storeIndex].el.style.height = "0";
    });
    const record = this.store.get(storeIndex);
    if (!record) {
      throw "Record not found";
    }
    const top = this.findTopTree();
    top.expandedIds[record.id] = true;
    top.fire("collapse", top, this, record, storeIndex);
  }
  renderRow(record, storeIndex) {
    let row = super.renderRow(record, storeIndex);
    row.cls("+data");
    row.id = Component.uniqueID();
    if (this.draggable) {
      row.draggable = true;
      row.ondragstart = this.onNodeDragStart.bind(this);
    }
    row.addEventListener("contextmenu", (e) => {
      e.stopPropagation();
      e.preventDefault();
      const sub = this._expand(row);
      sub.reload();
    });
    this.bindDropEvents(row);
    row.getElementsByClassName("caret")[0].on("click", (e) => {
      row.has(".expanded") ? this._collapse(row) : this._expand(row);
      e.preventDefault();
      e.stopPropagation();
    });
    return row;
  }
  onRowAppend(row, record, index) {
    if (this.findTopTree().expandedIds[record.id]) {
      this._expand(row);
    }
  }
  onNodeDragEnterAllowed(e, dropRow) {
    clearTimeout(this.dragOverTimeout);
    setTimeout(() => {
      this.dragOverTimeout = setTimeout(() => {
        this._expand(dropRow);
      }, 1e3);
    });
  }
  onNodeDragLeaveAllowed(e, dropRow) {
    clearTimeout(this.dragOverTimeout);
  }
  dropAllowed(e, dropRow) {
    const topTree = this.findTopTree();
    return dragData.row && !dragData.row.contains(dropRow);
  }
  onNodeDrop(e) {
    const dropPos = this.getDropPosition(e);
    if (!dropPos) {
      return;
    }
    e.preventDefault();
    e.stopPropagation();
    const dropRow = this.findDropRow(e), dropIndex = this.getRowElements().indexOf(dropRow);
    this.clearOverClasses(dropRow);
    clearTimeout(this.dragOverTimeout);
    const dropTree = this._expand(dropRow);
    dragData.dropTree = this;
    dragData.childrenTree = dropTree;
    const topTree = this.findTopTree();
    topTree.fire("drop", topTree, e, dropRow, dropIndex, dropPos, dragData);
    return false;
  }
};
var tree = (config) => createComponent(new Tree(), config);
export {
  AbstractDataSource,
  ArrayField,
  ArrayUtil,
  AutocompleteChips,
  AutocompleteField,
  Avatar,
  BrowserStore,
  BufferedFunction,
  Button,
  CardContainer,
  CardMenu,
  CheckboxColumn,
  CheckboxField,
  CheckboxGroup,
  CheckboxSelectColumn,
  ChipsField,
  Collection,
  ColorField,
  ColorMenu,
  ColorPicker,
  ComboBox,
  ComboBoxDefaultRenderer,
  CommitErrorType,
  Component,
  ContainerField,
  DataSourceForm,
  DataSourceStore,
  DateColumn,
  DateField,
  DateInterval,
  DatePicker,
  DateTime,
  DateTimeColumn,
  DescriptionList,
  DisplayField,
  DraggableComponent,
  DurationField,
  E,
  Field,
  Fieldset,
  Form,
  Format,
  FunctionUtil,
  HiddenField,
  HtmlField,
  InputField,
  List,
  ListField,
  ListPicker,
  MapField,
  Mask,
  Menu,
  MultiSelectToolbar,
  Notifier,
  NumberField,
  ObjectUtil,
  Observable,
  Paginator,
  QuoteStripper,
  REM_UNIT_SIZE,
  RadioField,
  RangeField,
  Recurrence,
  RecurrenceField,
  RecurrencePicker,
  RestDataSource,
  Router,
  RowSelect,
  SearchButton,
  SelectField,
  Splitter,
  State,
  Store,
  Table,
  TableColumn,
  TextAreaField,
  TextField,
  TimeField,
  Toolbar,
  Translate,
  Tree,
  TreeRowRenderer,
  Window2 as Window,
  a,
  arrayfield,
  autocomplete,
  autocompletechips,
  avatar,
  browser,
  browserStoreConnection,
  btn,
  cardmenu,
  cards,
  checkbox,
  checkboxcolumn,
  checkboxgroup,
  checkboxselectcolumn,
  chips,
  code,
  colorfield,
  colormenu,
  colorpicker,
  column,
  combobox,
  comp,
  containerfield,
  cookies,
  createComponent,
  datasourceform,
  datasourcestore,
  datecolumn,
  datefield,
  datepicker,
  datetimecolumn,
  displaydatefield,
  displayfield,
  dl,
  draggable,
  durationfield,
  fieldset,
  form,
  h1,
  h2,
  h3,
  h4,
  hiddenfield,
  hr,
  htmlfield,
  img,
  list,
  listfield,
  listpicker,
  mapfield,
  mask,
  menu,
  menucolumn,
  mstbar,
  numberfield,
  p,
  paginator,
  progress,
  radio,
  rangefield,
  recurrencefield,
  root,
  router,
  rowselect,
  searchbtn,
  section,
  select,
  small,
  splitter,
  store,
  t,
  table,
  tbar,
  tbarItems,
  textarea,
  textfield,
  timefield,
  translate,
  tree,
  win
};
/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Michael de Hart <mdhart@intermesh.nl>
 */
/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Joachim van de Haterd <jhaterd@intermesh.nl>
 */
//# sourceMappingURL=index.js.map
