<?php

namespace GO\Savemailas;

use Faker\Generator;
use GO;
use GO\Base\Db\ActiveRecord;
use GO\Base\Mail\EmailRecipients;
use go\core\acl\model\AclEntity;
use go\core\acl\model\AclItemEntity;
use go\core\model\Acl;
use go\core\model\Acl as GoAcl;
use go\core\model\Link;
use go\core\model\Search;
use go\core\orm\EntityType;
use go\core\orm\Query;
use go\modules\community\addressbook\model\Contact;
use go\modules\community\addressbook\model\Settings;
use GO\Professional\Module;
use GO\Savemailas\Model\LinkedEmail;


class SavemailasModule extends Module{
	
	public function depends()
	{
		return array('email');
	}
	
	public function autoInstall() {
		return true;
	}

	public function defineListeners() {
		Link::on(Link::EVENT_BEFORE_DELETE, static::class, 'onLinkDelete');
		AclEntity::on(AclItemEntity::EVENT_ACL_CHANGED, static::class, 'onAclChanged');
	}

	public static function initListeners() {
		GO\Email\Model\Account::model()->addListener('savedsentitem', self::class, 'onSentItemSaved');
	}

	static function onSentItemSaved(GO\Email\Model\Account $account, $savedMessage, $params = []) {

		$tags = self::findAutoLinkTags($savedMessage->getBody(), $account->id);

		if(empty($params)) {
			$params = ['subject' => (string)$savedMessage->getSubject()];
			foreach ([
							'to' => $savedMessage->getTo(),
							'cc' => $savedMessage->getCc(),
							'bcc' => $savedMessage->getBcc(),
							'from' => $savedMessage->getFrom()
						] as $k => $v) {
				if(is_array($v)) {
					$params[$k] = new EmailRecipients();
					foreach ($v as $email => $name) {
						$params[$k]->addRecipient($email, $name);
					}
					$params[$k] = (string)$params[$k];
				}
			}
		}
		self::_link($params, $savedMessage, $tags);
	}


	private static function _link(array $params, \GO\Base\Mail\Message $message, $tags=array())
	{
		$settings = Settings::get();
		$autoLinkContacts = \go\core\model\Module::isInstalled('community','addressbook') &&  in_array($settings->autoLink,['on','incl','excl']);


		if (!empty($params['link']) || $autoLinkContacts || count($tags)) {
			$path = 'email/' . date('mY') . '/sent_' .\GO::user()->id.'-'. uniqid(time()) . '.eml';

			$file = new \GO\Base\Fs\File(GO::config()->file_storage_path . $path);
			$file->parent()->create();

			$fbs = new \Swift_ByteStream_FileByteStream($file->path(), true);
			$message->toByteStream($fbs);

			if (!$file->exists()) {
				throw new \Exception("Failed to save email to file!");
			}

			$attributes = array();

			if(isset($params['from'])) {
				$attributes['from'] = $params['from'];
			} else {
				$alias = \GO\Email\Model\Alias::model()->findByPk($params['alias_id']);
				$attributes['from'] = (string)\GO\Base\Mail\EmailRecipients::createSingle($alias->email, $alias->name);
			}
			if (isset($params['to'])) {
				$attributes['to'] = $params['to'];
			}

			if (!empty($params['cc'])) {
				$attributes['cc'] = $params['cc'];
			}
			if (!empty($params['bcc'])) {
				$attributes['bcc'] = $params['bcc'];
			}
			$attributes['subject'] = !empty($params['subject']) ? $params['subject'] : GO::t("No subject", "email");
			$attributes['path'] = $path;

			$date = $message->getDate();

			$attributes['time'] = $date ? $date->format('U') : time();
			$attributes['uid']= $message->getId();

			$linkedModels = new \go\core\util\ArrayObject();

			if (!empty($link)) {
				//add link sent by composer as a tag to unify code
				$linkProps = explode(':', $params['link']);
				$tags = array_unshift($tags, ['model' => $linkProps[0], 'model_id' => $linkProps[1]]);
			}

			//process tags in the message body
			while($tag = array_shift($tags)){
				$linkModel = \GO\Savemailas\SavemailasModule::getLinkModel($tag['model'], $tag['model_id']);
				if($linkModel && $linkedModels->findKeyBy(function($item) use ($linkModel) { return $item->equals($linkModel); } ) === false){
					$attributes['acl_id']=$linkModel->findAclId();
					$linkedEmail = \GO\Savemailas\Model\LinkedEmail::model()->findSingleByAttributes(array(
						'uid'=>$attributes['uid'],
						'acl_id'=>$attributes['acl_id']));

					if(!$linkedEmail){
						$linkedEmail = new \GO\Savemailas\Model\LinkedEmail();
						$linkedEmail->setAttributes($attributes);
						$linkedEmail->save(true);
					}
					$linkedEmail->link($linkModel);

					$linkedModels[] = $linkModel;
				}
			}


			if ($autoLinkContacts) {
				$bookIds = $settings->getAutoLinkAddressBookIds();
				$to = new \GO\Base\Mail\EmailRecipients($params['to'].",".$params['bcc']);
				$to = $to->getAddresses();

				foreach ($to as $email=>$name) {
					$contacts = Contact::findByEmail($email, ['id', 'addressBookId', 'isOrganization'])->filter(['permissionLevel' => GoAcl::LEVEL_WRITE])->all();
					foreach($contacts as $contact) {
						/** @var Contact $contact */
						if(!$contact->isOrganization) {
							foreach($contact->findOrganizations(['id', 'addressBookId', 'name'])->filter(['permissionLevel' => GoAcl::LEVEL_WRITE]) as $o) {
								$contacts[] = $o;
							}
						}
					}

					foreach($contacts as $contact){
						// autoLink == 'on' always continue
						// autoLink == 'off' we never get here
						if(($settings->autoLink == 'incl' && !in_array($contact->addressBookId, $bookIds)) ||
							($settings->autoLink == 'excl' && in_array($contact->addressBookId, $bookIds)))
							continue; // skip linking


						if($contact && $linkedModels->findKeyBy(function($item) use ($contact) { return $item->equals($contact); } ) === false){
							$attributes['acl_id']= $contact->findAclId();
							$linkedEmail = \GO\Savemailas\Model\LinkedEmail::model()->findSingleByAttributes(array(
								'uid'=>$attributes['uid'],
								'acl_id'=>$attributes['acl_id']));

							if(!$linkedEmail){
								$linkedEmail = new \GO\Savemailas\Model\LinkedEmail();
								$linkedEmail->setAttributes($attributes);
								$linkedEmail->save(true);
							}

							$linkedEmail->link($contact);
						}
					}
				}
			}
		}
	}

	public static function findAutoLinkTags($body, $account_id=0)
	{
		preg_match_all('/\[link:([^]]+)\]/',$body, $matches, PREG_SET_ORDER);

		$tags = array();
		$unique=array();
		while($match=array_shift($matches)){

			$match[1] = strip_tags($match[1]);
			$match[1] = preg_replace('/\s+/', '',$match[1]);
			$match[1] = preg_replace('/&.+;/', '',$match[1]);

			//make sure we don't parse the same tag twice.
			if(!in_array($match[1], $unique)){
				$props = explode(',',base64_decode($match[1]));
				if($props[0]==$_SERVER['SERVER_NAME'] && count($props) == 4){
					$tag=array();

					if(!$account_id || $account_id==$props[1]){
						$tag['account_id'] = $props[1];
						$tag['model'] = $props[2];
						$tag['model_id'] = $props[3];

						$tags[]=$tag;
					}
				}

				$unique[]=$match[1];
			}

		}
		return $tags;
	}



	/**
	 * Fixes link acl's when for example a contact is moved to another address book
	 *
	 * @param AclItemEntity $entity
	 * @throws GO\Base\Exception\AccessDenied
	 */
	public static function onAclChanged(AclEntity $entity) {
		if($entity instanceof Search) {
			return;
		}
		$stmt = LinkedEmail::model()->findLinks($entity);

		if($stmt->rowCount()) {
			$aclId = $entity->findAclId();
			while ($linkedEmail = $stmt->fetch()) {
				$linkedEmail->acl_id = $aclId;
				$linkedEmail->save();
			}
		}
	}

	/**
	 * Because we've implemented the getter method "getOrganizationIds" the contact
	 * modSeq must be incremented when a link between two contacts is deleted or
	 * created.
	 *
	 * @param Link $link
	 */
	public static function onLinkDelete(Query $links) {

		$emailTypeId = LinkedEmail::entityType()->getId();
		$query = clone $links;

		$query->andWhere('(toEntityTypeId = :e1 OR fromEntityTypeId = :e2)')->bind([':e1'=> $emailTypeId, ':e2'=> $emailTypeId]);

		$emailLinks = Link::find()->mergeWith($query);

		foreach($emailLinks as $link) {

			$emails = [];
			if($link->getToEntity() == "LinkedEmail") {
				$emails[] = $link->toId;
			}

			if($link->getFromEntity() == "LinkedEmail") {
				$emails[] = $link->fromId;
			}

			foreach($emails as $id) {
				$linkedEmail = LinkedEmail::model()->findByPk($id);
				if($linkedEmail->countLinks() == 1) {
					$linkedEmail->delete(true);
				}
			}
		}

	}


	/**
	 * 
	 * @param string $entity
	 * @return EntityType
	 */
	public static function getLinkModel($modelName, $modelId) {
		
		//make it backwards compatible with old classnames. Strip off the namespace.
		
		$parts = explode("\\", $modelName);
		$entity = array_pop($parts);
		
		$entityType = EntityType::findByName($entity);
		
		if(!$entityType) {
			return false;
		}

		//Readable items can be linked!

		$cls = $entityType->getClassName();
		if(is_a($cls, ActiveRecord::class, true)) {
		
			$model = GO::getModel($cls)->findByPk($modelId, false, true);
			if(!$model) {//} || !$model->checkPermissionLevel(Acl2::WRITE_PERMISSION)) {
				return false;
			}	
		} else
		{
			//must be a go\core\orm\Entity (new)			
			$model = $cls::findById($modelId);
			if(!$model){//|| !$model->hasPermissionLevel(Acl::LEVEL_WRITE)) {
				return false;
			}
			
		}
		
		return $model;
	}

	public function demoMail(Generator $faker, $entity) {
		//link some demo mails
		$mimeFile = new \GO\Base\Fs\File(\GO::modules()->savemailas->path.'install/demo.eml');
		$linkedEmail = LinkedEmail::model()->createFromMimeFile($mimeFile, $entity);

		$mimeFile = new \GO\Base\Fs\File(\GO::modules()->savemailas->path.'install/demo2.eml');
		$linkedEmail = LinkedEmail::model()->createFromMimeFile($mimeFile, $entity);

	}

	public function demo(Generator $faker)
	{
		//link some demo mails
		$mimeFile = new \GO\Base\Fs\File(\GO::modules()->savemailas->path.'install/demo.eml');

		$contact = \go\modules\community\addressbook\Module::get()->demoContact($faker);

		$linkedEmail = LinkedEmail::model()->createFromMimeFile($mimeFile, $contact);

		Link::demo($faker, $linkedEmail);

		$mimeFile = new \GO\Base\Fs\File(\GO::modules()->savemailas->path.'install/demo2.eml');
		$linkedEmail = LinkedEmail::model()->createFromMimeFile($mimeFile, $contact);
		Link::demo($faker, $linkedEmail);

	}
}
