/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { createComponent } from "../Component.js";
import { t } from "../../Translate";
import { InputField } from "./InputField";
/**
 * NumberField component
 */
export class NumberField extends InputField {
    constructor() {
        super();
        this.baseCls = 'goui-form-field number';
        /**
         * Multiply value with this number on set and divide on get value
         */
        this.multiplier = 1;
        this.type = "number";
    }
    validate() {
        super.validate();
        const v = this.value;
        if (v && isNaN(v)) {
            this.setInvalid("Incorrect number format");
        }
        if (this.max !== undefined && v && v > this.max) {
            this.setInvalid(t("Number is bigger than the maximum of {max}.").replace("{max}", this.max.toLocaleString()));
        }
        if (this.min !== undefined && (!v || v < this.min)) {
            this.setInvalid(t("Number is bigger than the maximum of {max}.").replace("{max}", this.min.toLocaleString()));
        }
    }
    internalSetValue(v) {
        if (this.isEmptyNumber(v)) {
            v = undefined;
        }
        else if (isNaN(v)) {
            console.error("Invalid number given for field " + this.name, v);
            v = undefined;
        }
        else {
            v = +(v * this.multiplier).toFixed(this.decimals);
        }
        super.internalSetValue(v);
    }
    isEmptyNumber(v) {
        return (v === undefined || v === null || v === "");
    }
    internalGetValue() {
        let v = this.input.value == "" ? undefined : parseInt(this.input.value);
        if ((v === undefined || this.isEmptyNumber(v) || isNaN(v))) {
            return undefined;
        }
        return +(v / this.multiplier).toFixed(this.decimals);
    }
    /**
     * The step attribute is a number that specifies the granularity that the value must adhere to or the keyword any.
     *
     * @link https://developer.mozilla.org/en-US/docs/Web/HTML/Attributes/step
     * @param step
     */
    set step(step) {
        if (step === undefined) {
            this.input.removeAttribute('step');
        }
        else {
            this.input.setAttribute('step', step.toString());
        }
    }
    get step() {
        var _a;
        return parseFloat((_a = this.input.getAttribute('step')) !== null && _a !== void 0 ? _a : "0");
    }
    /**
     * Set the number of decimals. It uses the step attribute to accomplish this.
     *
     * @param decimals
     */
    set decimals(decimals) {
        if (!decimals) {
            this.input.removeAttribute('step');
        }
        else {
            this.input.setAttribute('step', '0.'.padEnd(decimals + 1, "0") + "1");
        }
    }
    get decimals() {
        const step = this.input.attr('step');
        if (!step) {
            return undefined;
        }
        return step.length - 2;
    }
    /**
     * The minimum number allowed
     *
     * @param min
     */
    set min(min) {
        if (min === undefined) {
            this.input.removeAttribute("min");
        }
        else {
            this.input.setAttribute('min', min.toString());
        }
    }
    get min() {
        const min = this.input.getAttribute('min');
        if (min === null) {
            return undefined;
        }
        return parseFloat(min);
    }
    /**
     * The maximum number allowed
     *
     * @param max
     */
    set max(max) {
        if (max === undefined) {
            this.input.removeAttribute("max");
        }
        else {
            this.input.setAttribute('max', max.toString());
        }
    }
    get max() {
        const max = this.input.getAttribute('max');
        if (max === null) {
            return undefined;
        }
        return parseFloat(max);
    }
}
export const numberfield = (config) => createComponent(new NumberField(), config);
//# sourceMappingURL=NumberField.js.map