/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { createComponent } from "../Component.js";
import { DateTime } from "../../util";
import { InputField } from "./InputField";
/**
 * Date field
 *
 * @property min The minimum value allowed. Same format as {@link DateField.value}.
 * @property max The maximum value allowed. Same format as {@link DateField.value}.
 *
 * @see Form
 */
export class DateField extends InputField {
    constructor() {
        super();
        this.baseCls = "goui-form-field date no-floating-label";
        this.type = "date";
    }
    /**
     * Also render time input
     *
     * @param withTime
     */
    set withTime(withTime) {
        let v = this.value, newType = withTime ? "datetime-local" : "date";
        if (newType != this.input.type) {
            this.input.type = newType;
            if (!v) {
                return;
            }
            if (withTime) {
                v = this.appendTime(v);
            }
            else {
                const parts = v.split("T");
                v = parts[0];
                this.defaultTime = parts[1];
            }
            this.input.value = v;
        }
    }
    appendTime(v) {
        var _a;
        return v + "T" + ((_a = this.defaultTime) !== null && _a !== void 0 ? _a : (new DateTime()).format("H:i"));
    }
    get withTime() {
        return this.type == "datetime-local";
    }
    internalSetValue(v) {
        if (v) {
            const Tindex = v.indexOf("T");
            if (this.withTime) {
                if (Tindex == -1) {
                    v = this.appendTime(v);
                }
            }
            else {
                if (Tindex != -1) {
                    const parts = v.split("T");
                    v = parts[0];
                    this.defaultTime = parts[1];
                }
            }
        }
        super.internalSetValue(v);
    }
    outputFormat() {
        return this.withTime ? "Y-m-dTH:i" : 'Y-m-d';
    }
    /**
     * The minimum number allowed
     *
     * The value of the time input is always in 24-hour format that includes leading zeros: hh:mm
     *
     * @param min
     */
    set min(min) {
        this.input.attr('min', min);
    }
    get min() {
        return this.input.attr('min');
    }
    /**
     * The maximum number allowed
     *
     * The value of the time input is always in 24-hour format that includes leading zeros: hh:mm
     *
     * @param max
     */
    set max(max) {
        this.input.attr('max', max);
    }
    get max() {
        return this.input.attr('max');
    }
    /**
     * Get the date as DateTime object
     */
    getValueAsDateTime() {
        let v = this.value, date;
        if (!v || !(date = DateTime.createFromFormat(v, this.outputFormat()))) {
            return undefined;
        }
        return date;
    }
}
/**
 * Shorthand function to create {@see DateField}
 *
 * @param config
 */
export const datefield = (config) => createComponent(new DateField(), config);
//# sourceMappingURL=DateField.js.map