/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Michael de Hart <mdhart@intermesh.nl>
 */
import { Component, ComponentEventMap } from "./Component.js";
import { Store, StoreRecord } from "../data/Store.js";
import { RowSelect, RowSelectConfig } from "./table/RowSelect.js";
import { Config, Listener, ObservableListenerOpts } from "./Observable.js";
export type RowRenderer = (record: any, row: HTMLElement, list: any, storeIndex: number) => string | Component[] | void;
export type listStoreType<ListType> = ListType extends List<infer StoreType> ? StoreType : never;
/**
 * @inheritDoc
 */
export interface ListEventMap<Type> extends ComponentEventMap<Type> {
    /**
     * Fires when the user scrolled to the bottom
     *
     * @param list
     */
    scrolleddown: (list: Type) => void;
    /**
     * Fires when the user sorts the list by a property
     *
     * @param list
     * @param property
     */
    sort: (list: Type, property: string) => void;
    /**
     * Fires when the user sorts the list by drag and drop
     *
     * @param list
     * @param dataIndex
     */
    sortchange: (list: Type, record: any, dropIndex: number, oldIndex: number) => void;
    /**
     * Fires when a row is mousedowned
     *
     * @param list
     * @param storeIndex
     * @param ev
     */
    rowmousedown: (list: Type, storeIndex: number, row: HTMLElement, ev: MouseEvent) => void;
    /**
     * Fires when a row is clicked
     *
     * @param list
     * @param storeIndex
     * @param ev
     */
    rowclick: (list: Type, storeIndex: number, row: HTMLElement, ev: MouseEvent) => void;
    /**
     * Fires when a row is double clicked
     *
     * @param list
     * @param storeIndex
     * @param ev
     */
    rowdblclick: (list: Type, storeIndex: number, row: HTMLElement, ev: MouseEvent) => void;
    /**
     * Fires when the delete key is pressed
     *
     * @example
     * ```
     * delete: async (list) => {
     * 	const ids = list.rowSelection!.selected.map(index => list.store.get(index)!.id);
     * 	await jmapds("Foo").confirmDestroy(ids);
     * }
     * ```
     *
     * @param list
     */
    delete: (list: Type) => void;
    /**
     * Fires when a row is right clicked
     *
     * @param list
     * @param storeIndex
     * @param ev
     */
    rowcontextmenu: (list: Type, storeIndex: number, row: HTMLElement, ev: MouseEvent) => void;
    /**
     * Fires when records are rendered into rows.
     *
     * @param list
     * @param records
     */
    renderrows: (list: Type, records: any[]) => void;
    /**
     * Fires when a row is clicked or navigated with arrows
     *
     * @param list
     * @param storeIndex
     * @param record
     */
    navigate: (list: Type, storeIndex: number) => void;
    /**
     * Fires when something was dropped
     *
     * @param tree The tree that contains the node that is dropped on
     * @param e The draag event
     * @param dropRow The row element that is dropped on
     * @param dropIndex Store index
     * @param position
     * @param dragData The arbitrary drag data that is set
     */
    drop: (list: Type, e: DragEvent, dropRow: HTMLElement, dropIndex: number, position: DROP_POSITION, dragData: any) => void;
    dropallowed: (list: Type, e: DragEvent, dropRow: HTMLElement, dragData: any) => void;
}
export type DROP_POSITION = "before" | "after" | "on";
export interface List<StoreType extends Store = Store> extends Component {
    on<K extends keyof ListEventMap<this>, L extends Listener>(eventName: K, listener: Partial<ListEventMap<this>>[K], options?: ObservableListenerOpts): L;
    un<K extends keyof ListEventMap<this>>(eventName: K, listener: Partial<ListEventMap<this>>[K]): boolean;
    fire<K extends keyof ListEventMap<this>>(eventName: K, ...args: Parameters<ListEventMap<any>[K]>): boolean;
}
/**
 * List component
 *
 * Create a list with a custom item renderer. Also capable of selecting rows.
 */
export declare class List<StoreType extends Store = Store> extends Component {
    readonly store: StoreType;
    readonly renderer: RowRenderer;
    /**
     * Shown when the list is empty.
     */
    emptyStateHtml: string;
    protected emptyStateTag: keyof HTMLElementTagNameMap;
    private emptyEl?;
    private rowSelect?;
    /**
     * Allow items to be dragged
     */
    draggable: boolean;
    /**
     * Allow to drop between items
     */
    dropBetween: boolean;
    /**
     * Allow to drop on items
     */
    dropOn: boolean;
    protected itemTag: keyof HTMLElementTagNameMap;
    /**
     * Row selection object
     * @param rowSelectionConfig
     */
    set rowSelectionConfig(rowSelectionConfig: boolean | Partial<RowSelectConfig>);
    set sortable(sortable: boolean);
    constructor(store: StoreType, renderer: RowRenderer, tagName?: keyof HTMLElementTagNameMap);
    get rowSelection(): RowSelect | undefined;
    get el(): HTMLElement;
    protected internalRender(): HTMLElement;
    protected onKeyDown(e: KeyboardEvent): void;
    protected initStore(): void;
    protected onRecordRemove(collection: StoreType, item: StoreRecord, index: number): void;
    protected onRecordAdd(collection: StoreType, item: StoreRecord, index: number): void;
    protected getRowElements(): HTMLElement[];
    private initNavigateEvent;
    protected renderEmptyState(): void;
    protected renderBody(): void;
    focusRow(index: number): void;
    protected renderRows(records: any[]): void;
    protected onRowAppend(row: HTMLElement, record: any, index: number): void;
    protected renderGroup(record: any): HTMLElement | DocumentFragment;
    protected renderRow(record: any, storeIndex: number): HTMLElement;
    private onMouseEvent;
    private findRowByEvent;
    protected bindDropEvents(row: HTMLElement): void;
    protected onNodeDragStart(e: DragEvent): void;
    protected onNodeDragEnd(e: DragEvent): void;
    protected onNodeDragEnter(e: DragEvent): void;
    protected onNodeDragEnterAllowed(e: DragEvent, dropRow: HTMLElement): void;
    protected onNodeDragLeave(e: DragEvent): void;
    protected onNodeDragLeaveAllowed(e: DragEvent, dropRow: HTMLElement): void;
    protected findDropRow(e: DragEvent): HTMLElement;
    protected clearOverClasses(dropRow: HTMLElement): void;
    protected onNodeDragOver(e: DragEvent): void;
    protected dropAllowed(e: DragEvent, dropRow: HTMLElement): boolean;
    protected getDropPosition(e: DragEvent): DROP_POSITION | undefined;
    protected onNodeDrop(e: DragEvent): void;
}
export type ListConfig<StoreType extends Store> = Omit<Config<List<StoreType>, ListEventMap<List<StoreType>>, "store" | "renderer">, "rowSelection">;
/**
 * Shorthand function to create {@see Table}
 *
 * @param config
 */
export declare const list: <StoreType extends Store>(config: ListConfig<StoreType>) => List<StoreType>;
//# sourceMappingURL=List.d.ts.map