<?php
/**
 * @copyright (c) 2019, Intermesh BV http://www.intermesh.nl
 * @author Merijn Schering <mschering@intermesh.nl>
 * @license http://www.gnu.org/licenses/agpl-3.0.html AGPLv3
 */
namespace go\modules\business\business;
							
use go\core;
use go\core\db\Criteria;
use go\core\model;
use go\core\model\User;
use go\core\orm\Filters;
use go\core\orm\Mapping;
use go\core\orm\Property;
use go\core\orm\Query;
use go\core\util\DateTime;
use go\modules\business\business\model\Activity;
use go\modules\business\business\model\Business;
use go\modules\business\business\model\Category;
use go\modules\business\business\model\Employee;
use go\modules\business\business\model\VatRate;
use go\modules\business\license\model\License;


class Module extends core\Module {
	/**
	 * The development status of this module
	 * @return string
	 */
	public function getStatus() : string{
		return self::STATUS_STABLE;
	}

	protected function rights(): array
	{
		return ['mayManage', 'mayManageEmployees'];
	}

	public function defineListeners() {
		User::on(Property::EVENT_MAPPING, static::class, 'onMap');
		model\UserDisplay::on(Property::EVENT_MAPPING, static::class, 'onMap');
		User::on(User::EVENT_FILTER, static::class, 'onUserFilter');
		model\UserDisplay::on(User::EVENT_FILTER, static::class, 'onUserFilter');
		User::on(User::EVENT_SORT, static::class, 'onUserSort');
		model\UserDisplay::on(User::EVENT_SORT, static::class, 'onUserSort');
	}

	public static function onMap(Mapping $mapping) {
		$mapping->addHasOne('employee', Employee::class, ['id' => 'id'], false);
	}

	public static function onUserFilter(Filters $filters) {
		$filters->add('isEmployee', function(Criteria $criteria, $value, Query $query) {
			if($value) {
				// left join to make sure it exists, maybe check if businessId is not empty too?
				$query->join('business_employee', 'e', 'e.id = u.id');
				$criteria->andWhere(
					(new Criteria())
						->andWhere('e.quitAt', 'IS', NULL)
						->orWhere('e.quitAt','>',new DateTime())
				);
			}
		});
		$filters->add('employmentDate', function (Criteria $criteria, $value, Query $query) {
			if($value) {
				$dt = new DateTime($value);
				$query->join('business_employee', 'bre', 'bre.id=u.id');
				$criteria->andWhere(
					( new Criteria())
						->andWhere('bre.quitAt', 'IS', NULL)
						->orWhere('bre.quitAt', '>=', $dt)
				);
			}

		});
	}

	public static function onUserSort(Query $query, core\util\ArrayObject $sort) {
		if(isset($sort['employee.hourlyCosts']) ||
			isset($sort['employee.hourlyRevenue']) ||
			isset($sort['employee.timeClosedUntil']) ||
			isset($sort['employee.quitAt'])
		) {
			$query->join('business_employee', 'employee', 'employee.id = ' . $query->getTableAlias() . '.id', 'LEFT');
		}
		if(isset($sort['employee.hourlyCosts'])) {
			$query->orderBy(['employee.hourlyCosts' => $sort['employee.hourlyCosts']], true);
			unset($sort['employee.hourlyCosts']);
		}
		if(isset($sort['employee.hourlyRevenue'])) {
			$query->orderBy(['employee.hourlyRevenue' => $sort['employee.hourlyRevenue']], true);
			unset($sort['employee.hourlyRevenue']);
		}
		if(isset($sort['employee.timeClosedUntil'])) {
			$query->orderBy(['employee.timeClosedUntil' => $sort['employee.timeClosedUntil']], true);
			unset($sort['employee.timeClosedUntil']);
		}
		if(isset($sort['employee.quitAt'])) {
			$query->orderBy(['employee.quitAt' => $sort['employee.quitAt']], true);
			unset($sort['employee.quitAt']);
		}
	}

	protected function afterInstall(model\Module $model): bool
	{
		// create empty business
		$business= new Business();
		$business->name = go()->t('Default business');

		$business->categories = [
			(new Category($business))->setValues(['name' => go()->t("Hours"), "isHours" => true]),
			(new Category($business))->setValues(['name' => go()->t("Services")]),
			(new Category($business))->setValues(['name' => go()->t("Materials")]),
		];

		if(!$business->save()) {
			throw new core\orm\exception\SaveException($business);
		}

		$holidays = new Activity();
		$holidays->code = 1;
		$holidays->budgetable = true;
		$holidays->name = go()->t("Holidays");
		$holidays->type = Activity::TYPE_LEAVE;
		$holidays->units = 8;
		$holidays->billable = false;
		if(!$holidays->save()) {
			throw new core\orm\exception\SaveException($holidays);
		}

		$sick = new Activity();
		$sick->code = 2;
		$sick->budgetable = false;
		$sick->name = go()->t("Sick");
		$sick->type = Activity::TYPE_ABSENCE;
		$sick->units = 8;
		$sick->billable = false;
		if(!$sick->save()) {
			throw new core\orm\exception\SaveException($sick);
		}






		return true;
	}


	public function getAuthor(): string
	{
		return "Intermesh BV <info@intermesh.nl>";
	}

	public function requiredLicense(): ?string
	{
		return 'groupoffice-pro';
	}

	public function isLicensed(): bool
	{
		if(!go()->getEnvironment()->hasIoncube()) {
			return false;
		}

		return License::has('groupoffice-pro') || License::has('billing');
	}

	public function autoInstall(): bool
	{
		return true;
	}

	public function getDependencies(): array
	{
		return ['community/addressbook'];
	}
}