<?php defined('FLATBOARD') or die('Flatboard Community.');
/**
 * Identicon
 *
 * @author 		Frédéric K.
 * @copyright	(c) 2020-2025
 * @license		http://opensource.org/licenses/MIT
 * @package		FlatBoard
 * @version		3.6.0
 * @update		2025-11-29
 */	

define('PATH_IDENTICON_AVATAR', UPLOADS_DIR . 'avatars' . DS);
define('PATH_GEN_AVATAR', 	 	PATH_IDENTICON_AVATAR . 'generates' . DS);
define('HTML_AVATAR', 		 	HTML_BASEPATH . DS . 'uploads' . DS . 'avatars' . DS);
define('HTML_GEN_AVATAR', 	 	HTML_AVATAR . 'generates' . DS);  
define('HTML_PLUGIN_AVATAR', 	HTML_BASEPATH . DS . 'plugin' . DS . 'identicon' . DS);  
define('IDENTICON_ONLINE', 		TRUE);                 

/**
 * Pré-installe les paramètres par défaut.
 */
function identicon_install()
{
    $plugin = 'identicon';
    
    // Vérifier si le plugin est déjà installé
    if (flatDB::isValidEntry('plugin', $plugin)) {
        return; // Si le plugin est déjà installé, ne rien faire
    }
	
    // Création des dossiers pour les avatars
    $folders = [PATH_IDENTICON_AVATAR, PATH_GEN_AVATAR];
    foreach ($folders as $folder) {    
        if (!is_dir($folder)) {
            mkdir($folder, 0777, true); // Créer le dossier avec les permissions appropriées
        }	
        // Créer un fichier index.html s'il n'existe pas
        if (!file_exists($folder . 'index.html')) {
            $f = fopen($folder . 'index.html', 'a+');
            fwrite($f, ''); // Écrire un contenu vide
            fclose($f);	
        }
    }

    // Initialiser les données du plugin
    $data = [
        $plugin . 'state' => true, 
        'taille' => '50', 
        'forme' => 'rounded-circle', 
        'dots' => true,
        'show_dot' => true,
        'show_badge' => true
    ];
    
    // Sauvegarder les données du plugin
    flatDB::saveEntry('plugin', $plugin, $data);         
}

/**
 * Vérifie si le plugin est installé.
 *
 * @param string $plugin Le nom du plugin à vérifier
 * @return bool Vrai si le plugin est installé, faux sinon
 */
function pluginIsHere($plugin) 
{
    return flatDB::isValidEntry('plugin', $plugin);
}

/**
 * Configuration de l'admin pour le plugin Identicon
 */
function identicon_config()
{    
    global $lang, $token; 
    $plugin = 'identicon';
    $out = '';
           
    // Traitement des données POST
    if (!empty($_POST) && CSRF::check($token)) {
        $data[$plugin . 'state'] = Util::isPOST('state') ? $_POST['state'] : '';  
        $data['taille'] = HTMLForm::clean($_POST['taille']);    
        $data['forme'] = HTMLForm::clean($_POST['forme']);
        $data['dots'] = Util::isPOST('dots') ? $_POST['dots'] : '';
        $data['show_dot'] = Util::isPOST('show_dot') ? $_POST['show_dot'] : '';
        $data['show_badge'] = Util::isPOST('show_badge') ? $_POST['show_badge'] : '';
               
        // Sauvegarder les données du plugin
        flatDB::saveEntry('plugin', $plugin, $data);
        
        // Afficher un message de succès
        $out .= Plugin::redirectMsg(
            $lang['data_save'],
            'config.php' . DS . 'plugin' . DS . $plugin,
            $lang['plugin'] . '&nbsp;<b>' . $lang[$plugin . 'name'] . '</b>'
        );   
    } else {
        // Lire les données du plugin si elles existent
        if (flatDB::isValidEntry('plugin', $plugin)) {
            $data = flatDB::readEntry('plugin', $plugin);
        }

        // Générer le formulaire de configuration
        $out .= HTMLForm::form('config.php' . DS . 'plugin' . DS . $plugin, '
            <div class="row"> 
                <div class="col">' .
                    HTMLForm::checkBox('state', $data[$plugin . 'state']) . '
                </div>				              	
                <div class="col">' .
                    HTMLForm::checkBox('dots', $data['dots']) . '
                </div>
            </div>
            <div class="row"> 
                <div class="col">' .
                    HTMLForm::checkBox('show_dot', isset($data['show_dot']) ? $data['show_dot'] : true) . '
                </div>				              	
                <div class="col">' .
                    HTMLForm::checkBox('show_badge', isset($data['show_badge']) ? $data['show_badge'] : true) . '
                </div>
            </div>						               	
            <div class="row">				
                <div class="col">' .
                    HTMLForm::text('taille', isset($data) ? $data['taille'] : '', 'number', '', '', 'pixel') .
                '</div>
                <div class="col">' .
                    HTMLForm::select('forme', [
                        'rounded-circle' => $lang['rounded-circle'], 
                        'rounded' => $lang['rounded'], 
                        'img-thumbnail' => $lang['img-thumbnail']
                    ], $data['forme']) . '
                </div>				
            </div>' .       
            HTMLForm::simple_submit()
        );
    }

    return $out; // Retourner le contenu généré
}

/**
 * Ajoute un peu de CSS pour le plugin Identicon
 * Optimisé avec cache statique
 */
function identicon_head()
{
    $plugin = 'identicon';
    $logFile = __DIR__ . '/../../identicon_debug.log';
    
    @file_put_contents($logFile, date('Y-m-d H:i:s') . ' - identicon_head() called' . PHP_EOL, FILE_APPEND);
    
    // Cache statique pour éviter les lectures répétées
    static $pluginDataCache = null;
    if ($pluginDataCache === null) {
        $pluginDataCache = flatDB::readEntry('plugin', $plugin);
        @file_put_contents($logFile, date('Y-m-d H:i:s') . ' - Plugin data loaded: ' . (is_array($pluginDataCache) ? 'yes' : 'no') . PHP_EOL, FILE_APPEND);
    }
    $data = $pluginDataCache;
    
    if ($data && isset($data[$plugin . 'state']) && $data[$plugin . 'state']) {
        @file_put_contents($logFile, date('Y-m-d H:i:s') . ' - Plugin is active, returning CSS' . PHP_EOL, FILE_APPEND);
        
        $showDot = isset($data['show_dot']) ? $data['show_dot'] : true;
        $showBadge = isset($data['show_badge']) ? $data['show_badge'] : true;
        
        $css = '<style>
            /* Conteneur pour positionner le statut */
            figure {
                position: relative;
                display: inline-block;
            }
            /* Conteneur interne pour avatar et badges */
            figure .avatar-wrapper {
                position: relative;
                display: inline-block;
                line-height: 0;
            }';
        
        // CSS pour le dot de statut (si activé)
        if ($showDot) {
            $css .= '
            .avatar-status {
                border-radius: 50%; 
                width: 15px; 
                height: 15px; 
                position: absolute; 
                bottom: 2px;
                right: 2px;
                border: 2px solid #fff;
                z-index: 10;
                box-sizing: border-box;
            }
            .avatar-status.green {
                background: rgba(51, 217, 178, 1);
                box-shadow: 0 0 0 0 rgba(51, 217, 178, 1);
                animation: pulse-green 2s infinite;
            }
            @keyframes pulse-green {
                0% { transform: scale(0.95); box-shadow: 0 0 0 0 rgba(51, 217, 178, 0.7); }
                70% { transform: scale(1); box-shadow: 0 0 0 10px rgba(51, 217, 178, 0); }
                100% { transform: scale(0.95); box-shadow: 0 0 0 0 rgba(51, 217, 178, 0); }
            }
            .avatar-status.red {
                background: rgba(255, 82, 82, 1);
                box-shadow: 0 0 0 0 rgba(255, 82, 82, 1);
                animation: pulse-red 2s infinite;
            }
            @keyframes pulse-red {
                0% { transform: scale(0.95); box-shadow: 0 0 0 0 rgba(255, 82, 82, 0.7); }
                70% { transform: scale(1); box-shadow: 0 0 0 10px rgba(255, 82, 82, 0); }
                100% { transform: scale(0.95); box-shadow: 0 0 0 0 rgba(255, 82, 82, 0); }
            }';
        }
        
        $css .= '
            figure:hover img {
                box-shadow: #e3e3e3 0 0 0 0.5rem;
            }
            figure img {
                transition: box-shadow 300ms ease-out;
                position: relative;
            }';
        
        // CSS pour le badge de rôle (si activé)
        if ($showBadge) {
            $css .= '
            /* Badge role admin/modo */
            .avatar-role-badge {
                position: absolute;
                top: 2px;
                left: 2px;
                width: 20px;
                height: 20px;
                border-radius: 50%;
                display: flex;
                align-items: center;
                justify-content: center;
                font-size: 10px;
                z-index: 15;
                border: 2px solid #fff;
                box-shadow: 0 2px 4px rgba(0,0,0,0.2);
            }
            .avatar-role-badge i {
                color: #fff;
            }
            .avatar-role-admin {
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            }
            .avatar-role-worker {
                background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
            }
            .avatar-role-badge:hover {
                transform: scale(1.1);
                transition: transform 0.2s ease;
            }';
        }
        
        $css .= '
        </style>';
        @file_put_contents($logFile, date('Y-m-d H:i:s') . ' - CSS generated, length: ' . strlen($css) . PHP_EOL, FILE_APPEND);
        return $css;
    }
    
    @file_put_contents($logFile, date('Y-m-d H:i:s') . ' - Plugin is inactive, returning empty' . PHP_EOL, FILE_APPEND);
    return '';
}

/**
 * Vérifie les membres connectés via le plugin online (corrigé et optimisé)
 * Utilise les nouvelles clés de données du plugin online : online_hits et online_auth_users
 *
 * @param string $username Le nom d'utilisateur à vérifier
 * @return string Statut des utilisateurs en ligne
 */
function checkOnline($username)
{
    global $sessionTrip;
    $plugin = 'online';

    // Vérifier si le plugin "online" est installé et si IDENTICON_ONLINE est activé
    if (!IDENTICON_ONLINE || !pluginIsHere($plugin)) {
        return '';
    }

    // Cache statique pour éviter les lectures répétées
    static $onlineDataCache = null;
    static $authUsersCache = null;
    static $onlineStateCache = null;
    static $lastCacheUpdate = 0;
    
    $currentTime = time();
    
    // Mettre à jour le cache toutes les 5 secondes
    if ($onlineDataCache === null || ($currentTime - $lastCacheUpdate) > 5) {
        try {
            $data = flatDB::readEntry('plugin', $plugin);
            $onlineStateCache = isset($data[$plugin . 'state']) ? $data[$plugin . 'state'] : false;
            
            if ($onlineStateCache) {
                // Charger les utilisateurs authentifiés (ce sont les utilisateurs connectés)
                $authUsersCache = flatDB::readEntry('plugin', $plugin . '_auth_users') ?: [];
                $onlineDataCache = flatDB::readEntry('plugin', $plugin . '_hits') ?: [];
                
                // Nettoyer les entrées expirées (timeout configurable, défaut 60s)
                $timeout = isset($data['timeout']) ? max(10, min(3600, intval($data['timeout']))) : 60;
                if (is_array($authUsersCache)) {
                    foreach ($authUsersCache as $ip => $info) {
                        if (!isset($info['time']) || ($currentTime - $info['time']) > $timeout) {
                            unset($authUsersCache[$ip]);
                        }
                    }
                }
            }
            $lastCacheUpdate = $currentTime;
        } catch (Exception $e) {
            // En cas d'erreur, retourner une chaîne vide
            return '';
        }
    }
    
    if (!$onlineStateCache || empty($authUsersCache) || !is_array($authUsersCache)) {
        return '<span class="avatar-status red"></span>'; // Hors ligne par défaut
    }

    // Calculer le trip de l'utilisateur
    $trip = HTMLForm::trip($sessionTrip, $username);
    
    // Extraire le nom d'utilisateur sans le hash
    $userWithoutHash = strstr($username, '@', true) ?: $username;
    $tripWithoutHash = strstr($trip, '@', true) ?: $trip;
    
    // Vérifier si l'utilisateur est dans la liste des utilisateurs authentifiés en ligne
    $isOnline = false;
    foreach ($authUsersCache as $ip => $info) {
        if (isset($info['username'])) {
            $authUsername = $info['username'];
            // Comparer avec le username, le trip, ou les versions sans hash
            if ($authUsername === $username || 
                $authUsername === $trip || 
                $authUsername === $userWithoutHash || 
                $authUsername === $tripWithoutHash) {
                $isOnline = true;
                break;
            }
        }
    }
    
    return $isOnline 
        ? '<span class="avatar-status green"></span>' 
        : '<span class="avatar-status red"></span>';
}

/**
 * Obtient le rôle d'un utilisateur à partir de son trip/username
 * Cherche dans les topics et replies récents avec cache optimisé
 *
 * @param string $username Le nom d'utilisateur/trip
 * @return string Le rôle ('admin', 'worker', ou '')
 */
function getUserRole($username)
{
    $logFile = __DIR__ . '/../../identicon_debug.log';
    
    if (empty($username)) {
        @file_put_contents($logFile, date('Y-m-d H:i:s') . ' - getUserRole: empty username' . PHP_EOL, FILE_APPEND);
        return '';
    }
    
    static $roleCache = [];
    static $lastCacheClean = 0;
    
    // Nettoyer le cache toutes les 5 minutes
    if (time() - $lastCacheClean > 300) {
        $roleCache = [];
        $lastCacheClean = time();
    }
    
    // Cache pour éviter les recherches répétées
    if (isset($roleCache[$username])) {
        return $roleCache[$username];
    }
    
    $role = '';
    
    try {
        @file_put_contents($logFile, date('Y-m-d H:i:s') . ' - getUserRole: searching for ' . $username . PHP_EOL, FILE_APPEND);
        
        // Chercher dans les topics récents (limite à 30 pour performance)
        $topics = flatDB::listEntry('topic');
        if (is_array($topics) && count($topics) > 0) {
            $topics = array_slice($topics, -30); // Derniers 30 topics
            foreach (array_reverse($topics) as $topic) { // Commencer par les plus récents
                if (empty($topic)) {
                    continue;
                }
                $topicEntry = flatDB::readEntry('topic', $topic);
                if (is_array($topicEntry) && isset($topicEntry['trip']) && $topicEntry['trip'] === $username && !empty($topicEntry['role'])) {
                    $role = $topicEntry['role'];
                    if ($role === 'admin') {
                        break; // Admin trouvé, arrêter
                    }
                    // Continuer pour worker si admin pas encore trouvé
                }
            }
        }
        
        // Si pas trouvé dans topics, chercher dans replies récents
        if (empty($role) || $role !== 'admin') {
            $replies = flatDB::listEntry('reply');
            if (is_array($replies) && count($replies) > 0) {
                $replies = array_slice($replies, -50); // Derniers 50 replies
                foreach (array_reverse($replies) as $reply) { // Commencer par les plus récents
                    if (empty($reply)) {
                        continue;
                    }
                    $replyEntry = flatDB::readEntry('reply', $reply);
                    if (is_array($replyEntry) && isset($replyEntry['trip']) && $replyEntry['trip'] === $username && !empty($replyEntry['role'])) {
                        $foundRole = $replyEntry['role'];
                        if ($foundRole === 'admin') {
                            $role = 'admin';
                            break; // Admin trouvé, arrêter
                        } elseif ($foundRole === 'worker' && $role !== 'admin') {
                            $role = 'worker';
                        }
                    }
                }
            }
        }
    } catch (Exception $e) {
        // En cas d'erreur, retourner une chaîne vide
        $errorMsg = date('Y-m-d H:i:s') . ' - getUserRole ERROR: ' . $e->getMessage() . ' in ' . $e->getFile() . ':' . $e->getLine() . PHP_EOL;
        @file_put_contents($logFile, $errorMsg, FILE_APPEND);
        $role = '';
    }
    
    @file_put_contents($logFile, date('Y-m-d H:i:s') . ' - getUserRole: found role ' . $role . ' for ' . $username . PHP_EOL, FILE_APPEND);
    
    $roleCache[$username] = $role;
    return $role;
}

/**
 * Affiche l'avatar de l'utilisateur (optimisé avec cache)
 * Accepte soit une string (trip) soit un array ['trip' => ..., 'role' => ...]
 *
 * @param string|array $username Le nom d'utilisateur/trip ou un array avec trip et role
 * @param string|null $role Le rôle de l'utilisateur (optionnel, sera cherché si non fourni)
 * @return string HTML de l'avatar
 */
function identicon_profile($username, $role = null)
{
    global $sessionTrip, $cur, $lang;
    $plugin = 'identicon';
    $logFile = __DIR__ . '/../../identicon_debug.log';
    
    // Log pour débogage
    $logEntry = date('Y-m-d H:i:s') . ' - identicon_profile called with: ' . (is_array($username) ? 'array' : var_export($username, true)) . PHP_EOL;
    @file_put_contents($logFile, $logEntry, FILE_APPEND);
    
    // Gérer le cas où $username est un array avec trip et role
    if (is_array($username)) {
        $role = $username['role'] ?? null;
        $username = $username['trip'] ?? '';
    }
    
    if (empty($username)) {
        @file_put_contents($logFile, date('Y-m-d H:i:s') . ' - Empty username, returning' . PHP_EOL, FILE_APPEND);
        return '';
    }
    
    // S'assurer que $lang est défini
    if (!isset($lang)) {
        $lang = [];
    }
    
    // Cache statique pour éviter les lectures répétées
    static $pluginDataCache = null;
    if ($pluginDataCache === null) {
        $pluginDataCache = flatDB::readEntry('plugin', $plugin);
        @file_put_contents($logFile, date('Y-m-d H:i:s') . ' - Plugin data loaded: ' . (is_array($pluginDataCache) ? 'yes' : 'no') . PHP_EOL, FILE_APPEND);
    }
    $data = $pluginDataCache;
    
    // Vérifier si le plugin est activé
    if (!$data || !isset($data[$plugin . 'state']) || !$data[$plugin . 'state']) {
        @file_put_contents($logFile, date('Y-m-d H:i:s') . ' - Plugin not activated, returning' . PHP_EOL, FILE_APPEND);
        return '';
    }
    
    // Récupération du pseudo sans le hash
    $user = strstr($username, '@', true) ?: $username;
    
    // Taille de l'avatar en configuration avec validation
    $size = isset($data['taille']) ? max(10, min(200, intval($data['taille']))) : 50;
    
    // Forme avec valeur par défaut
    $forme = isset($data['forme']) ? htmlspecialchars($data['forme']) : 'rounded-circle';
    
    // Conversion du nom d'utilisateur
    $h1 = hash('crc32', $username);
    $h2 = hash('crc32b', $username);
    
    // Points ou carrés ?
    $d = (isset($data['dots']) && $data['dots']) ? 'dot' : '';
    
    // Génération du nom de fichier
    $s = $h1 . ($h2[0] ?? '0');
    $file_path = PATH_GEN_AVATAR . $s . 'x' . $size . $d . '.png';
    $file_html = HTML_GEN_AVATAR . $s . 'x' . $size . $d . '.png';
    
    // Lien de génération de l'avatar avec paramètres
    $dots = $d ? '&d=dot' : '';
    $pic = HTML_PLUGIN_AVATAR . 'avatar.php?u=' . urlencode($username) . '&sz=' . $size . $dots;

    /**
     * INDICATEUR DE STATUT EN LIGNE / HORS LIGNE
     */	
    $online = '';
    $showDot = isset($data['show_dot']) ? $data['show_dot'] : true;
    if ($showDot && IDENTICON_ONLINE && pluginIsHere('online')) {
        $online = checkOnline($username);
    }

    /**
     * BADGE ADMIN/MODO
     */
    $roleBadge = '';
    $showBadge = isset($data['show_badge']) ? $data['show_badge'] : true;
    if ($showBadge) {
        if ($role === null) {
            $role = getUserRole($username);
        }
        
        if ($role === 'admin') {
            $roleBadge = '<span class="avatar-role-badge avatar-role-admin" data-toggle="tooltip" data-placement="top" title="' . htmlspecialchars($lang['admin'] ?? 'Administrator', ENT_QUOTES, 'UTF-8') . '"><i class="fa fa-user-secret"></i></span>';
        } elseif ($role === 'worker') {
            $roleBadge = '<span class="avatar-role-badge avatar-role-worker" data-toggle="tooltip" data-placement="top" title="' . htmlspecialchars($lang['worker'] ?? 'Moderator', ENT_QUOTES, 'UTF-8') . '"><i class="fa fa-user"></i></span>';
        }
    }

    /**
     * GÉNÉRATION ET AFFICHAGE DE L'AVATAR
     */					  	
    $identity = '';
    
    // Conteneur wrapper pour positionner correctement les badges
    $identity .= '<span class="avatar-wrapper">';
    
    // Avatar personnalisé placé dans uploads/avatars/ sans l'arobase
    // Exemple: pseudo@motdepasse = pseudo_motdepasse.png
    $avatar = str_replace(['#', '@'], '_', $username);
    $avatarPath = PATH_IDENTICON_AVATAR . $avatar . '.png';
    
    if (file_exists($avatarPath)) {
        // Avatar personnalisé trouvé
        $identity .= '<img class="' . $forme . '" src="' . HTML_AVATAR . htmlspecialchars($avatar) . '.png" width="' . $size . '" height="' . $size . '" alt="' . htmlspecialchars($user) . '" loading="lazy">' . $online . $roleBadge; 
    } elseif (is_numeric($username)) {
        // Si le pseudo est numérique, utiliser un avatar anonyme
        $anonymousPath = HTML_PLUGIN_AVATAR . 'anonymous.png';
        if (file_exists(PLUGIN_DIR . 'identicon' . DS . 'anonymous.png')) {
            $identity .= '<img class="' . $forme . '" src="' . $anonymousPath . '" width="' . $size . '" height="' . $size . '" alt="anonymous" loading="lazy">' . $online . $roleBadge;
        }
    } elseif (!empty($file_path) && file_exists($file_path)) {
        // Si l'avatar est déjà généré, le prendre en cache
        $identity .= '<img class="' . $forme . '" src="' . $file_html . '" width="' . $size . '" height="' . $size . '" alt="' . htmlspecialchars($user) . '" loading="lazy">' . $online . $roleBadge; 
    } else {
        // Sinon, donner l'URL de génération
        $identity .= '<img class="' . $forme . '" src="' . htmlspecialchars($pic) . '" width="' . $size . '" height="' . $size . '" alt="' . htmlspecialchars($user) . '" loading="lazy">' . $online . $roleBadge;    
    }
    
    $identity .= '</span>';
    
    @file_put_contents($logFile, date('Y-m-d H:i:s') . ' - identicon_profile returning identity (length: ' . strlen($identity) . ')' . PHP_EOL, FILE_APPEND);
    
    return $identity;
}
?>