<?php
defined('FLATBOARD') or die('Flatboard Community.');
/*
 * Project name: Flatboard
 * Project URL: https://flatboard.org
 * Author: Frédéric Kaplon and contributors
 * All Flatboard code is released under the MIT license.
*/

/**
 * Classe utilitaire pour Flatboard.
 * Fournit des méthodes pour la gestion des URLs, la validation des entrées,
 * le formatage des dates, l'envoi d'e-mails, la gestion des fichiers, etc.
 */
class Util
{
    /**
     * Constructeur protégé pour empêcher l'instanciation directe.
     */
    protected function __construct()
    {
        // Rien à initialiser ici
    }

    /**
     * Vérifie si une variable GET est définie et est une chaîne de caractères.
     *
     * @param string $name Le nom de la variable GET à vérifier.
     * @return bool Retourne `true` si la variable GET est définie et est une chaîne, sinon `false`.
     */
    public static function isGET(string $name): bool
    {
        return isset($_GET[$name]) && is_string($_GET[$name]);
    }

    /**
     * Vérifie si une variable POST est définie et est une chaîne de caractères.
     *
     * @param string $name Le nom de la variable POST à vérifier.
     * @return bool Retourne `true` si la variable POST est définie et est une chaîne, sinon `false`.
     */
    public static function isPOST(string $name): bool
    {
        return isset($_POST[$name]) && is_string($_POST[$name]);
    }

    /**
     * Vérifie si une entrée GET est valide pour un type donné.
     *
     * @param string $type Le type d'entrée à valider.
     * @param string $name Le nom de la variable GET à vérifier.
     * @return bool Retourne `true` si l'entrée GET est valide, sinon `false`.
     */
    public static function isGETValidEntry(string $type, string $name): bool
    {
        return self::isGET($name) && flatDB::isValidEntry($type, $_GET[$name]);
    }

    /**
     * Vérifie si un hook GET est valide.
     *
     * @param string $hook Le nom du hook à valider.
     * @param string $name Le nom de la variable GET à vérifier.
     * @return bool Retourne `true` si le hook GET est valide, sinon `false`.
     */
    public static function isGETValidHook(string $hook, string $name): bool
    {
        return self::isGET($name) && Plugin::isValidHook($hook, $_GET[$name]);
    }

    /**
     * Extrait les paramètres de l'URL à partir de `PATH_INFO`.
     *
     * @return array Un tableau associatif contenant les paramètres extraits de l'URL.
     */
    public static function fURL(): array
    {
        $out = [];

        if (isset($_SERVER['PATH_INFO'])) {
            $info = array_map(
                function($segment) {
                    return htmlspecialchars(trim($segment), ENT_QUOTES, 'UTF-8');
                },
                explode('/', trim($_SERVER['PATH_INFO'], '/'))
            );

            $infoNum = count($info);
            for ($i = 0; $i < $infoNum; $i += 2) {
                if (isset($info[$i]) && $info[$i] !== '') {
                    $out[$info[$i]] = $info[$i + 1] ?? '';
                }
            }
        }

        return $out;
    }

    /**
     * Récupère l'URL de base du site.
     *
     * @param array|null $server Tableau $_SERVER personnalisé (pour les tests).
     * @return string L'URL de base du site.
     */
    public static function baseURL(?array $server = null): string
    {
        $server = $server ?? $_SERVER;
        $excludedFiles = [
            'add.php', 'auth.php', 'config.php', 'delete.php',
            'edit.php', 'feed.php', 'index.php', 'install.php',
            'search.php', 'view.php', 'download.php', 'blog.php', 'forum.php'
        ];

        $siteUrl = str_replace($excludedFiles, '', $server['SCRIPT_NAME']);
        $https = (isset($server['HTTPS']) && strtolower($server['HTTPS']) === 'on') ? 'https://' : 'http://';
        return $https . $server['HTTP_HOST'] . $siteUrl;
    }

    /**
     * Récupère l'URL actuelle.
     *
     * @param array|null $server Tableau $_SERVER personnalisé (pour les tests).
     * @return string L'URL actuelle.
     */
    public static function getCurrent(?array $server = null): string
    {
        $server = $server ?? $_SERVER;
        $https = (!empty($server['HTTPS']) && strtolower($server['HTTPS']) === 'on') ? 'https://' : 'http://';
        return $https . $server['SERVER_NAME'] . $server['REQUEST_URI'];
    }

    /**
     * Redirige vers une nouvelle URL.
     *
     * @param string $loc Le chemin relatif ou absolu vers lequel rediriger.
     */
    public static function redirect(string $loc): void
    {
        header('Location: ' . self::baseURL() . $loc);
        exit;
    }

    /**
     * Calcule le numéro de page d'un élément dans une liste.
     *
     * @param mixed $item L'élément à rechercher.
     * @param array $items La liste des éléments.
     * @return int Le numéro de page de l'élément.
     */
    public static function onPage($item, array $items): int
    {
        return (int) (array_search($item, array_values($items), true) / 8) + 1;
    }

    /**
     * Formate un nombre entier en notation abrégée (ex: 1000 -> 1K).
     *
     * @param int $int Le nombre à formater.
     * @return string Le nombre formaté.
     */
    public static function shortNum(int $int): string
    {
        if ($int < 1000) {
            return (string) $int;
        }
        return round($int / 1000, 1) . 'K';
    }

    /**
     * Convertit les codes strftime en codes DateTime
     * 
     * @param string $strftimeFormat Format strftime
     * @return string Format DateTime équivalent
     */
    private static function convertStrftimeToDateTime(string $strftimeFormat): string
    {
        // Mapping des codes strftime vers DateTime
        $conversions = [
            '%A' => 'l',      // Nom complet du jour de la semaine
            '%a' => 'D',      // Nom abrégé du jour de la semaine
            '%B' => 'F',      // Nom complet du mois
            '%b' => 'M',      // Nom abrégé du mois
            '%d' => 'd',      // Jour du mois (01-31)
            '%e' => 'j',      // Jour du mois sans zéro initial (1-31)
            '%H' => 'H',      // Heure 24h (00-23)
            '%I' => 'h',      // Heure 12h (01-12)
            '%m' => 'm',      // Mois (01-12)
            '%M' => 'i',      // Minutes (00-59)
            '%p' => 'A',      // AM/PM
            '%P' => 'a',      // am/pm
            '%S' => 's',      // Secondes (00-59)
            '%T' => 'H:i:s',  // Heure complète (HH:MM:SS)
            '%Y' => 'Y',      // Année sur 4 chiffres
            '%y' => 'y',      // Année sur 2 chiffres
            '%z' => 'O',      // Décalage horaire
            '%Z' => 'T',      // Fuseau horaire
            '%%' => '%',      // Caractère %
        ];
        
        // Si le format ne contient pas de %, c'est probablement déjà un format DateTime
        if (strpos($strftimeFormat, '%') === false) {
            return $strftimeFormat;
        }
        
        return str_replace(array_keys($conversions), array_values($conversions), $strftimeFormat);
    }

    /**
     * Convertit un identifiant de date en une chaîne de date formatée.
     * Le format de l'ID est YYYY-MM-DDHHMMSS (sans séparateur entre date et heure).
     * Supporte les formats strftime (comme %A, %d, %B) et les formats DateTime.
     *
     * @param string $id L'identifiant contenant la date.
     * @param string $pattern Le format de date à utiliser (par défaut 'Y/m/d H:i').
     * @param bool $cooldate Indique si le format "cool" doit être utilisé pour les dates récentes.
     * @return string La date formatée.
     */
    public static function toDate(string $id, string $pattern = 'Y/m/d H:i', bool $cooldate = true): string
    {
        global $lang, $config;

        // Extraire les 16 premiers caractères (YYYY-MM-DDHHMMSS)
        $dateStr = substr($id, 0, 16);
        
        // Convertir le format YYYY-MM-DDHHMMSS en YYYY-MM-DD HH:MM:SS pour le parsing
        $formattedDate = null;
        if (preg_match('/^(\d{4})-(\d{2})(\d{2})(\d{2})(\d{2})(\d{2})$/', $dateStr, $matches)) {
            $formattedDate = "{$matches[1]}-{$matches[2]}-{$matches[3]} {$matches[4]}:{$matches[5]}:{$matches[6]}";
        } else {
            // Fallback : essayer de parser tel quel
            $formattedDate = $dateStr;
        }

        try {
            $dateTimeObj = new DateTimeImmutable($formattedDate, new DateTimeZone('Europe/Paris'));
            $timestamp = $dateTimeObj->getTimestamp();
        } catch (Exception $e) {
            // En cas d'erreur, utiliser la date actuelle
            error_log("Date parsing error for ID {$id}: " . $e->getMessage());
            $dateTimeObj = new DateTimeImmutable('now', new DateTimeZone('Europe/Paris'));
            $timestamp = $dateTimeObj->getTimestamp();
        }

        $diff = time() - $timestamp;

        // Convertir le pattern strftime en format DateTime si nécessaire
        $dateTimePattern = self::convertStrftimeToDateTime($pattern);
        $configDateFormat = isset($config['date_format']) ? self::convertStrftimeToDateTime($config['date_format']) : $dateTimePattern;

        if ($dateTimePattern === $configDateFormat && $cooldate && $diff < 604800) {
            $periods = [86400 => $lang['day'], 3600 => $lang['hour'], 60 => $lang['minute'], 1 => $lang['second']];
            foreach ($periods as $key => $value) {
                if ($diff >= $key) {
                    $num = (int) ($diff / $key);
                    return sprintf(
                        defined('TIMESTAMP') && TIMESTAMP ? '%d %s%s %s' : '%s %d %s%s',
                        $num,
                        $value,
                        $num > 1 ? $lang['plural'] : '',
                        $lang['ago']
                    );
                }
            }
        }

        if ($config['lang'] === 'fr-FR' && class_exists('IntlDateFormatter')) {
            static $formatter = null;
            if ($formatter === null) {
                $formatter = new IntlDateFormatter(
                    'fr_FR',
                    IntlDateFormatter::MEDIUM,
                    IntlDateFormatter::MEDIUM,
                    'Europe/Paris',
                    IntlDateFormatter::GREGORIAN,
                    "EEEE d MMMM y 'à' HH:mm"
                );
            }
            return ucwords($formatter->format($dateTimeObj));
        }

        return $dateTimeObj->format($dateTimePattern);
    }

	/**
	 * Vérifie la présence de la mention de copyright dans le fichier de langue.
	 * Affiche un message d'avertissement si la mention est manquante.
	 *
	 * @param string $langFile Le chemin vers le fichier de langue à vérifier.
	 */
	public static function checkCopy(string $langFile): void
	{
		global $lang, $config;

		// Vérifie si le fichier existe
		if (!file_exists($langFile)) {
			self::showCopyrightWarning("Le fichier de langue est introuvable : " . htmlspecialchars($langFile, ENT_QUOTES, 'UTF-8'));
			return;
		}

		// Vérifie la présence de la mention de copyright
		$content = @file_get_contents($langFile);
		if ($content === false) {
			self::showCopyrightWarning("Impossible de lire le fichier de langue : " . htmlspecialchars($langFile, ENT_QUOTES, 'UTF-8'));
			return;
		}

		if (strpos($content, 'false;">Flatboard') === false) {
			self::showCopyrightWarning();
		}
	}

	/**
	 * Affiche un message d'avertissement concernant le copyright.
	 *
	 * @param string|null $customMessage Message personnalisé à afficher.
	 */
	private static function showCopyrightWarning(?string $customMessage = null): void
	{
		global $lang, $config;

		// Contenu HTML à ajouter au fichier de langue
		$requiredContent = base64_decode('PGRpdiBzdHlsZT0idGV4dC1hbGlnbjpsZWZ0O2NvbG9yOmJsYWNrOyBiYWNrZ3JvdW5kLWNvbG9yOndoaXRlOyBwYWRkaW5nOjAuNWVtOyBvdmVyZmxvdzphdXRvO2ZvbnQtc2l6ZTpzbWFsbDsgZm9udC1mYW1pbHk6bW9ub3NwYWNlOyAiPjxzcGFuIHN0eWxlPSJjb2xvcjojZGQyNDAwOyI+J3Bvd2VyZWQnPC9zcGFuPiA8c3BhbiBzdHlsZT0iY29sb3I6IzAwNmZmODsiPj0mZ3Q7PC9zcGFuPiA8c3BhbiBzdHlsZT0iY29sb3I6I2RkMjQwMDsiPidDcmVhdGVkIHdpdGggJmx0O2EgaHJlZj0mcXVvdDtodHRwOi8vZmxhdGJvYXJkLmZyZWUuZnImcXVvdDsgb25jbGljaz0mcXVvdDt3aW5kb3cub3Blbih0aGlzLmhyZWYpOyByZXR1cm4gZmFsc2U7JnF1b3Q7Jmd0O0ZsYXRib2FyZCc8L3NwYW4+IDxzcGFuIHN0eWxlPSJjb2xvcjojMDA2ZmY4OyI+Ljwvc3Bhbj4oPHNwYW4gc3R5bGU9ImNvbG9yOiM0MDAwODA7Ij5kZWZpbmVkPC9zcGFuPig8c3BhbiBzdHlsZT0iY29sb3I6I2RkMjQwMDsiPidGTEFUQk9BUkRfUFJPJzwvc3Bhbj4pPzxzcGFuIHN0eWxlPSJjb2xvcjojZGQyNDAwOyI+JyBQcm9uPC9zcGFuPjo8c3BhbiBzdHlsZT0iY29sb3I6IzAwNmZmODsiPi48L3NwYW4+IDxzcGFuIHN0eWxlPSJjb2xvcjojZGQyNDAwOyI+JyZsdDsvYSZndDsgYW5kICZsdDtpIGNsYXNzPSZxdW90O2ZhIGZhLWhlYXJ0JnF1b3Q7Jmd0OyZsdDsvaSZndDsuJzwvc3Bhbj4sPC9kaXY+');

		// Message personnalisé ou message par défaut
		$message = $customMessage ?? sprintf(
			'Pour respecter la <a href="%s" target="_blank">licence MIT</a>, veuillez ajouter la mention de copyright suivante dans le fichier de langue :',
			'https://github.com/Fred89/flatboard/blob/master/LICENSE'
		);

		// Code HTML à ajouter dans le fichier de langue
		$codeToAdd = htmlspecialchars('$lang[\'powered\'] = \'Created with <a href="http://flatboard.free.fr" onclick="window.open(this.href); return false;">Flatboard</a>\'.(defined(\'FLATBOARD_PRO\') ? \' Pron\' : \'.\').\' and <i class="fa fa-heart"></i>.\';', ENT_QUOTES, 'UTF-8');

		// Chemin du fichier de langue
		$langFilePath = LANG_DIR . $config['lang'] . '.php';

		// Affichage du message d'avertissement
		echo <<<HTML
		<div class="container mt-4">
			<div class="alert alert-warning alert-dismissible fade show" role="alert" style="border-left: 4px solid #ffc107;">
				<div class="d-flex align-items-center">
					<div class="flex-shrink-0">
						<i class="fa fa-exclamation-triangle fa-2x" style="color: #ffc107;"></i>
					</div>
					<div class="flex-grow-1 ms-3">
						<h4 class="alert-heading">Attention - Mention de copyright manquante</h4>
						<p>$message</p>
						<div class="mb-3">
							<strong>Fichier concerné :</strong> <code>$langFilePath</code><br>
							<strong>Code à ajouter :</strong>
							<pre class="mt-2 p-2 bg-light border rounded">$codeToAdd</pre>
						</div>
						<hr>
						<p class="mb-0">
							<strong>Pourquoi ce message ?</strong><br>
							Flatboard est un logiciel open-source sous licence MIT. Pour soutenir le projet, nous demandons simplement de conserver cette mention de copyright dans les fichiers de langue.
						</p>
					</div>
				</div>
				<button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Fermer"></button>
			</div>
		</div>
		HTML;
	}

    /**
     * Récupère la taille d'un fichier en format lisible (B, KB, MB, GB, TB).
     *
     * @param string $file Chemin du fichier.
     * @param int $digits Nombre de chiffres après la virgule.
     * @return string|false Taille formatée ou `false` si le fichier n'existe pas.
     */
    public static function getFilesize(string $file, int $digits = 2)
    {
        if (!file_exists($file)) {
            return false;
        }

        $bytes = filesize($file);
        if ($bytes < 1024) {
            return $bytes . ' B';
        } elseif ($bytes < 1048576) {
            return round($bytes / 1024, $digits) . ' KB';
        } elseif ($bytes < 1073741824) {
            return round($bytes / 1048576, $digits) . ' MB';
        } elseif ($bytes < 1099511627776) {
            return round($bytes / 1073741824, $digits) . ' GB';
        } else {
            return round($bytes / 1099511627776, $digits) . ' TB';
        }
    }

    /**
     * Convertit une taille en octets en une chaîne lisible (B, KB, MB, GB, TB).
     *
     * @param int $size Taille en octets.
     * @return string Taille formatée.
     */
    public static function sizeConversion(int $size): string
    {
        if ($size < 0) {
            return '0 B';
        }

        $unit = ['B', 'KB', 'MB', 'GB', 'TB', 'PB'];
        $i = (int) floor(log($size, 1024));
        return round($size / pow(1024, $i), 2) . ' ' . $unit[$i];
    }

    /**
     * Récupère la mémoire utilisée par le script.
     *
     * @param string $type Type de mémoire à récupérer ('available', 'peak', 'usage').
     * @return string Taille de la mémoire formatée.
     */
    public static function getMemory(string $type = 'usage'): string
    {
        switch ($type) {
            case 'available':
                $memoryAvailable = (int) filter_var(ini_get("memory_limit"), FILTER_SANITIZE_NUMBER_INT);
                $size = $memoryAvailable * 1024 * 1024;
                break;
            case 'peak':
                $size = (int) memory_get_peak_usage(true);
                break;
            case 'usage':
                $size = (int) memory_get_usage(true);
                break;
            default:
                $size = 0;
        }

        return self::sizeConversion($size);
    }

    /**
     * Vérifie si les modules PHP requis sont installés.
     *
     * @param array $modules Liste des modules à vérifier.
     * @return string|null Message d'erreur ou `null` si tous les modules sont installés.
     */
    public static function checkModules(array $modules = ['mbstring', 'json', 'gd', 'dom']): ?string
    {
        $missing = [];
        foreach ($modules as $module) {
            if (!extension_loaded($module)) {
                $errorText = 'PHP module <b>' . $module . '</b> is not installed.';
                error_log('[ERROR] ' . $errorText, 0);
                $missing[] = $errorText;
            }
        }

        return !empty($missing) ? 'PHP modules missing:<br>' . implode('<br>', $missing) : null;
    }

    /**
     * Récupère une chaîne de langue formatée à partir des arguments fournis.
     *
     * @param string $format Le format de la chaîne, contenant des mots séparés par des espaces.
     * @return string La chaîne formatée.
     */
    public static function lang(string $format): string
    {
        global $lang;

        $argList = func_get_args();
        $wordList = [];

        foreach (explode(' ', $format) as $word) {
            $wordList[] = $lang[$word] ?? $word;
        }

        return vsprintf(implode($lang['useSpace'] ? ' ' : '', $wordList), array_slice($argList, 1));
    }

    /**
     * Teste si la fonction PHP `mail()` est disponible.
     *
     * @param bool $io Si `true`, affiche le résultat du test.
     * @param string $format Format d'affichage du message.
     * @return bool|string Retourne `true` si `mail()` est disponible, sinon un message d'erreur.
     */
    public static function testMail(bool $io = true, string $format = "<div class=\"alert alert-#color\" role=\"alert\">#symbol #message</div>\n")
    {
        global $lang;
        $return = function_exists('mail');

        if ($io) {
            $color = $return ? 'success' : 'danger';
            $symbol = $return ? '&#10004;' : '&#10007;';
            $message = $return ? $lang['mail_available'] : $lang['mail_not_available'];

            return str_replace(['#color', '#symbol', '#message'], [$color, $symbol, $message], $format);
        }

        return $return;
    }

    /**
     * Envoie un e-mail.
     *
     * @param string $name Nom de l'expéditeur.
     * @param string $from Adresse e-mail de l'expéditeur.
     * @param array|string $to Adresse(s) du(des) destinataire(s).
     * @param string $subject Sujet de l'e-mail.
     * @param string $body Contenu de l'e-mail.
     * @param string $contentType Type de contenu ('text' ou 'html').
     * @param array|string|false $cc Adresse(s) en copie carbone (CC).
     * @param array|string|false $bcc Adresse(s) en copie carbone invisible (BCC).
     * @return bool Retourne `true` si l'e-mail a été envoyé avec succès, sinon `false`.
     */
    public static function sendMail(
        string $name,
        string $from,
        $to,
        string $subject,
        string $body,
        string $contentType = "text",
        $cc = false,
        $bcc = false
    ): bool {
        if (is_array($to)) {
            $to = implode(', ', $to);
        }
        if (is_array($cc)) {
            $cc = implode(', ', $cc);
        }
        if (is_array($bcc)) {
            $bcc = implode(', ', $bcc);
        }

        $headers = [
            "From: " . $name . " <" . $from . ">",
            "Reply-To: " . $from,
            'MIME-Version: 1.0',
            'Content-Type: text/' . ($contentType === 'html' ? 'html' : 'plain') . '; charset="' . CHARSET . '"',
            'Content-Transfer-Encoding: 8bit',
            'Date: ' . date("D, j M Y G:i:s O")
        ];

        if (!empty($cc)) {
            $headers[] = 'Cc: ' . $cc;
        }
        if (!empty($bcc)) {
            $headers[] = 'Bcc: ' . $bcc;
        }

        return mail($to, $subject, $body, implode("\r\n", $headers));
    }

    /**
     * Retourne la forme plurielle ou singulière d'un mot en fonction du nombre.
     *
     * @param int $num Le nombre à évaluer.
     * @param string $plural La forme plurielle du mot.
     * @param string $single La forme singulière du mot.
     * @return string La forme appropriée du mot.
     */
    public static function pluralize(int $num, string $plural = 's', string $single = ''): string
    {
        return ($num === 0 || $num === 1) ? $single : $plural;
    }

    /**
     * Récupère la description en fonction de la page actuelle.
     *
     * @return string La description de la page.
     */
    public static function Description(): string
    {
        global $config, $cur, $out;
        return ($cur === 'home') ? $config['description'] : $out['subtitle'];
    }

    /**
     * Génère un bouton d'aide avec une info-bulle.
     *
     * @param string $phrase La phrase à afficher dans l'info-bulle.
     * @return string Le code HTML du bouton d'aide.
     */
    public static function Help(string $phrase): string
    {
        global $lang;
        $lang['more_info'] = $lang['more_info'] ?? '';

        return '&nbsp;<a class="text-primary" data-toggle="tooltip" data-placement="top" title="' . $lang['more_info'] . '">
                <i class="fa fa-info-circle fa-lg" aria-hidden="true" data-container="body" data-toggle="popover" data-placement="right" data-content="' . $phrase . '"></i>
            </a>';
    }

    /**
     * Supprime tous les fichiers d'un dossier.
     *
     * @param string $dir Le chemin du dossier dont les fichiers doivent être supprimés.
     * @throws RuntimeException Si un fichier ne peut pas être supprimé.
     */
    public static function deletecache(string $dir): void
    {
        global $lang;

        $files = glob($dir . '{,.}*', GLOB_BRACE);
        if (empty($files)) {
            Plugin::redirectMsg($lang['no_files_found'], 'config.php', $lang['config'], 'alert alert-warning');
            return;
        }

        foreach ($files as $file) {
            if (is_file($file) && !unlink($file)) {
                throw new RuntimeException("Impossible de supprimer le fichier : " . $file);
            }
        }

        Plugin::redirectMsg($lang['cache_clean'], 'config.php', $lang['config'], 'alert alert-success');
    }

    /**
     * Applique une compression HTTP (gzip) à la réponse si le client la supporte.
     *
     * Cette méthode vérifie si le client accepte la compression gzip ou x-gzip,
     * puis compresse la sortie avant de l'envoyer.
     * Si la compression n'est pas supportée, la sortie est envoyée sans modification.
     *
     * @return void Termine l'exécution du script après l'envoi de la réponse.
     * @throws RuntimeException Si les en-têtes ont déjà été envoyés.
     */
    public static function httpEncoding(): void
    {
        global $HTTP_ACCEPT_ENCODING;

        // Vérifie si les en-têtes ont déjà été envoyés
        if (headers_sent()) {
            throw new RuntimeException("Les en-têtes HTTP ont déjà été envoyés.");
        }

        // Récupère l'en-tête Accept-Encoding depuis $_SERVER si $HTTP_ACCEPT_ENCODING n'est pas défini
        $acceptEncoding = $HTTP_ACCEPT_ENCODING ?? ($_SERVER['HTTP_ACCEPT_ENCODING'] ?? '');

        // Détermine le type de compression disponible
        $encoding = false;
        if (strpos($acceptEncoding, 'x-gzip') !== false) {
            $encoding = 'x-gzip';
        } elseif (strpos($acceptEncoding, 'gzip') !== false) {
            $encoding = 'gzip';
        }

        // Si un encodage est trouvé, compresse le contenu
        if ($encoding) {
            // Récupère le contenu du tampon de sortie
            $contents = ob_get_contents();
            if ($contents === false) {
                throw new RuntimeException("Impossible de récupérer le contenu du tampon de sortie.");
            }

            // Nettoie le tampon de sortie
            ob_end_clean();

            // Envoie l'en-tête de compression
            header('Content-Encoding: ' . $encoding);

            // Envoie l'en-tête GZIP
            echo "\x1f\x8b\x08\x00\x00\x00\x00\x00";

            // Compresse le contenu et l'envoie
            echo gzcompress($contents, 9);

            // Termine le script
            exit;
        }

        // Si aucun encodage n'est disponible, envoie le contenu non compressé
        ob_end_flush();
        exit;
    }

    /**
     * Méthode de journalisation pour enregistrer des messages dans les logs.
     *
     * Cette méthode permet d'enregistrer des messages de log dans le journal des erreurs PHP.
     * Elle supporte les tableaux en les décomposant pour une meilleure lisibilité.
     *
     * @param string|array $text Le texte ou tableau à journaliser.
     * @param int $type Le type de log (0 pour error_log, 3 pour un fichier spécifique).
     * @return void
     */
    public static function journal($text, int $type = 0): void
    {
        if (is_array($text)) {
            error_log('------------------------', $type);
            error_log('Array', $type);
            error_log('------------------------', $type);
            foreach ($text as $key => $value) {
                error_log($key . '=>' . $value, $type);
            }
            error_log('------------------------', $type);
        }

        error_log(
            '(' . (defined('VERSION') ? VERSION : 'unknown') . ') (' .
            ($_SERVER['REQUEST_URI'] ?? 'CLI') . ') ' .
            (is_string($text) ? $text : print_r($text, true)),
            $type
        );
    }
}