<?php defined('FLATBOARD') or die('Flatboard Community.');
/*
 * Project name: Flatboard
 * Project URL: https://flatboard.org
 * Author: Frédéric Kaplon and contributors
 * All Flatboard code is released under the MIT license.
 * 
 * Classe pour la gestion des utilisateurs dans Flatboard.
 *
 * Fournit des méthodes statiques pour l'authentification, la vérification des rôles,
 * la gestion des clés, la protection des données et la gestion des bannissements.
 */
class User
{
    /** @var string Chemin du répertoire des données */
    private static $DATA_DIR = DATA_DIR;

    /** @var string Nom du fichier de clé */
    private static $KEY_FILE = 'key.php';

    /** @var string Nom du fichier de liste de bannissement */
    private static $BAN_FILE = BAN_FILE;

    /** @var string Rôle administrateur */
    private static $ROLE_ADMIN = 'admin';

    /** @var string Rôle modérateur */
    private static $ROLE_WORKER = 'worker';

    /** @var string Méthode de chiffrement */
    private static $ENCRYPT_METHOD = 'AES-256-CBC';

    /** @var int Durée maximale d'inactivité en secondes (30 minutes) */
    private const SESSION_TIMEOUT = 1800;

    /**
     * Constructeur protégé pour empêcher l'instanciation.
     */
    protected function __construct()
    {
        // Classe statique
    }

    /**
     * Initialise la session de manière sécurisée.
     *
     * Configure les paramètres de session pour améliorer la sécurité :
     * - Cookie HTTPOnly et Secure (si HTTPS).
     * - SameSite pour prévenir CSRF.
     * - Chemin de sauvegarde personnalisé si défini.
     *
     * @param string|null $savePath Chemin de sauvegarde des sessions (facultatif).
     * @return void
     * @throws RuntimeException Si la session ne peut pas être démarrée.
     */
    public static function secureSessionStart(?string $savePath = null): void
    {
        if (session_status() !== PHP_SESSION_NONE) {
            if (DEBUG_MODE) {
                error_log('User::secureSessionStart: Session déjà démarrée (status=' . session_status() . ').');
            }
            return; // Session déjà démarrée
        }

        // Configuration sécurisée
        ini_set('session.cookie_httponly', '1');
        ini_set('session.cookie_secure', !empty($_SERVER['HTTPS']) && strtolower($_SERVER['HTTPS']) === 'on' ? '1' : '0');
        ini_set('session.cookie_samesite', 'Strict');
        ini_set('session.use_strict_mode', '1');

        if ($savePath && is_writable($savePath)) {
            session_save_path($savePath);
            if (DEBUG_MODE) {
                error_log('User::secureSessionStart: Chemin de sauvegarde des sessions défini à ' . $savePath);
            }
        }

        if (!session_start()) {
            throw new RuntimeException('Impossible de démarrer la session sécurisée.');
        }

        // Initialisation des variables de session si non définies
        if (!isset($_SESSION['last_activity'])) {
            $_SESSION['last_activity'] = time();
            if (DEBUG_MODE) {
                error_log('User::secureSessionStart: Session initialisée, last_activity défini.');
            }
        }

        // Vérification du timeout (désactivé en mode débogage)
        if (!DEBUG_MODE && isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity'] > self::SESSION_TIMEOUT)) {
            self::logout();
            if (DEBUG_MODE) {
                error_log('User::secureSessionStart: Session expirée en raison du timeout.');
            }
        } else {
            $_SESSION['last_activity'] = time(); // Mise à jour de l'activité
        }
    }

    /**
     * Vérifie si l'utilisateur a des droits administratifs.
     *
     * @return bool True si l'utilisateur est administrateur, false sinon.
     */
    public static function isAdmin(): bool
    {
        self::secureSessionStart();
        if (!DEBUG_MODE) {
            try {
                self::validateSession();
            } catch (SecurityException $e) {
                if (DEBUG_MODE) {
                    error_log('User::isAdmin: ' . $e->getMessage());
                }
                return false;
            }
        }
        return isset($_SESSION['role']) && $_SESSION['role'] === self::$ROLE_ADMIN;
    }

    /**
     * Vérifie si l'utilisateur a des droits de modérateur ou administrateur.
     *
     * @return bool True si l'utilisateur est modérateur ou administrateur, false sinon.
     */
    public static function isWorker(): bool
    {
        self::secureSessionStart();
        if (!DEBUG_MODE) {
            try {
                self::validateSession();
            } catch (SecurityException $e) {
                if (DEBUG_MODE) {
                    error_log('User::isWorker: ' . $e->getMessage());
                }
                return false;
            }
        }
        return isset($_SESSION['role']) && in_array($_SESSION['role'], [self::$ROLE_WORKER, self::$ROLE_ADMIN], true);
    }

    /**
     * Valide l'intégrité de la session en vérifiant IP et User-Agent.
     *
     * @return void
     * @throws SecurityException Si la session est invalide (hijacking suspecté).
     */
    private static function validateSession(): void
    {
        if (DEBUG_MODE) {
            return; // Désactiver la validation en mode débogage
        }

        $ip = self::getRealIpAddr();
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';

        // Vérifie si les variables de session sont définies
        if (!isset($_SESSION['ip']) || !isset($_SESSION['user_agent'])) {
            if (DEBUG_MODE) {
                error_log('User::validateSession: Variables de session ip ou user_agent non définies. Session initiale ou non authentifiée.');
            }
            return; // Session non initialisée, pas d'erreur fatale
        }

        if ($_SESSION['ip'] !== $ip || $_SESSION['user_agent'] !== $userAgent) {
            $message = sprintf(
                'Session invalide : IP attendu=%s, IP actuel=%s; User-Agent attendu=%s, User-Agent actuel=%s',
                $_SESSION['ip'], $ip, $_SESSION['user_agent'], $userAgent
            );
            self::logFailedAttempt($message);
            self::logout();
            throw new SecurityException('Session invalide : IP ou User-Agent modifié. Hijacking suspecté.');
        }
    }

    /**
     * Authentifie un utilisateur.
     *
     * @param string $trip Identifiant de l'utilisateur (mot de passe).
     * @param array $config Configuration du site (par défaut : global $config).
     * @param array $lang Traductions (par défaut : global $lang).
     * @return bool True si l'authentification réussit, false sinon.
     * @throws InvalidArgumentException Si la syntaxe de l'identifiant est invalide.
     * @throws SecurityException En cas d'échec d'authentification (loggé).
     */
    public static function login(string $trip, array $config = [], array $lang = []): bool
    {
        global $config, $lang;
        $config = $config ?: $config;
        $lang = $lang ?: $lang;

        self::secureSessionStart();

        if (empty($trip)) {
            $_SESSION['bad_user_syntax'] = 1;
            self::logFailedAttempt('Syntaxe invalide pour trip.');
            return false;
        }
        
        // Inclusion sécurisée du parseur
        if (!class_exists('Parser', false)) {
            if (defined('LIB_DIR') && file_exists(LIB_DIR . 'Parser.lib.php')) {
                require_once LIB_DIR . 'Parser.lib.php';
            } else {
                self::logFailedAttempt('Fichier Parser.lib.php introuvable.');
                return false;
            }
        }

        if (!method_exists('Parser', 'translitIt')) {
            self::logFailedAttempt('Méthode Parser::translitIt non définie.');
            return false;
        }

        $tripNocrypt = HTMLForm::clean(Parser::translitIt($trip));
        $tripCrypt = HTMLForm::trip($tripNocrypt, $trip);

        // Liaison à IP et User-Agent
        $ip = self::getRealIpAddr();
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';

        // Authentification administrateur
        if (isset($config['admin']) && $tripCrypt === $config['admin']) {
            $_SESSION['role'] = self::$ROLE_ADMIN;
            $_SESSION['trip'] = $tripNocrypt;
            $_SESSION['mail'] = $config['mail'] ?? '';
            $_SESSION['ip'] = $ip;
            $_SESSION['user_agent'] = $userAgent;
            $_SESSION['last_activity'] = time();
            session_regenerate_id(true);
            if (DEBUG_MODE) {
                error_log('User::login: Authentification réussie pour admin, IP=' . $ip);
            }
            return true;
        }

        // Authentification modérateur
        if (isset($config['worker'][$tripCrypt])) {
            $_SESSION['role'] = self::$ROLE_WORKER;
            $_SESSION['trip'] = $tripNocrypt;
            $_SESSION['mail'] = $config['worker'][$tripCrypt];
            $_SESSION['ip'] = $ip;
            $_SESSION['user_agent'] = $userAgent;
            $_SESSION['last_activity'] = time();
            session_regenerate_id(true);
            if (DEBUG_MODE) {
                error_log('User::login: Authentification réussie pour worker, IP=' . $ip);
            }
            return true;
        }

        // Authentification utilisateur (format email)
        if (strpos($trip, '@') !== false) {
            $parts = explode('@', $trip);
            if (count($parts) === 2 && strlen($parts[0]) > 0 && strlen($parts[1]) > 0) {
                $_SESSION['role'] = '';
                $_SESSION['trip'] = $tripNocrypt;
                $_SESSION['mail'] = $tripNocrypt;
                $_SESSION['ip'] = $ip;
                $_SESSION['user_agent'] = $userAgent;
                $_SESSION['last_activity'] = time();
                session_regenerate_id(true);
                if (DEBUG_MODE) {
                    error_log('User::login: Authentification réussie pour utilisateur (email), IP=' . $ip);
                }
                return true;
            }
        }

        // Échec
        self::logFailedAttempt('Échec d\'authentification pour trip: ' . $tripNocrypt);
        return false;
    }

    /**
     * Déconnecte l'utilisateur et détruit la session.
     *
     * Régénère l'ID de session avant destruction pour éviter la réutilisation.
     *
     * @return void
     */
    public static function logout(): void
    {
        self::secureSessionStart();
        session_regenerate_id(true); // Régénère avant destruction
        session_unset();
        session_destroy();
        if (DEBUG_MODE) {
            error_log('User::logout: Session détruite pour IP=' . self::getRealIpAddr());
        }
    }

    /**
     * Journalise une tentative d'authentification échouée.
     *
     * @param string $message Message à journaliser.
     * @return void
     */
    private static function logFailedAttempt(string $message): void
    {
        $ip = self::getRealIpAddr();
        error_log("Tentative échouée [$ip]: $message");
    }

    /**
     * Vérifie si l'utilisateur est l'auteur d'une entrée.
     *
     * @param string $entry Identifiant de l'entrée.
     * @param string $data Type d'entrée ('topic', 'reply' ou autre).
     * @param string|null $sessionTrip Identifiant de session (par défaut : global $sessionTrip).
     * @return bool True si l'utilisateur est l'auteur, false sinon.
     * @throws RuntimeException Si l'entrée ne peut pas être lue.
     */
    public static function isAuthor($entry, $data = '', $sessionTrip = null): bool
    {
        global $sessionTrip;
        $sessionTrip = isset($sessionTrip) ? $sessionTrip : $sessionTrip;

        self::secureSessionStart();

        if (empty($entry)) {
            return false;
        }

        if (in_array($data, ['topic', 'reply'], true)) {
            $entryData = flatDB::readEntry($data, $entry);
            if ($entryData === null) {
                throw new RuntimeException("Impossible de lire l'entrée '$entry' de type '$data'.");
            }

            if (!isset($sessionTrip)) {
                return false;
            }

            $tripCrypt = HTMLForm::trip($sessionTrip, $entry);
            $isAuthor = $tripCrypt === ($entryData['trip'] ?? '');

            return strpos($tripCrypt, '@') !== false ? $isAuthor : isset($_SESSION[$entry]);
        }

        return isset($_SESSION[$entry]);
    }

    /**
     * Chiffre ou déchiffre une chaîne avec une clé sécurisée.
     * Migration transparente des anciennes données vers le nouveau format.
     *
     * @param string $string Chaîne à  chiffrer/déchiffrer.
     * @param string $action 'e' pour chiffrer, 'd' pour déchiffrer.
     * @return string|bool Chaîne chiffrée ou déchiffrée, false en cas d'erreur.
     * @throws RuntimeException Si la clé n'est pas définie.
     */
    public static function simple_crypt($string, $action = 'e')
    {
        if (empty($string)) {
            return false;
        }

        if (!defined('KEY')) {
            throw new RuntimeException("La constante KEY n'est pas définie pour le chiffrement.");
        }

        $key = hash('sha256', KEY);
        
        if ($action === 'e') {
            $iv = openssl_random_pseudo_bytes(16);
            $encrypted = openssl_encrypt($string, self::$ENCRYPT_METHOD, $key, 0, $iv);
            if ($encrypted === false) {
                return false;
            }
            return base64_encode($iv . $encrypted);
        }

        if ($action === 'd') {
            // Essai de déchiffrement moderne d'abord
            $modernResult = self::modernDecrypt($string, $key);
            if ($modernResult !== false) {
                return $modernResult;
            }
            
            // Si échec, essayer les méthodes legacy
            return self::legacyDecrypt($string, $key);
        }

        return false;
    }

    /**
     * Déchiffrement moderne avec AES-256-CBC.
     *
     * @param string $string Chaîne à déchiffrer.
     * @param string $key Clé de chiffrement.
     * @return string|bool Résultat déchiffré ou false.
     */
    private static function modernDecrypt($string, $key)
    {
        $data = base64_decode($string, true);
        if ($data === false || strlen($data) < 20) {
            return false;
        }
        
        $iv = substr($data, 0, 16);
        $encrypted = substr($data, 16);
        
        if (strlen($iv) !== 16) {
            return false;
        }
        
        return openssl_decrypt($encrypted, self::$ENCRYPT_METHOD, $key, 0, $iv);
    }

    /**
     * Déchiffrement legacy pour les anciennes données.
     * Essaie plusieurs méthodes communes de chiffrement/encodage.
     *
     * @param string $string Chaîne à déchiffrer.
     * @param string $key Clé de chiffrement.
     * @return string|bool Résultat déchiffré ou false.
     */
    private static function legacyDecrypt($string, $key)
    {
        // Méthode 1: Simple base64 (pas de chiffrement réel)
        $decoded = base64_decode($string, true);
        if ($decoded !== false && self::isPrintableString($decoded)) {
            if (defined('DEBUG_MODE') && DEBUG_MODE) {
                error_log('Legacy migration: Base64 simple détecté pour: ' . substr($decoded, 0, 10) . '...');
            }
            return $decoded;
        }
        
        // Méthode 2: XOR avec clé (ancienne méthode Flatboard)
        if ($decoded !== false) {
            $keyLength = strlen($key);
            $result = '';
            for ($i = 0; $i < strlen($decoded); $i++) {
                $result .= chr(ord($decoded[$i]) ^ ord($key[$i % $keyLength]));
            }
            if (self::isPrintableString($result)) {
                if (defined('DEBUG_MODE') && DEBUG_MODE) {
                    error_log('Legacy migration: XOR détecté pour: ' . substr($result, 0, 10) . '...');
                }
                return $result;
            }
        }
        
        // Méthode 3: ROT13 + base64 (autre méthode legacy possible)
        $rot13 = str_rot13($string);
        $rot13Decoded = base64_decode($rot13, true);
        if ($rot13Decoded !== false && self::isPrintableString($rot13Decoded)) {
            if (defined('DEBUG_MODE') && DEBUG_MODE) {
                error_log('Legacy migration: ROT13+Base64 détecté pour: ' . substr($rot13Decoded, 0, 10) . '...');
            }
            return $rot13Decoded;
        }
        
        // Méthode 4: Essayer un déchiffrement AES simple sans IV (anciennes versions OpenSSL)
        $simpleDecrypted = openssl_decrypt($string, 'AES-256-ECB', $key, 0);
        if ($simpleDecrypted !== false && self::isPrintableString($simpleDecrypted)) {
            if (defined('DEBUG_MODE') && DEBUG_MODE) {
                error_log('Legacy migration: AES-ECB détecté pour: ' . substr($simpleDecrypted, 0, 10) . '...');
            }
            return $simpleDecrypted;
        }
        
        // Méthode 5: Si rien ne fonctionne mais que ça ressemble à un hash/identifiant, on le retourne tel quel
        if (preg_match('/^[a-zA-Z0-9+\/=_-]{20,}$/', $string)) {
            if (defined('DEBUG_MODE') && DEBUG_MODE) {
                error_log('Legacy migration: Hash/ID détecté, retour tel quel: ' . substr($string, 0, 20) . '...');
            }
            return $string;
        }
        
        return false;
    }

    /**
     * Vérifie si une chaîne contient des caractères imprimables/valides.
     *
     * @param string $string Chaîne à vérifier.
     * @return bool True si la chaîne semble valide.
     */
    private static function isPrintableString($string)
    {
        if (empty($string)) {
            return false;
        }
        
        // Vérifier si c'est un email valide
        if (filter_var($string, FILTER_VALIDATE_EMAIL)) {
            return true;
        }
        
        // Vérifier si c'est du texte imprimable (alphanumériques + quelques caractères spéciaux)
        if (preg_match('/^[a-zA-Z0-9@._-]+$/', $string) && strlen($string) >= 3) {
            return true;
        }
        
        // Vérifier si ça contient majoritairement des caractères imprimables
        $printable = 0;
        $length = strlen($string);
        for ($i = 0; $i < $length; $i++) {
            if (ctype_print($string[$i]) || ctype_space($string[$i])) {
                $printable++;
            }
        }
        
        // Au moins 80% de caractères imprimables
        return ($printable / $length) >= 0.8;
    }

    /**
     * Masque un mot de passe en remplaçant les caractères internes par des étoiles.
     *
     * @param string $password Mot de passe à masquer.
     * @return string Mot de passe masqué.
     */
    public static function getStarred($password): string
    {
        $len = strlen($password);
        if ($len < 2) {
            return $password;
        }
        return substr($password, 0, 1) . str_repeat('*', $len - 2) . substr($password, -1);
    }

    /**
     * Protège un email en générant un lien mailto via JavaScript.
     *
     * @param string $email Adresse email à protéger.
     * @param string $word Texte à afficher pour le lien.
     * @return string Code HTML/JavaScript pour le lien email.
     * @throws InvalidArgumentException Si l'email est invalide.
     */
    public static function protect_email($email, $word): string
    {
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            throw new InvalidArgumentException("Adresse email invalide : '$email'.");
        }

        $parts = explode('@', $email);
        $jsCode = sprintf(
            '<script>var a="<a href=\'mailto:";var b="%s";var c="%s";var d="\' class=\'badge badge-dark\'><i class=\'fa fa-envelope\'></i> ";var e="%s";var f="</a>";document.write(a+b+"@"+c+d+e+f);</script><noscript>Activer JavaScript pour afficher le mail</noscript>',
            htmlspecialchars($parts[0], ENT_QUOTES, 'UTF-8'),
            htmlspecialchars($parts[1], ENT_QUOTES, 'UTF-8'),
            htmlspecialchars($word, ENT_QUOTES, 'UTF-8')
        );

        return $jsCode;
    }

    /**
     * Récupère l'adresse IP réelle de l'utilisateur.
     *
     * @return string Adresse IP détectée.
     */
    public static function getRealIpAddr(): string
    {
        $ip = $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
        if (!empty($_SERVER['HTTP_X_FORWARDED_FOR']) && filter_var($_SERVER['HTTP_X_FORWARDED_FOR'], FILTER_VALIDATE_IP)) {
            $ip = $_SERVER['HTTP_X_FORWARDED_FOR'];
        }
        return $ip;
    }

    /**
     * Cache statique pour la liste des IPs bannies (évite les lectures répétées)
     */
    private static $banListCache = null;
    private static $banListCacheTime = 0;
    private static $BAN_CACHE_TTL = 60; // Cache valide 60 secondes

    /**
     * Valide une adresse IP ou une plage CIDR.
     *
     * @param string $ip Adresse IP ou plage CIDR à valider.
     * @return bool True si valide, false sinon.
     */
    public static function isValidIpOrCidr(string $ip): bool
    {
        if (empty($ip)) {
            return false;
        }

        // Vérifier si c'est une plage CIDR (ex: 192.168.1.0/24)
        if (strpos($ip, '/') !== false) {
            list($ipPart, $cidr) = explode('/', $ip, 2);
            if (!filter_var($ipPart, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4 | FILTER_FLAG_IPV6)) {
                return false;
            }
            $cidr = (int)$cidr;
            $isIPv4 = filter_var($ipPart, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4);
            return ($isIPv4 && $cidr >= 0 && $cidr <= 32) || (!$isIPv4 && $cidr >= 0 && $cidr <= 128);
        }

        // Vérifier si c'est une IP simple
        return filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4 | FILTER_FLAG_IPV6) !== false;
    }

    /**
     * Vérifie si une IP correspond à une plage CIDR.
     *
     * @param string $ip Adresse IP à vérifier.
     * @param string $cidr Plage CIDR (ex: 192.168.1.0/24).
     * @return bool True si l'IP est dans la plage, false sinon.
     */
    private static function ipInCidr(string $ip, string $cidr): bool
    {
        if (strpos($cidr, '/') === false) {
            return $ip === $cidr;
        }

        list($subnet, $mask) = explode('/', $cidr, 2);
        $mask = (int)$mask;

        // IPv4
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
            $ipLong = ip2long($ip);
            $subnetLong = ip2long($subnet);
            if ($ipLong === false || $subnetLong === false) {
                return false;
            }
            $maskLong = -1 << (32 - $mask);
            return ($ipLong & $maskLong) === ($subnetLong & $maskLong);
        }

        // IPv6
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) {
            $ipBin = inet_pton($ip);
            $subnetBin = inet_pton($subnet);
            if ($ipBin === false || $subnetBin === false) {
                return false;
            }
            $bytes = (int)($mask / 8);
            $bits = $mask % 8;
            for ($i = 0; $i < $bytes; $i++) {
                if ($ipBin[$i] !== $subnetBin[$i]) {
                    return false;
                }
            }
            if ($bits > 0) {
                $maskByte = 0xFF << (8 - $bits);
                return (ord($ipBin[$bytes]) & $maskByte) === (ord($subnetBin[$bytes]) & $maskByte);
            }
            return true;
        }

        return false;
    }

    /**
     * Charge la liste des IPs bannies avec cache.
     *
     * @return array Liste des IPs/plages bannies.
     */
    private static function loadBanList(): array
    {
        $now = time();
        
        // Utiliser le cache si disponible et valide
        if (self::$banListCache !== null && ($now - self::$banListCacheTime) < self::$BAN_CACHE_TTL) {
            return self::$banListCache;
        }

        if (!file_exists(self::$BAN_FILE)) {
            self::$banListCache = [];
            self::$banListCacheTime = $now;
            return [];
        }

        $content = @file_get_contents(self::$BAN_FILE);
        if ($content === false) {
            self::$banListCache = [];
            self::$banListCacheTime = $now;
            return [];
        }

        $lines = explode("\n", $content);
        $banList = [];
        
        foreach ($lines as $line) {
            $line = trim($line);
            if (!empty($line) && self::isValidIpOrCidr($line)) {
                $banList[] = $line;
            }
        }

        self::$banListCache = $banList;
        self::$banListCacheTime = $now;
        
        return $banList;
    }

    /**
     * Invalide le cache de la liste des bannissements.
     */
    public static function invalidateBanCache(): void
    {
        self::$banListCache = null;
        self::$banListCacheTime = 0;
    }

    /**
     * Vérifie si une adresse IP est bannie (amélioré avec support CIDR et cache).
     *
     * @param string $ip Adresse IP à vérifier.
     * @return bool True si l'IP est bannie, false sinon.
     */
    public static function isBan(string $ip): bool
    {
        if (empty($ip) || !self::isValidIpOrCidr($ip)) {
            return false;
        }

        $banList = self::loadBanList();
        
        foreach ($banList as $bannedEntry) {
            if (self::ipInCidr($ip, $bannedEntry)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Vérifie si l'IP de l'utilisateur est bannie et redirige si nécessaire (optimisé).
     *
     * @param array $config Configuration du site (par défaut : global $config).
     * @param array $lang Traductions (par défaut : global $lang).
     * @throws RuntimeException Si le fichier de bannissement ou le thème ne peut pas être chargé.
     */
    public static function checkIP(array $config = [], array $lang = []): void
    {
        global $config, $lang;
        $config = $config ?: $config;
        $lang = $lang ?: $lang;

        $ip = self::getRealIpAddr();
        
        if (self::isBan($ip)) {
            $themeFile = THEME_DIR . ($config['theme'] ?? 'default') . DIRECTORY_SEPARATOR . 'banned.tpl.php';
            if (!file_exists($themeFile)) {
                throw new RuntimeException("Fichier de thème '$themeFile' introuvable pour l'affichage de la page de bannissement.");
            }
            require $themeFile;
            exit;
        }
    }
}

/**
 * Exception personnalisée pour les erreurs de sécurité.
 */
class SecurityException extends Exception {}
?>