"""
This program is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License,
or (at your option) any later version.


This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
See the GNU General Public License for more details.


You should have received a copy of the GNU General Public License
 along with this program.  If not, see <https://www.gnu.org/licenses/>.

Copyright © 2019 Cloud Linux Software Inc.

This software is also available under ImunifyAV commercial license,
see <https://www.imunify360.com/legal/eula>
"""
import asyncio
import contextlib
from logging import getLogger

from imav.malwarelib.subsys.ainotify import Inotify, Watcher

from defence360agent.api import inactivity
from defence360agent.contracts.config import MyImunifyConfig
from defence360agent.contracts.plugins import MessageSource
from defence360agent.feature_management.control import (
    is_native_feature_management_enabled,
)
from defence360agent.feature_management.utils import (
    sync_users as sync_feature_management_users,
)
from defence360agent.myimunify.model import sync_users as sync_myimunify_users
from defence360agent.subsys.panels import hosting_panel
from defence360agent.utils import finally_happened, recurring_check

logger = getLogger(__name__)


class PasswdWatcher(MessageSource):
    WATCH_PATH = b"/etc"
    WATCH_FILE = b"passwd"
    WATCH_MASK = Inotify.MOVED_TO

    async def create_source(self, loop, sink, watcher=None):
        self._loop = loop
        self._sink = sink

        self._hosting_panel = hosting_panel.HostingPanel()

        if watcher is None:
            self._watcher = Watcher(self._loop)
        else:
            self._watcher = watcher
        self._watcher.watch(self.WATCH_PATH, self.WATCH_MASK)

        if not await is_native_feature_management_enabled():
            await self.update_users_permissions()
        if MyImunifyConfig.ENABLED:
            await self.update_myimunify_users()

        self._watch_task = self._loop.create_task(self._process_events())

    async def shutdown(self):
        self._watch_task.cancel()
        with contextlib.suppress(asyncio.CancelledError):
            await self._watch_task

        self._watcher and self._watcher.close()

    async def update_users_permissions(self) -> bool:
        return await sync_feature_management_users(
            await self._hosting_panel.get_users()
        )

    async def update_myimunify_users(self) -> bool:
        return await sync_myimunify_users(
            self._sink, await self._hosting_panel.get_users()
        )

    @recurring_check(0)
    async def _process_events(self):
        event = await self._watcher.get_event()
        if event.name != self.WATCH_FILE:
            return
        if not await is_native_feature_management_enabled():
            # updating the users list may take a while on different panels
            with inactivity.track.task("permissions_update"):
                await finally_happened(
                    self.update_users_permissions, max_tries=5, delay=1
                )
                logger.info("Feature management permissions updated")
        if MyImunifyConfig.ENABLED:
            await self.update_myimunify_users()
