import time
from defence360agent.contracts import config
from defence360agent.contracts.messages import MessageType
from defence360agent.contracts.plugins import (
    MessageSink,
    MessageSource,
    expect,
)
from defence360agent.utils import recurring_check, Scope

POLLING_INTERVAL = config.int_from_envvar("READ_CONFIG_POLLING_INTERVAL", 30)


class ConfigWatcher(MessageSink, MessageSource):
    """Send ConfigUpdate message on [root's] config update.

    The config update is detected by polling config file's
    modification time.

    """

    SCOPE = Scope.AV

    def __init__(self):
        self._config = config.ConfigFile()
        self._last_notify_time = 0
        self._sink = None
        self._task = None

    async def create_sink(self, loop):
        "plugins.MessageSink method"

    @expect(MessageType.ConfigUpdate)
    async def on_config_update_message(self, message):
        # update the time, to avoid sending duplicate ConfigUpdate
        # messages after the "config update" command
        self._last_notify_time = message["timestamp"]

    async def create_source(self, loop, sink):
        self._sink = sink
        self._task = loop.create_task(self._check_config())

    async def shutdown(self):
        if self._task is not None:
            t, self._task = self._task, None
            t.cancel()
            await t
        self._sink = None

    @recurring_check(POLLING_INTERVAL)
    async def _check_config(self):
        if config.any_layer_modified_since(self._last_notify_time):
            # notify about the update
            message = MessageType.ConfigUpdate(
                conf=self._config, timestamp=time.time()
            )
            await self._sink.process_message(message)
            # update the time here, in case ConfigUpdate might stuck
            # in the queue for longer than the polling interval
            self._last_notify_time = message["timestamp"]
