import asyncio
import os.path
from typing import Optional

from defence360agent import files
from defence360agent.contracts.config import ANTIVIRUS_MODE
from defence360agent.model.simplification import Eula, run_in_executor


_MESSAGE_TEMPLATE = "message{}.txt"
_SUFFIX = "-av" if ANTIVIRUS_MODE else ""
_TEXT_TEMPLATE = "eula{}.txt"
_UPDATED_TEMPLATE = "updated{}.txt"


def _readfile(path: str, errors: Optional[str] = None) -> str:
    with open(path, errors=errors) as f:
        return f.read().strip()


def _get_path(template: str) -> str:
    return os.path.join(
        files.Index.files_path(files.EULA), template.format(_SUFFIX)
    )


async def is_accepted() -> bool:
    """Return True if latest EULA was accepted, False otherwise."""
    return await run_in_executor(asyncio.get_event_loop(), Eula.is_accepted)


async def accept() -> None:
    """Accepts EULA."""
    await run_in_executor(asyncio.get_event_loop(), Eula.accept)


async def update() -> None:
    """Updates latest EULA date from files."""
    await run_in_executor(
        asyncio.get_event_loop(), lambda: Eula.get_or_create(updated=updated())
    )


def text() -> str:
    """Return main text of the EULA."""
    return _readfile(_get_path(_TEXT_TEMPLATE), errors="ignore")


def message() -> str:
    """Return a message inviting to accept EULA."""
    return _readfile(_get_path(_MESSAGE_TEMPLATE))


def updated() -> str:
    """Return last EULA's update time."""
    return _readfile(_get_path(_UPDATED_TEMPLATE))
