import contextlib
import logging
import os
import os.path

from defence360agent.utils import importer


chattr = importer.get(
    module="imav.malwarelib.utils", name="chattr", default=None
)

logger = logging.getLogger(__name__)
ALT_PHP = "imunify360-alt-php.repo"
EA_PHP = "imunify360-ea-php-hardened.repo"
REPOS_DIR = "/etc/yum.repos.d/"


def irrelevant_repos(release):
    if "cloudlinux" in release:
        # CloudLinux doesn't need either
        return {ALT_PHP, EA_PHP}
    elif os.path.exists("/usr/local/cpanel/cpanel"):
        # cPanel does not need alt-php
        return set([ALT_PHP])
    else:
        # ea-php is only for cPanel
        return set([EA_PHP])


def fix_permissions():
    for repo_name in [ALT_PHP, EA_PHP]:
        path = REPOS_DIR + repo_name
        if not os.path.exists(path):
            continue
        with open(path) as f:
            chattr.subtract_flags(f.fileno(), chattr.FS_IMMUTABLE_FL)
            os.chmod(f.fileno(), 0o644)


def do_migrate():
    if not os.path.exists("/etc/redhat-release"):
        # we do not have to do anything on Ubuntu systems
        return
    with open("/etc/redhat-release") as f:
        release = f.read().lower()
    fix_permissions()
    for repo_name in irrelevant_repos(release):
        with contextlib.suppress(FileNotFoundError):
            os.unlink(REPOS_DIR + repo_name)


def migrate(migrator, database, fake=False, **kwargs):
    if fake:
        return
    try:
        do_migrate()
    except Exception:
        logger.exception("Failed to clean up HardenedPHP repositories")


def rollback(migrator, database, fake=False, **kwargs):
    pass
