"""Peewee migrations: ::

    > Model = migrator.orm['name']

    > migrator.sql(sql)
    > migrator.python(func, *args, **kwargs)
    > migrator.create_model(Model)
    > migrator.remove_model(Model, cascade=True)
    > migrator.add_fields(Model, **fields)
    > migrator.change_fields(Model, **fields)
    > migrator.remove_fields(Model, *field_names, cascade=True)
    > migrator.rename_field(Model, old_field_name, new_field_name)
    > migrator.rename_table(Model, new_table_name)
    > migrator.add_index(Model, *col_names, unique=False)
    > migrator.drop_index(Model, *col_names)
    > migrator.add_not_null(Model, *field_names)
    > migrator.drop_not_null(Model, *field_names)
    > migrator.add_default(Model, field_name, default)

"""

import peewee as pw


class MalwareScan(pw.Model):
    class Meta:
        db_table = "malware_scans"

    scanid = pw.CharField(primary_key=True)
    started = pw.IntegerField(null=False)
    completed = pw.IntegerField(null=False)
    type = pw.CharField(
        null=False, constraints=[pw.Check("type in ('on-demand', 'realtime')")]
    )
    path = pw.CharField(null=False)
    total_files = pw.IntegerField(null=False, default=0)


class MalwareHit(pw.Model):
    class Meta:
        db_table = "malware_hits"

    id = pw.IntegerField(primary_key=True)
    scanid = pw.ForeignKeyField(MalwareScan, null=False)

    user = pw.CharField(null=False)
    orig_file = pw.CharField(null=False)
    type = pw.CharField(null=False)
    restored = pw.BooleanField(null=False, default=False)


class MalwareIgnorePath(pw.Model):
    class Meta:
        db_table = "malware_ignore_path"

    path = pw.CharField(primary_key=True)


def migrate(migrator, database, fake=False, **kwargs):
    """Write your migrations here."""
    migrator.create_model(MalwareScan)
    migrator.create_model(MalwareHit)
    migrator.create_model(MalwareIgnorePath)


def rollback(migrator, database, fake=False, **kwargs):
    """Write your rollback migrations here."""
    MalwareScan = migrator.orm["malware_scans"]
    MalwareHit = migrator.orm["malware_hits"]
    MalwareIgnorePath = migrator.orm["malware_ignore_path"]
    MalwareScannedStat = migrator.orm["malware_stanned_stat"]

    migrator.drop_model(MalwareHit)
    migrator.drop_model(MalwareScan)
    migrator.drop_model(MalwareIgnorePath)
    migrator.drop_model(MalwareScannedStat)
