"""Peewee migrations: ::

    > Model = migrator.orm['name']

    > migrator.sql(sql)
    > migrator.python(func, *args, **kwargs)
    > migrator.create_model(Model)
    > migrator.remove_model(Model, cascade=True)
    > migrator.add_fields(Model, **fields)
    > migrator.change_fields(Model, **fields)
    > migrator.remove_fields(Model, *field_names, cascade=True)
    > migrator.rename_field(Model, old_field_name, new_field_name)
    > migrator.rename_table(Model, new_table_name)
    > migrator.add_index(Model, *col_names, unique=False)
    > migrator.drop_index(Model, *col_names)
    > migrator.add_not_null(Model, *field_names)
    > migrator.drop_not_null(Model, *field_names)
    > migrator.add_default(Model, field_name, default)

"""

from time import time

import peewee as pw


class Country(pw.Model):
    id = pw.CharField(primary_key=True, null=False)
    code = pw.CharField(max_length=2, unique=True, null=False)
    name = pw.CharField(null=False)

    class Meta:
        db_table = "country"


class CountrySubnets(pw.Model):
    country = pw.ForeignKeyField(Country, null=False)

    # 255.255.255.255/32 - max 18 symbols
    ip_net = pw.CharField(max_length=18, null=False)

    class Meta:
        db_table = "country_subnets"


class CountryList(pw.Model):
    # available list names
    WHITE = "WHITE"
    BLACK = "BLACK"

    IP_LISTS = (WHITE, BLACK)

    country = pw.ForeignKeyField(Country, primary_key=True, null=False)
    listname = pw.CharField(
        null=False, constraints=[pw.Check("listname in ('WHITE','BLACK')")]
    )

    ctime = pw.IntegerField(null=True, default=lambda: int(time()))  # are OK

    comment = pw.CharField(null=True)

    class Meta:
        db_table = "country_list"


def migrate(migrator, database, fake=False, **kwargs):
    """Write your migrations here."""

    IPList = migrator.orm["iplist"]

    Incident = migrator.orm["incident"]

    migrator.create_model(Country)

    migrator.add_fields(IPList, country=pw.ForeignKeyField(Country, null=True))
    migrator.add_fields(
        Incident, country=pw.ForeignKeyField(Country, null=True)
    )

    migrator.create_model(CountrySubnets)
    migrator.create_model(CountryList)


def rollback(migrator, database, fake=False, **kwargs):
    """Write your rollback migrations here."""
    Country = migrator.orm["country"]
    CountrySubnets = migrator.orm["country_subnets"]
    CountryList = migrator.orm["country_list"]
    IPList = migrator.orm["iplist"]
    Incident = migrator.orm["incident"]

    migrator.remove_fields(IPList, "country")
    migrator.remove_fields(Incident, "country")

    migrator.remove_model(CountrySubnets)
    migrator.remove_model(CountryList)
    migrator.remove_model(Country)
