import inspect

from defence360agent.feature_management.control import (
    is_native_feature_management_enabled,
)
from defence360agent.rpc_tools.lookup import wraps
from defence360agent.rpc_tools.validate import ValidationError


def builtin_feature_management_only(cls):
    """
    This decorator is intended to wrap rpc endpoint classes. It will
    throw ValidationError if native feature management is enabled
    when any of decorated class methods is called
    """

    def _wrapper(coro):
        @wraps(coro)
        async def wrapper(*args, **kwargs):
            if await is_native_feature_management_enabled():
                raise ValidationError(
                    "Command is disabled because native "
                    "feature management is enabled. "
                    "Please use your hosting panel interface"
                    " to manage features"
                )
            return await coro(*args, **kwargs)

        return wrapper

    assert inspect.isclass(cls), "This decorator can only be used for classes"

    for m_name, m_obj in getattr(cls, "__dict__", {}).items():
        if not m_name.startswith("_") and inspect.iscoroutinefunction(m_obj):
            wrapped = _wrapper(m_obj)
            setattr(cls, m_name, wrapped)
    return cls
