# -*- coding: utf-8 -*-

"""
common helper functions, related to processing api using custom binary
currently available for Directadmin and Plesk
"""

import os
import json
from typing import List, Tuple, Union, Optional
from clcommon.utils import exec_utility
from clcommon.clpwd import ClPwd

USERDOMAINS = '/usr/share/python-cllib/userdomains'


def get_domains_via_custom_binary() -> Tuple[int, Union[str, List[Tuple[str, str]]]]:
    """
    Calls USERDOMAINS(userdomains.c) bin and returns rc and loaded json output
    This is equals to call `userdomains(pwd.getpwuid(os.getuid()).pw_name, as_root=True)`

    Returns: return code of USERDOMAINS and json output
        if rc == 0: `out` is out from `userdomains()`
            NB: there is no tuples in JSON so da_out's type is List[List[str, str]] instead of List[Tuple[str, str]]
        if rc != 0: `out` type is str
        error codes:
            1: diradmin/psaadmin user not found
            2: not DA/Plesk panel
            3: seteuid() syscall failed
            4: executed as root
            10: exception during `userdomains()` call
            11: no such user in panel
    """
    rc, out = exec_utility(USERDOMAINS, [])
    try:
        res = json.loads(out)
    except json.JSONDecodeError:
        pass
    else:
        if rc == 0:
            return rc, [
                (row[0], row[1])
                for row in res
                if len(row) == 2 and all(isinstance(p, str) for p in row)
            ]

    return rc, out


def _docroot_under_user_via_custom_bin(domain: str) -> Optional[
    Tuple[str, str]]:
    """
    New method for getting doc_root for domain under user
    Method parses /usr/local/directadmin/... user's web server config
    See: _get_domains_list_as_root()
    :return: (doc_root, username) tuple
    """

    clpwd = ClPwd()
    user_pw = clpwd.get_pw_by_uid(os.getuid())[0]

    # domains type (if rc == 0): List[Tuple[str, str]]   # domain, docroot
    rc, domains = get_domains_via_custom_binary()
    if rc != 0:
        return None

    for dom, _docroot in domains:
        if dom == domain:
            return _docroot, user_pw.pw_name
    return None
