# coding=utf-8
#
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2019 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT

import os
import errno

PID_FILE = '/var/run/lvestats.pid'
RUNNING = 0
STOPPED = 1
DEAD = 2

status_description = {
    RUNNING: "Server is running",
    STOPPED: "Server is not running, Use 'service lvestats start' command to start it",
    DEAD: "lvestats server is dead but pid file exists. Use 'service lvestats restart' command to restart it"
}


def check_server_status():
    """
    Read pid from file. If file is not exists, then server is stopped
    Send kill signal 0 to check if process exists.
    Return status.

    From http://linux.die.net/man/2/kill :
        "If sig is 0, then no signal is sent, but error checking is still performed;
        this can be used to check for the existence of a process ID or process group ID."

    errno.ESRCH == "No such process"
    :return:
    """
    try:
        with open(PID_FILE, encoding='utf-8') as pid_file:
            pid = pid_file.readline().strip()
    except (IOError, OSError):
        return STOPPED
    else:
        try:
            os.kill(int(pid), 0)
        except OSError as e:
            if e.errno == errno.ESRCH:
                return DEAD
        return RUNNING
