# coding=utf-8
#
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2021 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENCE.TXT
#
import os

from clcommon.const import Feature
from clcommon.utils import ExternalProgramFailed, get_passenger_package_name, is_litespeed_running
from clconfig.ui_config_lib import get_ui_config
from clwizard.constants import MODULES_LOGS_DIR
from clwizard.exceptions import InstallationFailedException
from clwizard.utils import installed_interpreters_list

from .base import WizardInstaller


class RubyInstaller(WizardInstaller):
    LOG_FILE = os.path.join(MODULES_LOGS_DIR, 'ruby.log')
    _REQUIRED_CL_COMPONENT_SUPPORT = Feature.RUBY_SELECTOR

    def _install_ruby_versions(self, versions):
        groups = []
        for v in versions:
            package_name = 'alt-ruby' + v.replace('.', '')
            groups.append(package_name)
        self.app_logger.info("Try to install Ruby package with versions: %s", ','.join(versions))
        try:
            self._install_groups(*groups)
        except ExternalProgramFailed as e:
            raise InstallationFailedException() from e

    def _configure_for_litespeed_server(self):
        """
        In order to make ruby apps work on Litespeed server, we need to run an
        extra script

        See docs for details:
        https://www.litespeedtech.com/support/wiki/doku.php/litespeed_wiki:cloudlinux:enable_passenger
        :return:
        """
        litespeed_script = '/usr/local/lsws/admin/misc/enable_ruby_python_selector.sh'
        # we are going to simplify process of making python/ruby apps work under Litespeed
        # so in future we will not need to run extra script
        if not os.path.isfile(litespeed_script):
            return
        try:
            self._run_command(['sh', litespeed_script])
        except ExternalProgramFailed as e:
            raise InstallationFailedException() from e

    def _is_already_configured(self):
        """
        Check that the Ruby Selector is ready to work

        All configurations were done and Passenger is installed
        We always have at least one ruby package installed, because it's a
        dependency of the Passenger package, so we can skip that check
        If the config file does not exist, we ignore the uiSettings
        :return: already_configured status
        :rtype: bool
        """
        passenger_installed = self._is_package_installed(get_passenger_package_name())
        configs_ok = not get_ui_config() or not get_ui_config().get('uiSettings', {}).get('hideRubyApp')
        return passenger_installed and configs_ok

    def initial_status(self):
        installed_ruby_versions = []
        interpreters_list = installed_interpreters_list('ruby')
        for interpreter in interpreters_list:
            if interpreter.installed:
                installed_ruby_versions.append(interpreter.version)
        return {
            'already_configured': self._is_already_configured(),
            'options': {
                'installed_versions': installed_ruby_versions,
                'available_versions': self._get_available_versions('ruby'),
            },
        }

    def run_installation(self, options):
        install_versions = options.get('versions', [])
        self._install_passenger()
        self._install_ruby_versions(install_versions)
        # if Litespeed server is running - do some extra configuration
        if is_litespeed_running():
            self._configure_for_litespeed_server()
        # do not hide ruby app in web interface
        try:
            self._run_command(
                [
                    'cloudlinux-config',
                    'set',
                    '--json',
                    '--data={"options": {"uiSettings": {"hideRubyApp": false}}}',
                ]
            )
        except ExternalProgramFailed as e:
            raise InstallationFailedException() from e

    @classmethod
    def supported_options(cls):
        return {'versions'}
